# Sistema de Preservación de Filtros en Perfiles Vendidos

## 🎯 Descripción

Esta nueva funcionalidad permite que los filtros de búsqueda se mantengan automáticamente después de realizar operaciones de edición, renovación, cancelación o eliminación de perfiles vendidos. Esto mejora significativamente la experiencia del usuario al evitar tener que reconfigurar los filtros cada vez.

## ✨ Características Principales

### 🔄 Preservación Automática de Filtros
- **Operaciones que preservan filtros:**
  - Editar perfil vendido
  - Renovar suscripción
  - Cancelar suscripción
  - Eliminar perfil permanentemente
  - Agregar múltiples perfiles

### 📋 Filtros Preservados
Los siguientes parámetros se mantienen automáticamente:
- **Búsqueda general** (`buscar`) - Búsqueda en nombre, plataforma, email, teléfono
- **Plataforma** (`filtro_plataforma`) - Filtro por plataforma específica
- **Estado de suscripción** (`filtro_estado`) - Activa, Vencida, Cancelada
- **Estado de cuenta maestra** (`filtro_estado_cm`) - Activa, Inactiva, Con Problemas, Vencida
- **Tipo de cuenta** (`filtro_tipo_cuenta`) - Perfil o Completa
- **Vendedor** (`filtro_vendedor`) - Filtro por usuario vendedor
- **Fechas** (`filtro_fecha_inicio`, `filtro_fecha_fin`) - Rango de fechas de servicio
- **Precios** (`filtro_precio_min`, `filtro_precio_max`) - Rango de precios de venta
- **Ordenamiento** (`ordenar_por`, `orden`) - Campo y dirección de ordenamiento
- **Carga diferida** (`cargar_mas`) - Cantidad de registros a cargar

### 🎨 Mejoras Visuales
- **Indicador de filtros activos** con badge verde
- **Contador de resultados** en el header de filtros
- **Mensaje informativo** cuando se preservan filtros
- **Iconos dinámicos** que cambian de color según el estado

## 🛠️ Implementación Técnica

### Controlador (`actions/perfiles_vendidos_controlador.php`)

#### Función Helper
```php
function preservarFiltrosEnRedireccion(): string {
    $filtros_preservar = [];
    $parametros_filtro = [
        'buscar', 'filtro_estado', 'filtro_estado_cm', 'filtro_tipo_cuenta', 
        'filtro_vendedor', 'filtro_fecha_inicio', 'filtro_fecha_fin', 'filtro_precio_min', 
        'filtro_precio_max', 'ordenar_por', 'orden', 'cargar_mas'
    ];
    
    foreach ($parametros_filtro as $param) {
        if (isset($_POST[$param]) && $_POST[$param] !== '') {
            $filtros_preservar[$param] = $_POST[$param];
        } elseif (isset($_GET[$param]) && $_GET[$param] !== '') {
            $filtros_preservar[$param] = $_GET[$param];
        }
    }
    
    if (!empty($filtros_preservar)) {
        return "?" . http_build_query($filtros_preservar);
    }
    
    return "";
}
```

#### Uso en Redirecciones
```php
} elseif (in_array($accion, ['editar_perfil_vendido','renovar','cancelar_suscripcion','eliminar_permanente']) && $tipo_mensaje === 'success') {
    $redirect_url = "/digitalpremium/pages/perfiles_vendidos.php" . preservarFiltrosEnRedireccion();
}
```

### Vista (`pages/perfiles_vendidos.php`)

#### Mensaje de Filtros Preservados
```php
<?php if (isset($_GET['mensaje'])): ?>
    <div class="alert alert-<?php echo htmlspecialchars($_GET['tipo'] ?? 'info'); ?> alert-dismissible fade show rounded-3 shadow-sm" role="alert">
        <i class="fas fa-info-circle me-2"></i> <?php echo htmlspecialchars(urldecode($_GET['mensaje'])); ?>
        
        <?php 
        // Mostrar información sobre filtros preservados
        $filtros_activos = [];
        $parametros_filtro = ['buscar', 'filtro_estado', 'filtro_estado_cm', 'filtro_tipo_cuenta', 'filtro_vendedor', 'filtro_fecha_inicio', 'filtro_fecha_fin', 'filtro_precio_min', 'filtro_precio_max', 'ordenar_por', 'orden'];
        foreach ($parametros_filtro as $param) {
            if (isset($_GET[$param]) && $_GET[$param] !== '') {
                $filtros_activos[] = $param;
            }
        }
        
        if (!empty($filtros_activos)): ?>
            <div class="mt-2">
                <small class="text-muted">
                    <i class="fas fa-filter me-1"></i>Los filtros de búsqueda se han preservado automáticamente para tu comodidad.
                </small>
            </div>
        <?php endif; ?>
        
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
<?php endif; ?>
```

#### Indicador Visual de Filtros Activos
```php
<div class="card-header bg-light py-3">
    <a class="text-decoration-none text-body fw-semibold d-block" data-bs-toggle="collapse" href="#filtrosPerfilesCollapse" role="button" aria-expanded="<?php echo $filtros_aplicados_pv ? 'true' : 'false'; ?>" aria-controls="filtrosPerfilesCollapse">
        <i class="fas fa-filter me-2 <?php echo $filtros_aplicados_pv ? 'text-success' : 'text-secondary'; ?>"></i> 
        Filtros de Búsqueda
        <?php if ($filtros_aplicados_pv): ?>
            <span class="badge bg-success text-white ms-2"><?php echo $total_perfiles_count; ?> resultados</span>
        <?php endif; ?>
        <i class="fas fa-chevron-<?php echo $filtros_aplicados_pv ? 'up' : 'down'; ?> float-end ms-2" style="font-size: 0.8rem;"></i>
    </a>
</div>
```

#### Formulario de Filtros Completo
```php
<form action="" method="GET" class="row g-3 align-items-end">
    <!-- Primera fila -->
    <div class="col-lg-4 col-md-6">
        <label for="buscar">Buscar (Nombre, Plat., Email, Teléfono):</label>
        <input type="text" name="buscar" value="<?php echo htmlspecialchars($_GET['buscar'] ?? ''); ?>">
    </div>
    <div class="col-lg-2 col-md-6">
        <label for="filtro_plataforma">Plataforma:</label>
        <select name="filtro_plataforma">
            <option value="">Todas</option>
            <!-- Opciones dinámicas -->
        </select>
    </div>
    <div class="col-lg-2 col-md-6">
        <label for="filtro_estado">Estado Suscripción:</label>
        <select name="filtro_estado">
            <option value="">Todos</option>
            <!-- Opciones dinámicas -->
        </select>
    </div>
    <div class="col-lg-2 col-md-6">
        <label for="filtro_estado_cm">Estado Cta. Maestra:</label>
        <select name="filtro_estado_cm">
            <option value="">Todos</option>
            <!-- Opciones dinámicas -->
        </select>
    </div>
    
    <!-- Segunda fila -->
    <div class="col-lg-2 col-md-6">
        <label for="filtro_tipo_cuenta">Tipo de Cuenta:</label>
        <select name="filtro_tipo_cuenta">
            <option value="">Todos</option>
            <option value="PERFIL">Perfil</option>
            <option value="COMPLETA">Completa</option>
        </select>
    </div>
    <div class="col-lg-2 col-md-6">
        <label for="filtro_vendedor">Vendedor:</label>
        <select name="filtro_vendedor">
            <option value="">Todos</option>
            <!-- Opciones dinámicas -->
        </select>
    </div>
    <div class="col-lg-2 col-md-6">
        <label for="filtro_fecha_inicio">Fecha Inicio:</label>
        <input type="date" name="filtro_fecha_inicio">
    </div>
    <div class="col-lg-2 col-md-6">
        <label for="filtro_fecha_fin">Fecha Fin:</label>
        <input type="date" name="filtro_fecha_fin">
    </div>
    
    <!-- Tercera fila -->
    <div class="col-lg-2 col-md-6">
        <label for="filtro_precio_min">Precio Mín:</label>
        <input type="number" step="0.01" name="filtro_precio_min">
    </div>
    <div class="col-lg-2 col-md-6">
        <label for="filtro_precio_max">Precio Máx:</label>
        <input type="number" step="0.01" name="filtro_precio_max">
    </div>
    <div class="col-lg-2 col-md-6">
        <label for="ordenar_por">Ordenar por:</label>
        <select name="ordenar_por">
            <option value="fecha_inicio_servicio">Fecha Inicio</option>
            <option value="fecha_fin_servicio">Fecha Fin</option>
            <option value="precio_venta">Precio</option>
            <option value="nombre_completo">Cliente</option>
            <option value="nombre_plataforma">Plataforma</option>
            <option value="estado_suscripcion">Estado</option>
        </select>
    </div>
    <div class="col-lg-2 col-md-6">
        <label for="orden">Orden:</label>
        <select name="orden">
            <option value="DESC">Descendente</option>
            <option value="ASC">Ascendente</option>
        </select>
    </div>
    
    <!-- Botones -->
    <div class="col-lg-2 col-md-6 d-flex align-items-end">
        <button type="submit" class="btn btn-info btn-sm">Aplicar</button>
        <a href="perfiles_vendidos.php" class="btn btn-outline-secondary btn-sm">Limpiar</a>
    </div>
</form>
```

## 🚀 Flujo de Trabajo

### 1. Usuario Aplica Filtros
- El usuario configura filtros de búsqueda en la página de perfiles vendidos
- Los filtros se aplican y se muestran los resultados filtrados

### 2. Usuario Realiza Operación
- El usuario edita, renueva, cancela o elimina un perfil
- El sistema procesa la operación en el controlador

### 3. Preservación Automática
- El controlador detecta que es una operación que debe preservar filtros
- Se llama a `preservarFiltrosEnRedireccion()` para obtener los parámetros
- Se construye la URL de redirección con los filtros preservados

### 4. Redirección con Filtros
- El usuario es redirigido a `perfiles_vendidos.php` con los filtros aplicados
- Se muestra un mensaje de éxito con información sobre filtros preservados
- Los filtros están activos y funcionando automáticamente

## 💡 Beneficios

### Para el Usuario
- **No pierde el contexto** de búsqueda después de operaciones
- **Ahorra tiempo** al no tener que reconfigurar filtros
- **Mejor experiencia** de usuario en el flujo de trabajo
- **Visibilidad clara** del estado de los filtros activos

### Para el Sistema
- **Consistencia** en la navegación del usuario
- **Eficiencia** en el flujo de trabajo
- **Reducción** de errores de usuario
- **Mejor usabilidad** general

## 🔧 Configuración

### Parámetros Personalizables
Los filtros que se preservan se pueden modificar en la función `preservarFiltrosEnRedireccion()`:

```php
$parametros_filtro = [
    'buscar', 'filtro_estado', 'filtro_estado_cm', 'filtro_tipo_cuenta', 
    'filtro_vendedor', 'filtro_fecha_inicio', 'filtro_fecha_fin', 'filtro_precio_min', 
    'filtro_precio_max', 'ordenar_por', 'orden', 'cargar_mas'
];
```

### Agregar Nuevos Filtros
Para agregar un nuevo filtro que se preserve:

1. Agregar el parámetro a `$parametros_filtro`
2. Asegurar que el parámetro se maneje en la lógica de filtrado
3. Verificar que se incluya en el formulario de filtros

## 🧪 Casos de Prueba

### Escenario 1: Edición con Filtros
1. Aplicar filtros de búsqueda
2. Editar un perfil vendido
3. Verificar que se redirija con filtros preservados
4. Confirmar que los filtros estén activos

### Escenario 2: Renovación con Filtros
1. Aplicar filtros de búsqueda
2. Renovar una suscripción
3. Verificar que se redirija con filtros preservados
4. Confirmar que los filtros estén activos

### Escenario 3: Sin Filtros
1. No aplicar filtros
2. Realizar una operación
3. Verificar que se redirija sin parámetros adicionales

## 📝 Notas de Implementación

### Compatibilidad
- **Backward compatible**: No afecta funcionalidad existente
- **Fallback seguro**: Si no hay filtros, funciona como antes
- **Manejo de errores**: Preserva filtros solo en operaciones exitosas

### Rendimiento
- **Impacto mínimo**: La función helper es eficiente
- **Sin consultas adicionales**: Solo procesa parámetros existentes
- **Optimizado**: Usa `http_build_query()` para construir URLs

### Seguridad
- **Validación**: Los parámetros se procesan de forma segura
- **Sanitización**: Se usa `htmlspecialchars()` en la vista
- **Sin inyección**: Los parámetros se pasan de forma controlada

## 🔮 Futuras Mejoras

### Posibles Extensiones
- **Filtros personalizados**: Permitir guardar combinaciones de filtros
- **Historial de filtros**: Recordar filtros usados recientemente
- **Filtros por defecto**: Configurar filtros automáticos por usuario
- **Exportación de filtros**: Compartir configuraciones de filtros

### Integración con Otros Módulos
- **Cuentas maestras**: Aplicar preservación de filtros
- **Clientes**: Extender funcionalidad a gestión de clientes
- **Dashboard**: Preservar filtros en operaciones del dashboard

## 📚 Referencias

- **Archivo principal**: `pages/perfiles_vendidos.php`
- **Controlador**: `actions/perfiles_vendidos_controlador.php`
- **Función helper**: `preservarFiltrosEnRedireccion()`
- **Parámetros de filtro**: Definidos en la función helper

## 🔧 Lógica de Ordenamiento

### Campos de Ordenamiento Permitidos
```php
$campos_ordenamiento_permitidos = [
    'fecha_inicio_servicio', 'fecha_fin_servicio', 'precio_venta', 
    'nombre_completo', 'nombre_plataforma', 'estado_suscripcion'
];
```

### Consulta SQL con Ordenamiento Dinámico
```sql
ORDER BY
    CASE
        WHEN pv.estado_suscripcion = 2 THEN 4
        WHEN DATEDIFF(pv.fecha_fin_servicio, CURDATE()) < 0 THEN 1
        WHEN DATEDIFF(pv.fecha_fin_servicio, CURDATE()) <= 7 THEN 2
        ELSE 3
    END ASC,
    CASE 
        WHEN :ordenar_por = 'fecha_inicio_servicio' THEN pv.fecha_inicio_servicio
        WHEN :ordenar_por = 'fecha_fin_servicio' THEN pv.fecha_fin_servicio
        WHEN :ordenar_por = 'precio_venta' THEN pv.precio_venta
        WHEN :ordenar_por = 'nombre_completo' THEN c.nombre_completo
        WHEN :ordenar_por = 'nombre_plataforma' THEN p.nombre_plataforma
        WHEN :ordenar_por = 'estado_suscripcion' THEN pv.estado_suscripcion
        ELSE pv.fecha_inicio_servicio
    END :orden
```

### Ejecución de la Consulta
```php
$stmt_perfiles = $pdo->prepare($sql_perfiles);

// Agregar parámetros de ordenamiento
$params_ejecucion = array_merge($params_pv, [
    ':ordenar_por' => $orden_por,
    ':orden' => $orden
]);

$stmt_perfiles->execute($params_ejecucion);
```

---

**Desarrollado para DigitalPremium** - Sistema de gestión de suscripciones digitales
