<?php
// FILE: /digitalpremium/actions/clientes_agregar_ajax.php
header('Content-Type: application/json; charset=utf-8');
if (session_status() === PHP_SESSION_NONE) { session_start(); }

// --- Verificación de autenticación ---
if (!isset($_SESSION['id_usuario'])) {
    echo json_encode(['success' => false, 'message' => 'Acceso no autorizado.']);
    exit();
}

require_once '../config/db.php';
require_once '../includes/tenant_helper.php'; // OWNER global

$id_usuario_sesion = (int)($_SESSION['id_usuario'] ?? 0);
$rol               = strtolower(trim($_SESSION['rol'] ?? 'ventas'));
if ($rol === 'super_admin') $rol = 'superadmin';

// permisos (mismo criterio que en controlador)
$allow_create = in_array($rol, ['superadmin','admin','ventas'], true);
if (!$allow_create) {
    echo json_encode(['success' => false, 'message' => 'No tienes permisos para crear clientes.']);
    exit();
}

$OWNER_ID = get_data_owner_id($pdo); // clientes compartidos

// Helper: normalizar teléfono a dígitos
$norm = function (?string $t): string {
    return preg_replace('/\D+/', '', (string)$t ?? '');
};
// SQL para normalizar en BD
$sql_norm_tel = "REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(telefono,'+',''),'-',''),' ',''),'(',''),')',''),'.','')";

try {
    // ----- Entrada -----
    $nombre   = trim($_POST['nombre_completo'] ?? '');
    $telefono = trim($_POST['telefono'] ?? '');
    $email    = trim($_POST['email'] ?? '');
    $notas    = trim($_POST['notas_cliente'] ?? '');

    // ----- Validaciones -----
    if ($nombre === '') {
        echo json_encode(['success' => false, 'message' => 'El nombre del cliente es obligatorio.']);
        exit();
    }
    if ($telefono === '') {
        echo json_encode(['success' => false, 'message' => 'El número de teléfono es obligatorio.']);
        exit();
    }

    $tel_norm = $norm($telefono);
    if ($tel_norm === '' || strlen($tel_norm) < 9 || strlen($tel_norm) > 15) {
        echo json_encode(['success' => false, 'message' => 'El teléfono debe contener solo dígitos (9 a 15).']);
        exit();
    }

    if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['success' => false, 'message' => 'El formato del email no es válido.']);
        exit();
    }

    // ----- Duplicado global por TEL (normalizado, sin id_usuario) -----
    $stmt = $pdo->prepare("SELECT 1 FROM clientes WHERE {$sql_norm_tel} = :t LIMIT 1");
    $stmt->execute([':t' => $tel_norm]);
    if ($stmt->fetchColumn()) {
        echo json_encode(['success' => false, 'message' => 'Ya existe un cliente con ese número de teléfono.']);
        exit();
    }

    // (Opcional) Duplicado por EMAIL global (si lo quieres global)
    if ($email !== '') {
        $stmt = $pdo->prepare("SELECT 1 FROM clientes WHERE email = :email LIMIT 1");
        $stmt->execute([':email' => $email]);
        if ($stmt->fetchColumn()) {
            echo json_encode(['success' => false, 'message' => 'Ya existe un cliente con ese email.']);
            exit();
        }
    }

    // ----- Inserción -----
    $pdo->beginTransaction();

    $stmt = $pdo->prepare("
        INSERT INTO clientes (id_usuario, nombre_completo, telefono, email, notas_cliente, estado_cliente)
        VALUES (:id_owner, :nombre_completo, :telefono, :email, :notas_cliente, 1)
    ");
    $stmt->execute([
        ':id_owner'        => $OWNER_ID,
        ':nombre_completo' => $nombre,
        ':telefono'        => $telefono,
        ':email'           => ($email !== '' ? $email : null),
        ':notas_cliente'   => ($notas !== '' ? $notas : null)
    ]);

    $new_id = $pdo->lastInsertId();
    $pdo->commit();

    echo json_encode([
        'success' => true,
        'message' => 'Cliente agregado con éxito.',
        'cliente' => [
            'id_cliente'      => $new_id,
            'nombre_completo' => $nombre,
            'telefono'        => $telefono,
            'email'           => ($email !== '' ? $email : null)
        ]
    ]);
    exit();

} catch (PDOException $e) {
    if ($pdo && $pdo->inTransaction()) { $pdo->rollBack(); }

    // Si tienes índices únicos, capturamos el duplicado por clave:
    if (isset($e->errorInfo[1]) && (int)$e->errorInfo[1] === 1062) {
        echo json_encode(['success' => false, 'message' => 'Registro duplicado.']); exit();
    }

    error_log('Error BD clientes_agregar_ajax: ' . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Error de base de datos.']);
    exit();

} catch (Exception $e) {
    if ($pdo && $pdo->inTransaction()) { $pdo->rollBack(); }
    error_log('Error inesperado clientes_agregar_ajax: ' . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Error inesperado.']);
    exit();
}
