<?php
// FILE: /digitalpremium/actions/ajax_agregar_cuenta_maestra.php
header('Content-Type: application/json; charset=utf-8');

if (session_status() === PHP_SESSION_NONE) { session_start(); }

// --- Auth mínima ---
if (!isset($_SESSION['id_usuario'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Acceso no autorizado. Tu sesión ha expirado.']);
    exit();
}

require_once '../config/db.php';

$id_usuario_sesion = (int)($_SESSION['id_usuario'] ?? 0);
$rol_usuario       = strtolower(trim($_SESSION['rol'] ?? 'ventas'));
if ($rol_usuario === 'super_admin') $rol_usuario = 'superadmin';

// ---- Permisos (solo para esta acción: crear cuenta maestra) ----
function canCreateCuenta(string $rol): bool {
    switch ($rol) {
        case 'superadmin':
        case 'admin':
        case 'ventas':
            return true;        // pueden crear
        case 'cortes':
        default:
            return false;       // cortes NO crea
    }
}
if (!canCreateCuenta($rol_usuario)) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Permiso denegado para crear cuentas maestras.']);
    exit();
}

try {
    // --- Validación de sesión en BD (opcional, útil) ---
    $stmt_user = $pdo->prepare("SELECT id_usuario FROM usuarios WHERE id_usuario = ?");
    $stmt_user->execute([$id_usuario_sesion]);
    if (!$stmt_user->fetchColumn()) {
        session_destroy();
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Sesión inválida. Vuelve a iniciar sesión.']);
        exit();
    }

    // --- Entradas ---
    $id_plataforma  = filter_input(INPUT_POST, 'id_plataforma', FILTER_VALIDATE_INT);
    $login_raw      = trim($_POST['email_cuenta'] ?? '');     // puede ser correo o usuario
    $password       = trim($_POST['password_cuenta'] ?? '');
    $perfiles_total = filter_input(INPUT_POST, 'perfiles_total', FILTER_VALIDATE_INT);
    $fecha_venc     = trim($_POST['fecha_vencimiento_proveedor'] ?? '');

    $nombre_proveedor = trim($_POST['nombre_proveedor'] ?? '');
    $fecha_compra     = trim($_POST['fecha_compra_proveedor'] ?? '');
    $costo_str        = isset($_POST['costo_compra']) ? str_replace(',', '.', trim((string)$_POST['costo_compra'])) : '';
    $notas_cuenta     = trim($_POST['notas_cuenta'] ?? '');

    // --- Validaciones básicas ---
    if (!$id_plataforma) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Debes seleccionar una plataforma.']);
        exit();
    }
    if ($login_raw === '') {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'El usuario/correo de la cuenta es obligatorio.']);
        exit();
    }
    if ($password === '') {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'La contraseña no puede estar vacía.']);
        exit();
    }
    if ($perfiles_total === false || $perfiles_total === null || $perfiles_total < 1) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => '“Perfiles totales” debe ser un entero mayor o igual a 1.']);
        exit();
    }

    // validar fecha vencimiento (Y-m-d)
    $dt_venc = DateTime::createFromFormat('Y-m-d', $fecha_venc);
    $fecha_venc_valida = $dt_venc && $dt_venc->format('Y-m-d') === $fecha_venc;
    if (!$fecha_venc_valida) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'La fecha de vencimiento no es válida.']);
        exit();
    }

    // validar fecha compra si llega
    if ($fecha_compra !== '') {
        $dt_compra = DateTime::createFromFormat('Y-m-d', $fecha_compra);
        if (!($dt_compra && $dt_compra->format('Y-m-d') === $fecha_compra)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'La fecha de compra no es válida.']);
            exit();
        }
    } else {
        $fecha_compra = null;
    }

    // costo (acepta "10,50" o "10.50")
    $costo_compra = ($costo_str === '' ? null : filter_var($costo_str, FILTER_VALIDATE_FLOAT));

    // Traer límite de perfiles de la plataforma (si existe)
    $stmt_plat = $pdo->prepare("SELECT max_perfiles_cuenta FROM plataformas WHERE id_plataforma = :id AND estado = 1");
    $stmt_plat->execute([':id' => $id_plataforma]);
    $max_perfiles = $stmt_plat->fetchColumn();
    if ($max_perfiles !== false && $max_perfiles !== null) {
        if ((int)$perfiles_total > (int)$max_perfiles) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => "El máximo de perfiles para esta plataforma es {$max_perfiles}."
            ]);
            exit();
        }
    }

    // --- Duplicado GLOBAL por (plataforma, login case-insensitive) ---
    $stmt_dup = $pdo->prepare("
        SELECT 1
        FROM cuentas_maestras
        WHERE id_plataforma = :plat
          AND LOWER(email_cuenta) = :login_lower
        LIMIT 1
    ");
    $stmt_dup->execute([
        ':plat'        => $id_plataforma,
        ':login_lower' => mb_strtolower($login_raw, 'UTF-8'),
    ]);
    if ($stmt_dup->fetchColumn()) {
        http_response_code(409);
        echo json_encode(['success' => false, 'message' => 'Ya existe una cuenta de esa plataforma con ese correo/usuario.']);
        exit();
    }

    // --- Inserción ---
    $pdo->beginTransaction();

    $stmt_ins = $pdo->prepare("
        INSERT INTO cuentas_maestras
            (id_usuario, id_plataforma, email_cuenta, password_cuenta, nombre_proveedor,
             fecha_compra_proveedor, fecha_vencimiento_proveedor, costo_compra,
             perfiles_total, perfiles_disponibles, notas_cuenta, estado_cuenta)
        VALUES
            (:usr, :plat, :login, :pwd, :prov,
             :fcompra, :fvenc, :costo,
             :ptot, :pdisp, :notas, 1)
    ");
    $stmt_ins->execute([
        ':usr'     => $id_usuario_sesion,
        ':plat'    => $id_plataforma,
        ':login'   => $login_raw, // dejamos el casing original
        ':pwd'     => $password,  // en texto plano (como vienes usando)
        ':prov'    => ($nombre_proveedor !== '' ? $nombre_proveedor : null),
        ':fcompra' => $fecha_compra,
        ':fvenc'   => $fecha_venc,
        ':costo'   => $costo_compra,
        ':ptot'    => $perfiles_total,
        ':pdisp'   => $perfiles_total,
        ':notas'   => ($notas_cuenta !== '' ? $notas_cuenta : null),
    ]);

    $new_id = $pdo->lastInsertId();
    $pdo->commit();

    echo json_encode([
        'success' => true,
        'message' => '¡Cuenta maestra agregada con éxito!',
        'cuenta'  => [
            'id_cuenta_maestra'    => $new_id,
            'id_plataforma'        => $id_plataforma,
            'email_cuenta'         => $login_raw,
            'password_cuenta'      => $password,
            'perfiles_total'       => $perfiles_total,
            'perfiles_disponibles' => $perfiles_total
        ]
    ]);
    exit();

} catch (PDOException $e) {
    if ($pdo && $pdo->inTransaction()) { $pdo->rollBack(); }

    // Duplicado por índice único en BD
    if (isset($e->errorInfo[1]) && (int)$e->errorInfo[1] === 1062) {
        http_response_code(409);
        echo json_encode(['success' => false, 'message' => 'El usuario/correo ya está registrado para esta plataforma.']);
        exit();
    }

    error_log('PDO Error ajax_agregar_cuenta_maestra: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Error de base de datos.']);
    exit();

} catch (Exception $e) {
    if ($pdo && $pdo->inTransaction()) { $pdo->rollBack(); }
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    exit();
}
