<?php
// FILE: /actions/gestion_usuarios_controlador.php (VERSIÓN FINAL CORREGIDA Y MEJORADA)

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Bloqueo de seguridad: Solo el superadmin puede realizar estas acciones
if (!isset($_SESSION['id_usuario']) || $_SESSION['rol'] !== 'superadmin') {
    header("Location: /digitalpremium/pages/dashboard.php?mensaje=" . urlencode("Acceso no autorizado.") . "&tipo=danger");
    exit();
}

require_once '../config/db.php';
$redirect_url = '/digitalpremium/pages/gestion_usuarios.php';

$accion = $_POST['accion'] ?? null;
$mensaje = '';
$tipo_mensaje = 'danger';

try {
    $pdo->beginTransaction();

    switch ($accion) {
        case 'crear_usuario':
            // 1. Validar datos de entrada
            if (empty($_POST['nombre_completo']) || empty($_POST['username']) || empty($_POST['password']) || empty($_POST['rol']) || empty($_POST['duracion_membresia'])) {
                throw new Exception("Todos los campos para crear un usuario son requeridos.");
            }
            
            $nombre_completo = trim($_POST['nombre_completo']);
            $username = trim($_POST['username']);
            $password = $_POST['password']; // Contraseña sin hashear para la verificación
            $rol = $_POST['rol'];
            $duracion_dias = filter_var($_POST['duracion_membresia'], FILTER_VALIDATE_INT);
            $telefono = trim($_POST['telefono'] ?? null);
            
            if ($duracion_dias === false || $duracion_dias <= 0) {
                throw new Exception("La duración de la membresía debe ser un número positivo.");
            }

            // 2. Verificación proactiva de usuario duplicado (MEJORA CLAVE)
            $stmt_check = $pdo->prepare("SELECT id_usuario FROM usuarios WHERE username = ?");
            $stmt_check->execute([$username]);
            if ($stmt_check->fetch()) {
                throw new Exception("El nombre de usuario '{$username}' ya existe. Por favor, elige otro.");
            }
            
            // 3. Preparar datos para la inserción
            $password_hashed = password_hash($password, PASSWORD_DEFAULT);
            $fecha_fin_membresia = (new DateTime('now', new DateTimeZone('America/Lima')))->modify("+$duracion_dias days")->format('Y-m-d');
            
            // 4. Insertar el nuevo usuario (CON EL CAMPO TELEFONO)
            $stmt = $pdo->prepare("INSERT INTO usuarios (nombre_completo, username, password, rol, estado, fecha_fin_membresia, telefono) VALUES (?, ?, ?, ?, 1, ?, ?)");
            $stmt->execute([$nombre_completo, $username, $password_hashed, $rol, $fecha_fin_membresia, $telefono]);
            $nuevo_id_usuario = $pdo->lastInsertId();

            // 5. Clonar configuraciones por defecto para el nuevo usuario
            $stmt_plantilla = $pdo->query("SELECT * FROM configuraciones WHERE id_usuario IS NULL");
            $plantilla_configs = $stmt_plantilla->fetchAll(PDO::FETCH_ASSOC);

            if (!empty($plantilla_configs)) {
                $sql_insert = "INSERT INTO configuraciones (id_usuario, clave_config, valor_config, nombre_visible, descripcion_config, tipo_input, opciones_select, grupo_config, orden_grupo) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
                $stmt_insert = $pdo->prepare($sql_insert);
                foreach ($plantilla_configs as $config) {
                    $stmt_insert->execute([$nuevo_id_usuario, $config['clave_config'], $config['valor_config'], $config['nombre_visible'], $config['descripcion_config'], $config['tipo_input'], $config['opciones_select'], $config['grupo_config'], $config['orden_grupo']]);
                }
            }
            
            $mensaje = "Usuario '{$nombre_completo}' creado y configuraciones clonadas exitosamente.";
            $tipo_mensaje = 'success';
            break;

        case 'editar_usuario':
            $id_usuario_editar = filter_var($_POST['id_usuario'], FILTER_VALIDATE_INT);
            if (!$id_usuario_editar) throw new Exception("ID de usuario inválido.");
            
            $nombre_completo = trim($_POST['nombre_completo']);
            $username = trim($_POST['username']);
            $telefono = trim($_POST['telefono'] ?? null);

            // Verificación proactiva de usuario duplicado en edición (MEJORA CLAVE)
            $stmt_check = $pdo->prepare("SELECT id_usuario FROM usuarios WHERE username = ? AND id_usuario != ?");
            $stmt_check->execute([$username, $id_usuario_editar]);
            if ($stmt_check->fetch()) {
                throw new Exception("El nombre de usuario '{$username}' ya está en uso por otro usuario. Elige otro.");
            }

            // Construcción dinámica y segura de la consulta UPDATE (CON CAMPO TELEFONO)
            $sql_parts = ["nombre_completo = :nombre", "username = :username", "rol = :rol", "estado = :estado", "fecha_fin_membresia = :fecha_fin", "telefono = :telefono"];
            $params = [
                ':nombre' => $nombre_completo, 
                ':username' => $username, 
                ':rol' => $_POST['rol'], 
                ':estado' => $_POST['estado'], 
                ':fecha_fin' => empty($_POST['fecha_fin_membresia']) ? null : $_POST['fecha_fin_membresia'],
                ':telefono' => empty($telefono) ? null : $telefono,
                ':id_usuario' => $id_usuario_editar
            ];
            
            if (!empty($_POST['password'])) {
                $sql_parts[] = "password = :password";
                $params[':password'] = password_hash($_POST['password'], PASSWORD_DEFAULT);
            }
            
            $sql = "UPDATE usuarios SET " . implode(", ", $sql_parts) . " WHERE id_usuario = :id_usuario";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);

            $mensaje = "Usuario actualizado correctamente.";
            $tipo_mensaje = 'success';
            break;

        case 'eliminar_usuario':
            $id_usuario_eliminar = filter_var($_POST['id_usuario'], FILTER_VALIDATE_INT);
            if (!$id_usuario_eliminar) throw new Exception("ID de usuario inválido.");

            // La eliminación en cascada en la BD (ON DELETE CASCADE) se encarga del resto.
            $stmt = $pdo->prepare("DELETE FROM usuarios WHERE id_usuario = ?");
            $stmt->execute([$id_usuario_eliminar]);

            $mensaje = "Usuario y todos sus datos asociados han sido eliminados permanentemente.";
            $tipo_mensaje = 'success';
            break;

        default:
            throw new Exception("Acción no reconocida.");
    }
    
    $pdo->commit();

} catch (Exception | PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $mensaje = $e->getMessage();
    $tipo_mensaje = 'danger';
    error_log("Error en gestion_usuarios_controlador.php: " . $e->getMessage());
}

header("Location: {$redirect_url}?mensaje=" . urlencode($mensaje) . "&tipo=" . $tipo_mensaje);
exit();