<?php
// FILE: /digitalpremium/actions/plataformas_controlador.php
declare(strict_types=1);

if (session_status() === PHP_SESSION_NONE) session_start();

/* 1) Autenticación */
if (!isset($_SESSION['id_usuario'])) {
    header("Location: /digitalpremium/login.php?error=" . urlencode("Acceso no autorizado"));
    exit();
}

require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../includes/audit_helper.php';
$pdo = function_exists('db') ? db() : $pdo;

/* 2) Roles y helpers */
$uid = (int)($_SESSION['id_usuario'] ?? 0);
$rol = strtolower((string)($_SESSION['rol'] ?? ''));

$isAdmin  = in_array($rol, ['admin','super_admin','superadmin'], true);
$isVentas = ($rol === 'ventas');
$isCortes = ($rol === 'cortes');

$accion        = $_POST['accion'] ?? $_GET['accion'] ?? null;
$mensaje       = '';
$tipo_mensaje  = 'danger';

// La función audit ahora está en el helper de auditoría

try {
    switch ($accion) {

        /* --------- AGREGAR --------- */
        case 'agregar_plataforma':
            // Permisos: admin o ventas
            if (!($isAdmin || $isVentas)) {
                throw new Exception("Acceso denegado. No tienes permiso para gestionar las plataformas.");
            }

            $nombre = trim((string)($_POST['nombre_plataforma'] ?? ''));
            $max    = filter_var($_POST['max_perfiles_cuenta'] ?? null, FILTER_VALIDATE_INT);
            $desc   = trim((string)($_POST['descripcion'] ?? ''));
            // ventas NO puede desactivar; fuerza activo
            $estado = $isAdmin ? (isset($_POST['estado']) ? 1 : 0) : 1;

            if ($nombre === '')                        throw new Exception("El nombre de la plataforma es obligatorio.");
            if ($max === false || $max < 1)            throw new Exception("Máx. perfiles por cuenta debe ser un entero ≥ 1.");

            // Duplicado por nombre (case-insensitive)
            $chk = $pdo->prepare("SELECT id_plataforma FROM plataformas WHERE LOWER(nombre_plataforma)=LOWER(?) LIMIT 1");
            $chk->execute([$nombre]);
            if ($chk->fetchColumn())                   throw new Exception("Ya existe una plataforma con ese nombre.");

            $ins = $pdo->prepare("INSERT INTO plataformas (nombre_plataforma, max_perfiles_cuenta, descripcion, estado)
                                  VALUES (?, ?, ?, ?)");
            $ins->execute([$nombre, (int)$max, ($desc === '' ? null : $desc), (int)$estado]);
            $idNew = (int)$pdo->lastInsertId();

            audit($pdo, $uid, $rol, 'create', 'plataformas', $idNew, [
                'nombre'=>$nombre,'max_perfiles_cuenta'=>$max,'estado'=>$estado
            ]);

            $mensaje = "Plataforma agregada exitosamente.";
            $tipo_mensaje = 'success';
            break;

        /* --------- EDITAR --------- */
        case 'editar_plataforma':
            // Permisos: admin o ventas (ventas no cambia estado)
            if (!($isAdmin || $isVentas)) {
                throw new Exception("Acceso denegado. No tienes permiso para gestionar las plataformas.");
            }

            $id     = filter_var($_POST['id_plataforma'] ?? null, FILTER_VALIDATE_INT);
            $nombre = trim((string)($_POST['nombre_plataforma'] ?? ''));
            $max    = filter_var($_POST['max_perfiles_cuenta'] ?? null, FILTER_VALIDATE_INT);
            $desc   = trim((string)($_POST['descripcion'] ?? ''));

            if (!$id)                                   throw new Exception("ID de plataforma inválido.");
            if ($nombre === '')                          throw new Exception("El nombre de la plataforma es obligatorio.");
            if ($max === false || $max < 1)             throw new Exception("Máx. perfiles por cuenta debe ser un entero ≥ 1.");

            // Determinar estado según rol
            if ($isAdmin) {
                $estado = isset($_POST['estado']) ? 1 : 0;
            } else {
                // ventas mantiene el estado actual
                $cur = $pdo->prepare("SELECT estado FROM plataformas WHERE id_plataforma = ?");
                $cur->execute([$id]);
                $estado = (int)$cur->fetchColumn();
                if ($estado !== 0 && $estado !== 1) $estado = 1;
            }

            // Duplicado por nombre en otro ID
            $chk = $pdo->prepare("SELECT id_plataforma FROM plataformas WHERE LOWER(nombre_plataforma)=LOWER(?) AND id_plataforma <> ? LIMIT 1");
            $chk->execute([$nombre, $id]);
            if ($chk->fetchColumn())                    throw new Exception("Ya existe otra plataforma con ese nombre.");

            $upd = $pdo->prepare("UPDATE plataformas
                                     SET nombre_plataforma = ?, max_perfiles_cuenta = ?, descripcion = ?, estado = ?
                                   WHERE id_plataforma = ?");
            $upd->execute([$nombre, (int)$max, ($desc === '' ? null : $desc), (int)$estado, $id]);

            audit($pdo, $uid, $rol, 'update', 'plataformas', (int)$id, [
                'nombre'=>$nombre,'max_perfiles_cuenta'=>$max,'estado'=>$estado
            ]);

            $mensaje = "Plataforma actualizada exitosamente.";
            $tipo_mensaje = 'success';
            break;

        /* --------- ELIMINAR --------- */
        case 'eliminar_plataforma':
            // Permisos: SOLO admin/super_admin
            if (!$isAdmin) {
                throw new Exception("Acceso denegado. Solo administradores pueden eliminar plataformas.");
            }

            $id = filter_var($_POST['id_plataforma'] ?? null, FILTER_VALIDATE_INT);
            if (!$id)                                   throw new Exception("ID de plataforma inválido.");

            // Iniciar transacción para eliminación en cascada
            $pdo->beginTransaction();
            
            try {
                // Contar registros que se van a eliminar
                $countCuentas = $pdo->prepare("SELECT COUNT(*) FROM cuentas_maestras WHERE id_plataforma = ?");
                $countCuentas->execute([$id]);
                $totalCuentas = (int)$countCuentas->fetchColumn();
                
                // Contar perfiles vendidos que se van a eliminar (a través de cuentas maestras)
                $countPerfiles = $pdo->prepare("
                    SELECT COUNT(*) FROM perfiles_vendidos pv 
                    INNER JOIN cuentas_maestras cm ON pv.id_cuenta_maestra = cm.id_cuenta_maestra 
                    WHERE cm.id_plataforma = ?
                ");
                $countPerfiles->execute([$id]);
                $totalPerfiles = (int)$countPerfiles->fetchColumn();
                
                // Eliminar perfiles vendidos asociados primero (por dependencias)
                if ($totalPerfiles > 0) {
                    $delPerfiles = $pdo->prepare("
                        DELETE pv FROM perfiles_vendidos pv 
                        INNER JOIN cuentas_maestras cm ON pv.id_cuenta_maestra = cm.id_cuenta_maestra 
                        WHERE cm.id_plataforma = ?
                    ");
                    $delPerfiles->execute([$id]);
                    
                    audit($pdo, $uid, $rol, 'delete_cascade', 'perfiles_vendidos', 0, [
                        'plataforma_id' => $id,
                        'registros_eliminados' => $totalPerfiles,
                        'motivo' => 'Eliminación de plataforma'
                    ]);
                }
                
                // Eliminar cuentas maestras asociadas
                if ($totalCuentas > 0) {
                    $delCuentas = $pdo->prepare("DELETE FROM cuentas_maestras WHERE id_plataforma = ?");
                    $delCuentas->execute([$id]);
                    
                    audit($pdo, $uid, $rol, 'delete_cascade', 'cuentas_maestras', 0, [
                        'plataforma_id' => $id,
                        'registros_eliminados' => $totalCuentas,
                        'motivo' => 'Eliminación de plataforma'
                    ]);
                }
                
                // Finalmente eliminar la plataforma
                $del = $pdo->prepare("DELETE FROM plataformas WHERE id_plataforma = ?");
                $del->execute([$id]);

                audit($pdo, $uid, $rol, 'delete', 'plataformas', (int)$id, [
                    'cuentas_eliminadas' => $totalCuentas,
                    'perfiles_eliminados' => $totalPerfiles
                ]);
                
                // Confirmar transacción
                $pdo->commit();
                
                // Mensaje informativo
                if ($totalCuentas > 0 || $totalPerfiles > 0) {
                    $mensaje = "✅ Plataforma eliminada exitosamente. Se eliminaron {$totalCuentas} cuentas maestras y {$totalPerfiles} perfiles vendidos asociados.";
                } else {
                    $mensaje = "✅ Plataforma eliminada exitosamente.";
                }
                $tipo_mensaje = 'success';
                
            } catch (Exception $e) {
                // Revertir transacción en caso de error
                $pdo->rollBack();
                throw new Exception("Error durante la eliminación en cascada: " . $e->getMessage());
            }
            break;

        /* --------- CAMBIAR ESTADO (si tuvieras acción separada) --------- */
        case 'cambiar_estado_plataforma':
            if (!$isAdmin) {
                throw new Exception("Acceso denegado. Solo administradores pueden cambiar el estado.");
            }
            $id     = filter_var($_POST['id_plataforma'] ?? null, FILTER_VALIDATE_INT);
            $estado = filter_var($_POST['estado'] ?? null, FILTER_VALIDATE_INT);
            if (!$id || !in_array($estado, [0,1], true)) throw new Exception("Datos inválidos para cambiar estado.");

            $upd = $pdo->prepare("UPDATE plataformas SET estado = ? WHERE id_plataforma = ?");
            $upd->execute([(int)$estado, (int)$id]);

            audit($pdo, $uid, $rol, 'update', 'plataformas', (int)$id, ['estado'=>(int)$estado]);

            $mensaje = "Estado de la plataforma actualizado.";
            $tipo_mensaje = 'success';
            break;

        default:
            throw new Exception("Acción no reconocida.");
    }

} catch (PDOException $e) {
    $mensaje = ($e->getCode() === '23000')
        ? "Error de integridad (posible nombre duplicado)."
        : "Error de base de datos: " . $e->getMessage();

} catch (Exception $e) {
    $mensaje = $e->getMessage();
}

/* Redirección a la lista con mensaje */
header("Location: /digitalpremium/pages/plataformas.php?mensaje=" . urlencode($mensaje) . "&tipo=" . urlencode($tipo_mensaje));
exit;
