<?php
// FILE: /digitalpremium/actions/proveedores_controlador.php

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Verificar autenticación
if (!isset($_SESSION['id_usuario'])) {
    header("Location: /digitalpremium/login.php?error=" . urlencode("Acceso no autorizado"));
    exit();
}

require_once '../config/db.php';

// <<< INICIO DE CORRECCIONES >>>

// --- 1. OBTENER DATOS DE SESIÓN ---
$id_usuario_sesion = $_SESSION['id_usuario'];
$rol_usuario_sesion = $_SESSION['rol'];

$accion = $_POST['accion'] ?? $_GET['accion'] ?? null;
$mensaje = '';
$tipo_mensaje = 'danger';

try {
    // Envolver todo en una transacción para consistencia
    $pdo->beginTransaction();

    switch ($accion) {
        case 'agregar':
            if (empty($_POST['nombre_proveedor'])) {
                throw new Exception("El nombre del proveedor es requerido.");
            }
            
            $nombre_proveedor = trim($_POST['nombre_proveedor']);
            $contacto_nombre = trim($_POST['contacto_nombre'] ?? null);
            $contacto_telefono = trim($_POST['contacto_telefono'] ?? null);
            $contacto_email = trim($_POST['contacto_email'] ?? null);
            $notas_proveedor = trim($_POST['notas_proveedor'] ?? null);
            $estado_proveedor = filter_var($_POST['estado_proveedor'], FILTER_VALIDATE_INT);
            
            if ($estado_proveedor === false || !in_array($estado_proveedor, [0, 1])) {
                throw new Exception("Estado de proveedor inválido.");
            }
            if (!empty($contacto_email) && filter_var($contacto_email, FILTER_VALIDATE_EMAIL) === false) {
                throw new Exception("El formato del email de contacto no es válido.");
            }

            // --- 2. AÑADIR id_usuario AL INSERTAR ---
            $sql = "INSERT INTO proveedores (id_usuario, nombre_proveedor, contacto_nombre, contacto_telefono, contacto_email, notas_proveedor, estado_proveedor) 
                    VALUES (:id_usuario, :nombre_proveedor, :contacto_nombre, :contacto_telefono, :contacto_email, :notas_proveedor, :estado_proveedor)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                ':id_usuario' => $id_usuario_sesion, // Asignar el propietario
                ':nombre_proveedor' => $nombre_proveedor,
                ':contacto_nombre' => empty($contacto_nombre) ? null : $contacto_nombre,
                ':contacto_telefono' => empty($contacto_telefono) ? null : $contacto_telefono,
                ':contacto_email' => empty($contacto_email) ? null : $contacto_email,
                ':notas_proveedor' => empty($notas_proveedor) ? null : $notas_proveedor,
                ':estado_proveedor' => $estado_proveedor
            ]);
            $mensaje = "Proveedor agregado exitosamente.";
            $tipo_mensaje = 'success';
            break;

        case 'editar':
            if (empty($_POST['id_proveedor']) || empty($_POST['nombre_proveedor'])) {
                throw new Exception("Faltan datos requeridos (ID o Nombre) para la edición.");
            }

            $id_proveedor = filter_var($_POST['id_proveedor'], FILTER_VALIDATE_INT);
            $nombre_proveedor = trim($_POST['nombre_proveedor']);
            $contacto_nombre = trim($_POST['contacto_nombre'] ?? null);
            $contacto_telefono = trim($_POST['contacto_telefono'] ?? null);
            $contacto_email = trim($_POST['contacto_email'] ?? null);
            $notas_proveedor = trim($_POST['notas_proveedor'] ?? null);
            $estado_proveedor = filter_var($_POST['estado_proveedor'], FILTER_VALIDATE_INT);

            if ($id_proveedor === false) throw new Exception("ID de proveedor inválido.");
            
            // --- 3. AÑADIR CONDICIÓN DE PROPIEDAD AL UPDATE ---
            $sql = "UPDATE proveedores SET nombre_proveedor = :nombre_proveedor, contacto_nombre = :contacto_nombre, 
                    contacto_telefono = :contacto_telefono, contacto_email = :contacto_email, 
                    notas_proveedor = :notas_proveedor, estado_proveedor = :estado_proveedor 
                    WHERE id_proveedor = :id_proveedor";
            
            $params = [
                ':nombre_proveedor' => $nombre_proveedor,
                ':contacto_nombre' => empty($contacto_nombre) ? null : $contacto_nombre,
                ':contacto_telefono' => empty($contacto_telefono) ? null : $contacto_telefono,
                ':contacto_email' => empty($contacto_email) ? null : $contacto_email,
                ':notas_proveedor' => empty($notas_proveedor) ? null : $notas_proveedor,
                ':estado_proveedor' => $estado_proveedor,
                ':id_proveedor' => $id_proveedor
            ];

            if ($rol_usuario_sesion !== 'superadmin') {
                $sql .= " AND id_usuario = :id_usuario_sesion";
                $params[':id_usuario_sesion'] = $id_usuario_sesion;
            }

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);

            if ($stmt->rowCount() === 0 && $rol_usuario_sesion !== 'superadmin') {
                throw new Exception("Acción no autorizada. No tienes permiso para modificar este proveedor.");
            }

            $mensaje = "Proveedor actualizado exitosamente.";
            $tipo_mensaje = 'success';
            break;

        case 'cambiar_estado':
            if (empty($_POST['id_proveedor']) || !isset($_POST['nuevo_estado'])) {
                throw new Exception("ID de proveedor o nuevo estado no proporcionado.");
            }
            
            $id_proveedor = filter_var($_POST['id_proveedor'], FILTER_VALIDATE_INT);
            $nuevo_estado = filter_var($_POST['nuevo_estado'], FILTER_VALIDATE_INT);

            if ($id_proveedor === false || $nuevo_estado === false || !in_array($nuevo_estado, [0, 1])) {
                throw new Exception("Datos inválidos.");
            }

            // --- 4. AÑADIR CONDICIÓN DE PROPIEDAD AL CAMBIAR ESTADO ---
            $sql = "UPDATE proveedores SET estado_proveedor = :nuevo_estado WHERE id_proveedor = :id_proveedor";
            $params = [':nuevo_estado' => $nuevo_estado, ':id_proveedor' => $id_proveedor];

            if ($rol_usuario_sesion !== 'superadmin') {
                $sql .= " AND id_usuario = :id_usuario_sesion";
                $params[':id_usuario_sesion'] = $id_usuario_sesion;
            }

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);

            if ($stmt->rowCount() === 0 && $rol_usuario_sesion !== 'superadmin') {
                throw new Exception("Acción no autorizada. No tienes permiso para modificar este proveedor.");
            }

            $mensaje = "Estado del proveedor actualizado exitosamente.";
            $tipo_mensaje = 'success';
            break;
        
        default:
            throw new Exception("Acción no reconocida.");
    }

    $pdo->commit();

} catch (PDOException $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    if ($e->getCode() == 23000) { 
        $mensaje = "Error: Ya existe un proveedor con ese nombre. Por favor, elige otro.";
    } else {
       $mensaje = "Error de base de datos: " . $e->getMessage();
    }
    error_log("Error PDO en proveedores_controlador: " . $e->getMessage());
} catch (Exception $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    $mensaje = $e->getMessage();
}

// Redirigir de vuelta a la página de proveedores con el mensaje
header("Location: /digitalpremium/pages/proveedores.php?mensaje=" . urlencode($mensaje) . "&tipo=" . urlencode($tipo_mensaje));
exit();
?>