<?php
// FILE: /digitalpremium/auth/procesar_login.php
declare(strict_types=1);

if (session_status() === PHP_SESSION_NONE) session_start();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header("Location: /digitalpremium/login.php");
    exit();
}

/* 1) CSRF */
$csrf = $_POST['csrf_token'] ?? '';
if (empty($csrf) || !hash_equals($_SESSION['csrf_token'] ?? '', $csrf)) {
    header("Location: /digitalpremium/login.php?error=csrf");
    exit();
}

/* 2) Conexión y helper del dueño de data */
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../includes/tenant_helper.php';

/* 3) Datos */
$username = trim((string)($_POST['username'] ?? ''));
$password = (string)($_POST['password'] ?? '');

if ($username === '' || $password === '') {
    header("Location: /digitalpremium/login.php?error=campos_vacios");
    exit();
}

try {
    // Soporta $pdo o función db()
    /** @var PDO $db */
    $db = function_exists('db') ? db() : $pdo;

    /* 4) Consulta usuario (incluye 'estado') */
    $stmt = $db->prepare("
        SELECT id_usuario, username, password, nombre_completo, rol, estado
        FROM usuarios
        WHERE username = ?
        LIMIT 1
    ");
    $stmt->execute([$username]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        header("Location: /digitalpremium/login.php?error=credenciales");
        exit();
    }

    /* 5) Verifica estado activo */
    if ((int)$user['estado'] !== 1) {
        header("Location: /digitalpremium/login.php?error=no_activo");
        exit();
    }

    /* 6) Verifica password */
    if (!password_verify($password, $user['password'])) {
        header("Location: /digitalpremium/login.php?error=credenciales");
        exit();
    }

    /* 7) Sesión segura */
    session_regenerate_id(true);
    $_SESSION['id_usuario']      = (int)$user['id_usuario'];
    $_SESSION['username']        = (string)$user['username'];
    $_SESSION['nombre_completo'] = (string)($user['nombre_completo'] ?? '');
    $_SESSION['rol']             = (string)$user['rol']; // superadmin | admin | ventas | cortes

    // Rota token CSRF post-login
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));

    // MUY IMPORTANTE: fijar el DUEÑO ÚNICO DE LA DATA en sesión
    $_SESSION['data_owner_id'] = get_data_owner_id($db);

    /* 8) Auditoría (best-effort, no rompe login si falla) */
    try {
        require_once __DIR__ . '/../includes/audit_helper.php';
        auditarLogin($db, (int)$user['id_usuario'], (string)$user['rol'], $user['username']);
    } catch (Throwable $e) {
        // No bloquea el login si falla la auditoría
    }

    /* 9) Redirección */
    header("Location: /digitalpremium/index.php");
    exit();

} catch (Throwable $e) {
    // Opcional: log de error
    // error_log("Login error: " . $e->getMessage());
    header("Location: /digitalpremium/login.php?error=db_error");
    exit();
}
