<?php
/**
 * FILE: /digitalpremium/healthcheck.php
 * Verifica la conexión PDO y muestra datos básicos de la BD.
 * - HTML bonito por defecto
 * - Modo JSON con ?json=1
 * - Seguro para usar en local (no imprime trazas ni credenciales)
 */

declare(strict_types=1);

// Carga tu conexión (usa el db.php que te pasé antes)
require_once __DIR__ . '/config/db.php';

// Si quieres bloquear este check fuera de local, descomenta:
// if ((getenv('DP_ENV') ?: 'local') !== 'local') { http_response_code(403); exit('Forbidden'); }

$isJson = isset($_GET['json']);
$status = [
    'ok'        => false,
    'message'   => 'Sin verificar',
    'db'        => null,
    'version'   => null,
    'time_zone' => null,
    'tables'    => null,
    'env'       => (getenv('DP_ENV') ?: 'local'),
];

try {
    // Obtén el PDO desde el helper del db.php
    $db = function_exists('db') ? db() : (isset($pdo) ? $pdo : null);
    if (!$db) {
        throw new RuntimeException('No se encontró el objeto PDO. Revisa config/db.php.');
    }

    // Prueba mínima
    $pong = (int)$db->query('SELECT 1')->fetchColumn();

    // Info de versión/BD/tz
    $info = $db->query("
        SELECT
            VERSION()   AS version,
            DATABASE()  AS dbname,
            @@time_zone AS time_zone
    ")->fetch();

    // Conteo de tablas (requiere permisos de lectura de information_schema)
    $tables = (int)$db
        ->query("SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE()")
        ->fetchColumn();

    $status['ok']        = ($pong === 1);
    $status['message']   = $status['ok'] ? 'Conexión exitosa ✅' : 'Conexión establecida, pero la prueba SELECT 1 falló';
    $status['db']        = $info['dbname'] ?? null;
    $status['version']   = $info['version'] ?? null;
    $status['time_zone'] = $info['time_zone'] ?? null;
    $status['tables']    = $tables;

} catch (Throwable $e) {
    http_response_code(500);
    $status['ok'] = false;
    $status['message'] = 'Error de conexión ❌: ' . $e->getMessage();
}

// Salida JSON si se pide ?json=1
if ($isJson) {
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($status, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

// Salida HTML bonita
?>
<!doctype html>
<html lang="es">
<head>
<meta charset="utf-8">
<title>Healthcheck — DigitalPremium</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<style>
  :root { color-scheme: dark light; }
  body { font-family: system-ui, -apple-system, Segoe UI, Roboto, Ubuntu, Cantarell, "Helvetica Neue", Arial, "Noto Sans", "Apple Color Emoji", "Segoe UI Emoji"; margin: 0; padding: 2rem; background: #0b0f14; color: #e6eef5; }
  .card { max-width: 720px; margin: 0 auto; background: #121824; border: 1px solid #1e2a3a; border-radius: 16px; padding: 1.5rem 1.75rem; box-shadow: 0 10px 30px rgba(0,0,0,.25); }
  h1 { margin: 0 0 1rem; font-size: 1.5rem; display: flex; align-items: center; gap: .6rem; }
  .ok { color: #34d399; }
  .fail { color: #f87171; }
  .grid { display: grid; grid-template-columns: 1fr 2fr; gap: .5rem 1rem; margin-top: .5rem; }
  .k { opacity: .8; }
  .footer { margin-top: 1rem; font-size: .9rem; opacity: .75; }
  code { background: #0f1520; padding: .15rem .4rem; border-radius: 6px; }
  a { color: #93c5fd; text-decoration: none; }
  a:hover { text-decoration: underline; }
</style>
</head>
<body>
  <div class="card">
    <h1>
      <span>🔌 Healthcheck BD</span>
      <?php if ($status['ok']): ?>
        <span class="ok">Conexión exitosa</span>
      <?php else: ?>
        <span class="fail">Conexión fallida</span>
      <?php endif; ?>
    </h1>

    <div class="grid">
      <div class="k">Entorno</div><div><?= htmlspecialchars((string)$status['env']) ?></div>
      <div class="k">Base de datos</div><div><?= htmlspecialchars((string)$status['db']) ?></div>
      <div class="k">Versión servidor</div><div><?= htmlspecialchars((string)$status['version']) ?></div>
      <div class="k">Zona horaria</div><div><?= htmlspecialchars((string)$status['time_zone']) ?></div>
      <div class="k"># Tablas</div><div><?= htmlspecialchars((string)$status['tables']) ?></div>
      <div class="k">Mensaje</div><div><?= htmlspecialchars((string)$status['message']) ?></div>
    </div>

    <div class="footer">
      Tip: puedes obtener <strong>JSON</strong> desde <code>?json=1</code> (ej.: <a href="?json=1">healthcheck.php?json=1</a>).
    </div>
  </div>
</body>
</html>
