<?php
// FILE: /digitalpremium/includes/access_helper.php
// Unificado: visibilidad de módulos + ACL por acción (view/create/edit/delete/disable)

if (session_status() === PHP_SESSION_NONE) session_start();

/* =========================
 * Normalización de rol
 * ========================= */
function normalizeRole(?string $rol): string {
    $r = strtolower(trim($rol ?? 'ventas'));
    $r = str_replace(['-', ' '], '_', $r);
    // Canonizamos a: superadmin | admin | ventas | cortes
    if ($r === 'super_admin') $r = 'superadmin';
    return in_array($r, ['superadmin','admin','ventas','cortes'], true) ? $r : 'ventas';
}

/* =========================
 * Landing por rol
 * ========================= */
function defaultLandingForRole(string $role): string {
    $role = normalizeRole($role);
    return $role === 'superadmin'
        ? '/digitalpremium/pages/dashboard.php'
        : '/digitalpremium/pages/perfiles_vendidos.php';
}

/* =========================
 * ACL por acción (fino)
 * =========================
 * Estructura: modulo => [view, create, edit, delete, disable]
 */
function buildACL(string $role): array {
    $role = normalizeRole($role);
    $ALL_FALSE = ['view'=>false,'create'=>false,'edit'=>false,'delete'=>false,'disable'=>false];

    $mods = [
        'dashboard','dashboard_analytics',
        'clientes','plataformas','cuentas_maestras','perfiles_vendidos','proveedores',
        'finanzas','usuarios','config',
    ];
    $acl = [];
    foreach ($mods as $m) $acl[$m] = $ALL_FALSE;

    switch ($role) {
        case 'superadmin':
        case 'admin':
            foreach ($acl as $k => $_) {
                $acl[$k] = ['view'=>true,'create'=>true,'edit'=>true,'delete'=>true,'disable'=>true];
            }
            break;

        case 'ventas':
            // Vendedores: pueden ver/crear/editar; NO eliminar/NO dar de baja
            $acl['clientes']          = ['view'=>true,'create'=>true,'edit'=>true,'delete'=>false,'disable'=>false];
            $acl['cuentas_maestras']  = ['view'=>true,'create'=>true,'edit'=>true,'delete'=>false,'disable'=>false];
            $acl['perfiles_vendidos'] = ['view'=>true,'create'=>true,'edit'=>true,'delete'=>false,'disable'=>false];
            // Lo demás oculto
            break;

        case 'cortes':
            // Cortes: pueden EDITAR CUENTAS MAESTRAS (NO crear, NO eliminar)
            $acl['cuentas_maestras']  = ['view'=>true,'create'=>false,'edit'=>true,'delete'=>false,'disable'=>false];
            // Perfiles vendidos solo lectura (según último pedido: NO eliminar)
            $acl['perfiles_vendidos'] = ['view'=>true,'create'=>false,'edit'=>false,'delete'=>false,'disable'=>false];
            // Lo demás oculto
            break;
    }

    // Dashboards/finanzas/usuarios/config visibles solo para superadmin/admin
    if (in_array($role, ['superadmin','admin'], true)) {
        $acl['dashboard']['view']           = true;
        $acl['dashboard_analytics']['view'] = true;
        $acl['finanzas']['view']            = true;
        $acl['usuarios']['view']            = true;
        $acl['config']['view']              = true;
    }

    return $acl;
}

/** Comprueba permiso fino */
function can(string $module, string $action): bool {
    if (empty($_SESSION['ACL']) || !is_array($_SESSION['ACL'])) {
        $role = normalizeRole($_SESSION['rol'] ?? 'ventas');
        $_SESSION['ACL'] = buildACL($role);
    }
    return !empty($_SESSION['ACL'][$module][$action]);
}

/* ===========================================================
 * COMPATIBILIDAD: visibilidad de módulos (menús) + overrides
 * =========================================================== */

/** Visibilidad por rol (para menú). No controla acciones; solo “mostrar/ocultar”. */
function defaultRoleModules(string $rol): array {
    $role = normalizeRole($rol);
    $all = [
        'dashboard',
        'clientes',
        'plataformas',
        'cuentas_maestras',
        'perfiles_vendidos',
        'proveedores',
        'finanzas',
        'usuarios',
        'config',
        'reportes',        // si no lo usas, no pasa nada
        'cortes',          // idem
        'configuraciones', // idem
        'gestion_usuarios',// idem
        'auditoria'        // idem
    ];

    // Arranque: todos ocultos
    $base = array_fill_keys($all, false);

    if ($role === 'superadmin' || $role === 'admin') {
        foreach ($base as $k => $_) $base[$k] = true;
    } elseif ($role === 'ventas') {
        $base['clientes']          = true;
        $base['cuentas_maestras']  = true;
        $base['perfiles_vendidos'] = true;
        // Dashboard oculto para que el landing sea Perfiles Vendidos
    } elseif ($role === 'cortes') {
        $base['cuentas_maestras']  = true;
        $base['perfiles_vendidos'] = true; // lectura
    }

    return $base;
}

/** Overrides en DB (si existen). Si no, no rompe. */
function fetchDbOverrides(PDO $db, int $id_usuario, string $rol): array {
    $out = [];
    try {
        $stmt = $db->prepare("SELECT modulo, permitido FROM roles_accesos WHERE rol = ?");
        $stmt->execute([normalizeRole($rol)]);
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $r) {
            $out[$r['modulo']] = (bool)$r['permitido'];
        }
    } catch (Throwable $e) {}
    try {
        $stmt = $db->prepare("SELECT modulo, permitido FROM usuarios_accesos WHERE id_usuario = ?");
        $stmt->execute([$id_usuario]);
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $r) {
            $out[$r['modulo']] = (bool)$r['permitido'];
        }
    } catch (Throwable $e) {}
    return $out;
}

/** Carga visibilidad de módulos en sesión (para el sidebar) */
function loadModuleAccessToSession(PDO $db, int $id_usuario, string $rol): void {
    $base = defaultRoleModules($rol);
    $ovr  = fetchDbOverrides($db, $id_usuario, $rol);
    foreach ($ovr as $mod => $perm) $base[$mod] = (bool)$perm;

    $_SESSION['accesos'] = $base;
    $_SESSION['modulos_permitidos'] = array_keys(array_filter($base));
}

/** Helpers de menú */
function hasModuleAccess(string $mod): bool {
    return !empty($_SESSION['accesos'][$mod]);
}
function requireModuleAccess(string $mod): void {
    if (!hasModuleAccess($mod)) {
        header("Location: /digitalpremium/pages/perfiles_vendidos.php?mensaje=" . urlencode("No tienes acceso a '$mod'.") . "&tipo=danger");
        exit();
    }
}
