<?php
/**
 * Helper de Auditoría para DigitalPremium
 * 
 * Este archivo contiene funciones para registrar todas las acciones de los usuarios
 * en la tabla auditoria_acciones de manera consistente.
 */

/**
 * Registra una acción en la auditoría
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario que realiza la acción
 * @param string $rol_usuario Rol del usuario
 * @param string $accion Tipo de acción (create, update, delete, login, logout, renovar, renew, etc.)
 * @param string $entidad Entidad afectada (usuarios, perfiles_vendidos, plataformas, clientes, etc.)
 * @param int $entidad_id ID de la entidad afectada
 * @param array $detalles Detalles adicionales de la acción (opcional)
 * @return bool True si se registró correctamente, False en caso contrario
 */
function registrarAuditoria(PDO $pdo, int $id_usuario, string $rol_usuario, string $accion, string $entidad, int $entidad_id, array $detalles = []): bool {
    try {
        // Verificar que la tabla existe
        $sql_check = "SHOW TABLES LIKE 'auditoria_acciones'";
        $stmt_check = $pdo->prepare($sql_check);
        $stmt_check->execute();
        
        if ($stmt_check->rowCount() === 0) {
            // Crear la tabla si no existe
            crearTablaAuditoria($pdo);
        }
        
        // Insertar el registro de auditoría
        $sql = "INSERT INTO auditoria_acciones 
                (id_usuario, rol_usuario, accion, entidad, entidad_id, detalle_json, ip, user_agent) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([
            $id_usuario,
            $rol_usuario,
            $accion,
            $entidad,
            $entidad_id,
            json_encode($detalles, JSON_UNESCAPED_UNICODE),
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
        
        return $result;
        
    } catch (Exception $e) {
        // Log del error (no bloquear la operación principal)
        error_log("Error en auditoría: " . $e->getMessage());
        return false;
    }
}

/**
 * Crea la tabla auditoria_acciones si no existe
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @return bool True si se creó correctamente o ya existía
 */
function crearTablaAuditoria(PDO $pdo): bool {
    try {
        $sql = "
        CREATE TABLE IF NOT EXISTS `auditoria_acciones` (
            `id_audit` int(11) NOT NULL AUTO_INCREMENT,
            `id_usuario` int(11) NOT NULL,
            `rol_usuario` varchar(50) NOT NULL,
            `accion` varchar(50) NOT NULL,
            `entidad` varchar(100) NOT NULL,
            `entidad_id` int(11) NOT NULL,
            `detalle_json` longtext DEFAULT NULL,
            `ip` varchar(45) DEFAULT NULL,
            `user_agent` text DEFAULT NULL,
            `creado_en` timestamp NOT NULL DEFAULT current_timestamp(),
            PRIMARY KEY (`id_audit`),
            KEY `idx_usuario` (`id_usuario`),
            KEY `idx_entidad` (`entidad`, `entidad_id`),
            KEY `idx_accion` (`accion`),
            KEY `idx_creado_en` (`creado_en`),
            KEY `idx_rol_usuario` (`rol_usuario`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        ";
        
        $stmt = $pdo->prepare($sql);
        return $stmt->execute();
        
    } catch (Exception $e) {
        error_log("Error creando tabla auditoría: " . $e->getMessage());
        return false;
    }
}

/**
 * Función de auditoría simplificada para uso en controladores
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $uid ID del usuario
 * @param string $rol Rol del usuario
 * @param string $accion Acción realizada
 * @param string $entidad Entidad afectada
 * @param int $entidad_id ID de la entidad
 * @param array $det Detalles adicionales
 */
function audit(PDO $pdo, int $uid, string $rol, string $accion, string $entidad, int $entidad_id, array $det = []): void {
    registrarAuditoria($pdo, $uid, $rol, $accion, $entidad, $entidad_id, $det);
}

/**
 * Función de auditoría para acciones de usuarios
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param string $accion Acción realizada
 * @param array $detalle Detalles adicionales
 */
function auditar(PDO $pdo, int $id_usuario, string $rol, string $accion, array $detalle = []): void {
    registrarAuditoria($pdo, $id_usuario, $rol, $accion, 'usuarios', $id_usuario, $detalle);
}

/**
 * Registra auditoría de login
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param string $username Username del usuario
 */
function auditarLogin(PDO $pdo, int $id_usuario, string $rol, string $username): void {
    $detalles = [
        'username' => $username,
        'tipo' => 'login',
        'fecha' => date('Y-m-d H:i:s')
    ];
    registrarAuditoria($pdo, $id_usuario, $rol, 'login', 'usuarios', $id_usuario, $detalles);
}

/**
 * Registra auditoría de logout
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 */
function auditarLogout(PDO $pdo, int $id_usuario, string $rol): void {
    $detalles = [
        'tipo' => 'logout',
        'fecha' => date('Y-m-d H:i:s')
    ];
    registrarAuditoria($pdo, $id_usuario, $rol, 'logout', 'usuarios', $id_usuario, $detalles);
}

/**
 * Registra auditoría de creación de perfil vendido
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param int $id_perfil ID del perfil vendido
 * @param array $detalles Detalles del perfil
 */
function auditarCrearPerfil(PDO $pdo, int $id_usuario, string $rol, int $id_perfil, array $detalles = []): void {
    registrarAuditoria($pdo, $id_usuario, $rol, 'create', 'perfiles_vendidos', $id_perfil, $detalles);
}

/**
 * Registra auditoría de renovación de perfil
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param int $id_perfil ID del perfil vendido
 * @param array $detalles Detalles de la renovación
 */
function auditarRenovarPerfil(PDO $pdo, int $id_usuario, string $rol, int $id_perfil, array $detalles = []): void {
    registrarAuditoria($pdo, $id_usuario, $rol, 'renovar', 'perfiles_vendidos', $id_perfil, $detalles);
}

/**
 * Registra auditoría de eliminación de perfil
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param int $id_perfil ID del perfil vendido
 * @param array $detalles Detalles de la eliminación
 */
function auditarEliminarPerfil(PDO $pdo, int $id_usuario, string $rol, int $id_perfil, array $detalles = []): void {
    registrarAuditoria($pdo, $id_usuario, $rol, 'delete', 'perfiles_vendidos', $id_perfil, $detalles);
}

/**
 * Registra auditoría de edición de perfil
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param int $id_perfil ID del perfil vendido
 * @param array $detalles Detalles de la edición
 */
function auditarEditarPerfil(PDO $pdo, int $id_usuario, string $rol, int $id_perfil, array $detalles = []): void {
    registrarAuditoria($pdo, $id_usuario, $rol, 'update', 'perfiles_vendidos', $id_perfil, $detalles);
}

/**
 * Registra auditoría de creación de plataforma
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param int $id_plataforma ID de la plataforma
 * @param array $detalles Detalles de la plataforma
 */
function auditarCrearPlataforma(PDO $pdo, int $id_usuario, string $rol, int $id_plataforma, array $detalles = []): void {
    registrarAuditoria($pdo, $id_usuario, $rol, 'create', 'plataformas', $id_plataforma, $detalles);
}

/**
 * Registra auditoría de edición de plataforma
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param int $id_plataforma ID de la plataforma
 * @param array $detalles Detalles de la edición
 */
function auditarEditarPlataforma(PDO $pdo, int $id_usuario, string $rol, int $id_plataforma, array $detalles = []): void {
    registrarAuditoria($pdo, $id_usuario, $rol, 'update', 'plataformas', $id_plataforma, $detalles);
}

/**
 * Registra auditoría de creación de cliente
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param int $id_cliente ID del cliente
 * @param array $detalles Detalles del cliente
 */
function auditarCrearCliente(PDO $pdo, int $id_usuario, string $rol, int $id_cliente, array $detalles = []): void {
    registrarAuditoria($pdo, $id_usuario, $rol, 'create', 'clientes', $id_cliente, $detalles);
}

/**
 * Registra auditoría de edición de cliente
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param int $id_cliente ID del cliente
 * @param array $detalles Detalles de la edición
 */
function auditarEditarCliente(PDO $pdo, int $id_usuario, string $rol, int $id_cliente, array $detalles = []): void {
    registrarAuditoria($pdo, $id_usuario, $rol, 'update', 'clientes', $id_cliente, $detalles);
}

/**
 * Registra auditoría de eliminación de cliente
 * 
 * @param PDO $pdo Conexión a la base de datos
 * @param int $id_usuario ID del usuario
 * @param string $rol Rol del usuario
 * @param int $id_cliente ID del cliente
 * @param array $detalles Detalles de la eliminación
 */
function auditarEliminarCliente(PDO $pdo, int $id_usuario, string $rol, int $id_cliente, array $detalles = []): void {
    registrarAuditoria($pdo, $id_usuario, $rol, 'delete', 'clientes', $id_cliente, $detalles);
}



