<?php
// Archivo: includes/config_functions.php

if (!function_exists('getConfigValue')) {
    /**
     * Obtiene el valor de una configuración específica desde la base de datos.
     * Cachea los valores en una variable estática para la duración de la solicitud.
     *
     * @param string $clave La clave de la configuración a obtener.
     * @param mixed $default Valor a devolver si la clave no se encuentra.
     * @return mixed El valor de la configuración o el valor por defecto.
     */
    function getConfigValue(string $clave, $default = null) {
        static $config_cache = null;
        global $pdo; // Asume que $pdo está disponible globalmente (desde db.php)

        if ($config_cache === null) {
            $config_cache = [];
            if ($pdo) {
                try {
                    $stmt = $pdo->query("SELECT clave_config, valor_config FROM configuraciones");
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $config_cache[$row['clave_config']] = $row['valor_config'];
                    }
                } catch (PDOException $e) {
                    // Manejar el error, quizás loguearlo o devolver array vacío
                    // error_log("Error al cargar configuraciones: " . $e->getMessage());
                    // Por ahora, si hay error, el caché permanecerá vacío y se usará el default.
                }
            }
        }

        return $config_cache[$clave] ?? $default;
    }
}

if (!function_exists('setConfigValue')) {
    /**
     * Establece o actualiza el valor de una configuración específica en la base de datos.
     *
     * @param string $clave La clave de la configuración a establecer.
     * @param mixed $valor El nuevo valor para la configuración.
     * @return bool True si fue exitoso, False en caso contrario.
     */
    function setConfigValue(string $clave, $valor): bool {
        global $pdo; // Asume que $pdo está disponible globalmente

        if (!$pdo) {
            return false;
        }

        try {
            $stmt = $pdo->prepare("UPDATE configuraciones SET valor_config = :valor WHERE clave_config = :clave");
            $stmt->execute([':valor' => $valor, ':clave' => $clave]);

            // Para invalidar el caché en la misma solicitud si se actualiza un valor
            // (aunque getConfigValue se llama usualmente antes que setConfigValue en la misma carga de página)
            // O simplemente no invalidar y la próxima carga de página lo leerá actualizado.
            // Para ser más robusto, podrías tener una función para recargar el caché.

            return $stmt->rowCount() > 0; // Devuelve true si alguna fila fue afectada
        } catch (PDOException $e) {
            // error_log("Error al establecer configuración " . $clave . ": " . $e->getMessage());
            return false;
        }
    }
}
?>