<?php
// Archivo: includes/message_helper.php (Versión FINAL Y CENTRALIZADA de todas las funciones auxiliares)

// Asegurarse de que $pdo esté disponible globalmente (asumiendo que db.php lo configura)
global $pdo;

/**
 * Función para obtener valores de configuración de la base de datos.
 * Esta función es crucial para centralizar los mensajes y DEBE SER MULTIUSUARIO.
 * @param string $key La clave de la configuración a buscar.
 * @param int $id_usuario El ID del usuario actual.
 * @param mixed $default El valor por defecto si la clave no se encuentra.
 * @return string|null El valor de la configuración o el valor por defecto.
 */
if (!function_exists('getConfigValue')) {
    function getConfigValue($key, $id_usuario, $default = null) {
        global $pdo; // Asegura acceso a la conexión PDO
        if (!$pdo) {
            error_log("Error: PDO connection not available in getConfigValue for key {$key}.");
            return $default;
        }
        try {
            // 1. Intentar obtener la configuración específica del usuario (PRIORIDAD AL USUARIO)
            $stmt = $pdo->prepare("SELECT valor_config FROM configuraciones WHERE clave_config = :key AND id_usuario = :id_usuario");
            $stmt->execute([':key' => $key, ':id_usuario' => $id_usuario]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($result) {
                return $result['valor_config'];
            }

            // 2. Si no se encuentra para el usuario, intentar obtener la configuración global (id_usuario IS NULL)
            $stmt = $pdo->prepare("SELECT valor_config FROM configuraciones WHERE clave_config = :key AND id_usuario IS NULL");
            $stmt->execute([':key' => $key]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            return $result ? $result['valor_config'] : $default;

        } catch (PDOException | Exception $e) { 
            error_log("Error en getConfigValue para clave {$key}, usuario {$id_usuario}: " . $e->getMessage());
            return null; 
        }
    }
}

/**
 * Limpia un número de teléfono para ser usado en enlaces de WhatsApp.
 * Elimina caracteres no numéricos y añade prefijo de país si aplica.
 * @param string $telefono El número de teléfono a limpiar.
 * @return string El número de teléfono limpio, listo para WhatsApp.
*/
if (!function_exists('limpiarNumeroWhatsApp')) {
    function limpiarNumeroWhatsApp(string $telefono): string {
        $telefono_limpio = preg_replace('/[^0-9]/', '', $telefono);

        // Lógica para añadir prefijo de país (AJUSTAR SEGÚN NECESIDAD)
        // Ejemplo para Perú (código 51, números móviles suelen ser de 9 dígitos y empezar con 9)
        if (strlen($telefono_limpio) == 9 && substr($telefono_limpio, 0, 1) === '9') {
            return '51' . $telefono_limpio; 
        }
        // Puedes añadir más lógicas para otros países si tu base de datos no guarda el prefijo internacional
        // Ejemplo: if (strlen($telefono_limpio) == 9 && !str_starts_with($telefono_limpio, '51')) $telefono_limpio = '51' . $telefono_limpio;

        return $telefono_limpio;
    }
}

if (!function_exists('generarMensajeParaCliente')) {
    /**
     * Genera un mensaje formateado para el cliente, usando plantillas de configuración.
     * Puede generar un mensaje para uno o varios perfiles vendidos.
     *
     * @param PDO $pdo Objeto PDO para la conexión a la BD.
     * @param int|array $id_perfil_vendido_o_array ID de un perfil vendido o un array de IDs de perfiles vendidos.
     * @param int $id_usuario_sesion ID del usuario actual de la sesión (para obtener configuraciones del usuario logueado).
     * @param string $tipo_mensaje 'bienvenida' o 'recordatorio_vencimiento'.
     * @param string|null $tipo_activacion_form Valor del tipo de mensaje seleccionado en el formulario (si aplica).
     * @return string|null El mensaje formateado o null si hay error.
     */
    function generarMensajeParaCliente(PDO $pdo, $id_perfil_vendido_o_array, int $id_usuario_sesion, string $tipo_mensaje = 'bienvenida', ?string $tipo_activacion_form = null): ?string
    {
        // --- INICIO DEBUG: Comprobar acceso a $pdo dentro de la función ---
        if (!isset($pdo) || !$pdo instanceof PDO) {
            error_log("DEBUG: \$pdo NO está disponible en generarMensajeParaCliente.");
            return null; // Salir si PDO no está disponible
        } else {
            error_log("DEBUG: \$pdo está disponible en generarMensajeParaCliente.");
        }
        // --- FIN DEBUG ---

        $perfiles_a_procesar = is_array($id_perfil_vendido_o_array) ? $id_perfil_vendido_o_array : [$id_perfil_vendido_o_array];
        $mensajes_individuales = [];
        $primer_cliente_nombre = ''; // Para el saludo consolidado

        foreach ($perfiles_a_procesar as $id_perfil_vendido) {
            try {
                // Obtener datos necesarios para el mensaje
                $stmt_datos = $pdo->prepare("
                    SELECT 
                        pv.*, 
                        c.nombre_completo AS cliente_nombre, 
                        c.telefono, 
                        p.nombre_plataforma AS servicio_nombre, 
                        cm.email_cuenta, 
                        cm.password_cuenta,
                        pv.nombre_perfil_cliente AS perfil_nombre, 
                        pv.pin_perfil_cliente AS perfil_pin, 
                        pv.fecha_fin_servicio AS fecha_vencimiento_db,
                        DATEDIFF(pv.fecha_fin_servicio, CURDATE()) as dias_restantes_num,
                        pv.id_usuario AS id_usuario_propietario_perfil,
                        p.max_perfiles_cuenta -- Necesario para inferir el tipo de servicio
                    FROM perfiles_vendidos pv
                    JOIN clientes c ON pv.id_cliente = c.id_cliente
                    JOIN cuentas_maestras cm ON pv.id_cuenta_maestra = cm.id_cuenta_maestra
                    JOIN plataformas p ON cm.id_plataforma = p.id_plataforma
                    WHERE pv.id_perfil_vendido = :id_perfil_vendido
                ");
                $stmt_datos->execute([':id_perfil_vendido' => $id_perfil_vendido]);
                $datos = $stmt_datos->fetch(PDO::FETCH_ASSOC);

                if (!$datos) {
                    error_log("No se encontraron datos para generar mensaje para id_perfil_vendido: " . $id_perfil_vendido);
                    continue; // Skip to the next profile if data not found
                }

                if (empty($primer_cliente_nombre)) {
                    $primer_cliente_nombre = $datos['cliente_nombre'] ?? '';
                }

                // Usar el ID del usuario de la sesión para las configuraciones, no el del perfil vendido,
                // ya que las configuraciones de mensaje son por el usuario logueado.
                $id_usuario_para_config = $id_usuario_sesion;

                // === LÓGICA CLAVE: Inferir TIPO_SERVICIO_VENDIDO (Perfil o Cuenta Completa) ===
                // Esta lógica es crucial para la intuitividad del mensaje.
                $servicio_nombre_original = $datos['servicio_nombre'] ?? '';
                $perfil_nombre_original = $datos['perfil_nombre'] ?? '';
                $servicio_nombre_lower = strtolower($servicio_nombre_original);
                $perfil_nombre_lower = strtolower($perfil_nombre_original);
                
                $tipo_servicio_vendido_para_cliente = 'Perfil'; // Asunción por defecto, la mayoría serán perfiles.

                // Casos donde podría ser "Cuenta Completa" (para el cliente)
                if (strpos($servicio_nombre_lower, 'cuenta completa') !== false || strpos($servicio_nombre_lower, 'familiar') !== false) {
                    // Si la plataforma es "Cuenta Completa" o "Familiar":
                    // Se considera "Cuenta Completa" si el nombre del perfil es genérico (ej. "Cuenta 1", "Principal") 
                    // o si el nombre del perfil contiene "Completa"
                    // O si la plataforma tiene max_perfiles_cuenta = 1 (vendida como unidad indivisible).
                    if ( ($perfil_nombre_lower === 'principal' || str_contains($perfil_nombre_lower, 'completa') || str_contains($perfil_nombre_lower, 'total')) || ($datos['max_perfiles_cuenta'] == 1 && !str_contains($servicio_nombre_lower, 'perfil')) ) {
                        $tipo_servicio_vendido_para_cliente = 'Cuenta Completa';
                    } else {
                        // Si la plataforma es "Cuenta Completa/Familiar" pero el perfil tiene un nombre específico (ej. "Juanito", "Sala Niños"),
                        // es un perfil individual dentro de una cuenta compartida.
                        $tipo_servicio_vendido_para_cliente = 'Perfil';
                    }
                } elseif (str_contains($servicio_nombre_lower, 'perfil') || str_contains($servicio_nombre_lower, 'dispositivos')) {
                    // Si la plataforma ya se llama "Netflix - PERFIL" o "CapCut - DISPOSITIVOS", es un perfil.
                    $tipo_servicio_vendido_para_cliente = 'Perfil';
                }
                // FIN LÓGICA DE DETECCIÓN DE TIPO_SERVICIO_VENDIDO


                // Preparar valores para los marcadores de posición
                $reemplazos_base = [
                    '[CLIENTE_NOMBRE]' => htmlspecialchars($datos['cliente_nombre'] ?? ''),
                    '[SERVICIO_NOMBRE]' => htmlspecialchars($servicio_nombre_original), 
                    '[TIPO_SERVICIO_VENDIDO]' => htmlspecialchars($tipo_servicio_vendido_para_cliente), 
                    '[CUENTA_EMAIL]' => htmlspecialchars($datos['email_cuenta'] ?? ''), 
                    '[CUENTA_PASSWORD]' => htmlspecialchars($datos['password_cuenta'] ?? ''), 
                    '[PERFIL_NOMBRE]' => htmlspecialchars($perfil_nombre_original),
                    '[PERFIL_PIN]' => htmlspecialchars($datos['pin_perfil_cliente'] ?? 'No aplica'), 
                    '[FECHA_VENCIMIENTO]' => date("d/m/Y", strtotime($datos['fecha_vencimiento_db'] ?? 'now')), 
                ];

                $mensaje_actual = "";

                if ($tipo_mensaje == 'bienvenida') {
                    // Se utiliza el valor del formulario si se proporciona, de lo contrario, se recurre a la configuración.
                    $tipo_activacion_to_use = $tipo_activacion_form ?? getConfigValue('tipo_mensaje_activacion', $id_usuario_para_config, 'perfil_con_pass'); 
                    
                    // --- MODIFICACIONES A LOS VALORES POR DEFECTO PARA EL MENSAJE DE BIENVENIDA ---
                    // IMPORTANTE: Si tienes configuraciones en tu BD para estas claves, esas se usarán en su lugar.
                    // Para que esto funcione consistentemente, asegúrate de que no haya configuraciones personalizadas
                    // en la BD para estas claves, o actualízalas en la BD con los valores a continuación.

                    // General Greeting and Service Ready
                    $saludo_y_intro = getConfigValue('msg_postventa_saludo_intro', $id_usuario_para_config, 
                        "*¡Hola [CLIENTE_NOMBRE]!* 👋\n\n" .
                        "🎉 *Tu servicio de [SERVICIO_NOMBRE] está listo.*\n" .
                        "✨ *¡Disfrútalo con acceso garantizado!* ✨"
                    );

                    // Details Access Header
                    $intro_acceso_template = getConfigValue('msg_postventa_intro_credenciales', $id_usuario_para_config, 
                        "*Detalles de acceso:*"
                    );
                    
                    // Template for profile with password
                    $credenciales_plantilla_cuerpo_perfil_con_pass = getConfigValue('msg_postventa_perfil_con_pass', $id_usuario_para_config, 
                        "📧 *Email:* [CUENTA_EMAIL]\n" .
                        "🔑 *Contraseña:* [CUENTA_PASSWORD]\n" .
                        "👤 *Perfil:* [PERFIL_NOMBRE]\n" .
                        "🔒 *PIN:* [PERFIL_PIN]"
                    );
                    
                    // Template for profile TV-only (no change from previous)
                    $credenciales_plantilla_cuerpo_perfil_solo_tv = getConfigValue('msg_postventa_perfil_solo_tv', $id_usuario_para_config, 
                        'Abre la app de [SERVICIO_NOMBRE] en tu TV. Verás un código en pantalla (generalmente alfanumérico). Por favor, bríndame ese código para activarlo remotamente. 🖥️✨\n\nUsuario asociado: [CUENTA_EMAIL]\nPerfil: [PERFIL_NOMBRE]\nPIN: [PERFIL_PIN]'
                    );

                    // Template for full account with password (no change from previous)
                    $credenciales_plantilla_cuerpo_cuenta_completa_con_pass = getConfigValue('msg_postventa_cuenta_completa_con_pass', $id_usuario_para_config, 
                        '📧 Email: [CUENTA_EMAIL]\n🔑 Contraseña: [CUENTA_PASSWORD]\n\nRecuerda crear tus perfiles y establecer un PIN si la plataforma lo permite.'
                    );
                    
                    // Template for full account TV-only (no change from previous)
                    $credenciales_plantilla_cuerpo_cuenta_completa_solo_tv = getConfigValue('msg_postventa_cuenta_completa_solo_tv', $id_usuario_para_config, 
                        'Abre la app de [SERVICIO_NOMBRE] en tu TV. Verás un código en pantalla (generalmente alfanumérico). Por favor, bríndame ese código para activarla remotamente. 🖥️✨\n\nUsuario asociado: [CUENTA_EMAIL]\n\nRecuerda crear tus perfiles y establecer un PIN si la plataforma lo permite.'
                    );

                    // Service Validity
                    $vencimiento_info = getConfigValue('msg_postventa_vencimiento', $id_usuario_para_config, 
                        "📍 *Vigencia del servicio: Acceso garantizado hasta el [FECHA_VENCIMIENTO].*"
                    );
                    
                    // Usage Disclaimer
                    $disclaimer = getConfigValue('msg_postventa_disclaimer', $id_usuario_para_config, 
                        "⚠️ *Recuerda: El uso está permitido en un solo dispositivo a la vez.*"
                    );
                    
                    // Closing Remark
                    $despedida = getConfigValue('msg_postventa_despedida', $id_usuario_para_config, 
                        "😊 *¡Gracias por confiar en nosotros y disfruta tu servicio!*"
                    );

                    $credenciales_plantilla_cuerpo = "";
                    
                    // === Lógica mejorada para seleccionar la plantilla de credenciales/activación ===
                    switch ($tipo_activacion_to_use) {
                        case 'perfil_con_pass':
                            $credenciales_plantilla_cuerpo = $credenciales_plantilla_cuerpo_perfil_con_pass;
                            break;
                        case 'perfil_solo_tv':
                            $credenciales_plantilla_cuerpo = $credenciales_plantilla_cuerpo_perfil_solo_tv;
                            break;
                        case 'cuenta_completa_con_pass':
                            $credenciales_plantilla_cuerpo = $credenciales_plantilla_cuerpo_cuenta_completa_con_pass;
                            break;
                        case 'cuenta_completa_solo_tv':
                            $credenciales_plantilla_cuerpo = $credenciales_plantilla_cuerpo_cuenta_completa_solo_tv;
                            break;
                        default:
                            // Fallback to a general format if configuration is unknown/null
                            $credenciales_plantilla_cuerpo = "📧 *Email:* [CUENTA_EMAIL]\n🔑 *Contraseña:* [CUENTA_PASSWORD]\n👤 *Perfil:* [PERFIL_NOMBRE]\n🔒 *PIN:* [PERFIL_PIN]\n\n(Revisa la configuración de 'Formato de Mensaje de Activación')";
                            error_log("Configuración 'tipo_mensaje_activacion' desconocida o nula para usuario {$id_usuario_para_config}: " . $tipo_activacion_to_use);
                            break;
                    }
                    
                                         // Final construction of the welcome message
                     // For multiple profiles, we'll build the message without saludo_y_intro to avoid duplication
                     if (count($perfiles_a_procesar) > 1) {
                         // Multiple profiles: include service name and service-specific content
                         $mensaje_actual = "🎯 *[SERVICIO_NOMBRE]*\n\n" .
                                           $intro_acceso_template . "\n\n" .
                                           $credenciales_plantilla_cuerpo . "\n\n" .
                                           $vencimiento_info . "\n\n" .
                                           $disclaimer;
                     } else {
                         // Single profile: include full message with saludo and despedida
                         $mensaje_actual = $saludo_y_intro . "\n\n" .
                                           $intro_acceso_template . "\n" .
                                           $credenciales_plantilla_cuerpo . "\n\n" .
                                           $vencimiento_info . "\n\n" .
                                           $disclaimer . "\n\n" .
                                           $despedida;
                     }


                } elseif ($tipo_mensaje == 'recordatorio_vencimiento') {
                    $dias_restantes_num = (int)$datos['dias_restantes_num'];
                    $dias_texto = '';

                    // Determinar el texto de días restantes y el emoji
                    if ($dias_restantes_num <= 0) {
                        $dias_texto = 'VENCIÓ' . ($dias_restantes_num < 0 ? ' hace ' . abs($dias_restantes_num) . ' día(s)' : '') . ' 🔴';
                    } else {
                        $dias_texto = 'en ' . $dias_restantes_num . ' día(s) ⏳';
                    }
                    
                    $reemplazos_base['[DIAS_RESTANTES_TEXTO]'] = $dias_texto;
                    $reemplazos_base['[FECHA_VENCIMIENTO_FORMATO]'] = $reemplazos_base['[FECHA_VENCIMIENTO]'];

                    // --- INICIO DEBUG: Valores de plantillas de recordatorio ---
                    $saludo_rec = getConfigValue('msg_recordatorio_saludo', $id_usuario_para_config, "¡Hola [CLIENTE_NOMBRE]! 👋 (Default)");
                    $cuerpo_rec = getConfigValue('msg_recordatorio_cuerpo', $id_usuario_para_config, "🔔 Te recordamos que tu suscripción a [SERVICIO_NOMBRE] ([TIPO_SERVICIO_VENDIDO]) está próxima a vencer el [FECHA_VENCIMIENTO_FORMATO] ([DIAS_RESTANTES_TEXTO]). (Default)");
                    $accion_rec = getConfigValue('msg_recordatorio_accion', $id_usuario_para_config, "Para renovar tu servicio y seguir disfrutando sin interrupciones, por favor contáctanos. ¡Que tengas un buen día! ✨ (Default)");
                    error_log("DEBUG: msg_recordatorio_saludo: " . ($saludo_rec ? substr($saludo_rec, 0, 50) . '...' : 'VACÍO/NULL'));
                    error_log("DEBUG: msg_recordatorio_cuerpo: " . ($cuerpo_rec ? substr($cuerpo_rec, 0, 50) . '...' : 'VACÍO/NULL'));
                    error_log("DEBUG: msg_recordatorio_accion: " . ($accion_rec ? substr($accion_rec, 0, 50) . '...' : 'VACÍO/NULL'));
                    // --- FIN DEBUG ---

                    $cuerpo_final_recordatorio = $cuerpo_rec; 

                    if ($dias_restantes_num <= 0) {
                        $cuerpo_final_recordatorio = str_replace(
                            ['está próxima a vencer el', 'próxima a vencer el'], 
                            'VENCIÓ el', 
                            $cuerpo_final_recordatorio
                        );
                        $cuerpo_final_recordatorio = str_replace('([DIAS_RESTANTES_TEXTO])', '', $cuerpo_final_recordatorio);
                    }

                    $mensaje_actual = $saludo_rec . "\n\n" . $cuerpo_final_recordatorio . "\n\n" . $accion_rec;

                } else {
                    error_log("Tipo de mensaje no reconocido: " . $tipo_mensaje);
                    continue; // Skip to next profile
                }

                // Reemplazar todos los marcadores de posición en el mensaje actual
                foreach ($reemplazos_base as $placeholder => $valor) {
                    $mensaje_actual = str_replace($placeholder, $valor, $mensaje_actual);
                }
                
                $mensajes_individuales[] = $mensaje_actual;

            } catch (PDOException | Exception $e) { 
                error_log("Error de BD al generar mensaje para cliente (perfil {$id_perfil_vendido}): " . $e->getMessage());
                continue; // Continue to next profile on error
            }
        }

        // Unir todos los mensajes individuales if there is more than one
        if (count($mensajes_individuales) > 1) {
            // For multiple profiles, create a cleaner, more organized message
            $saludo_general = getConfigValue('msg_postventa_saludo_intro_multiple', $id_usuario_sesion, 
                "*¡Hola [CLIENTE_NOMBRE]!* 👋\n\n" .
                "🎉 *Tus servicios están listos.*\n" .
                "✨ *¡Disfrútalos con acceso garantizado!* ✨"
            );
            $despedida_general = getConfigValue('msg_postventa_despedida_multiple', $id_usuario_sesion, 
                "😊 *¡Gracias por confiar en nosotros y disfruta tus servicios!*"
            );

            // Build a cleaner consolidated message
            $mensaje_final_consolidado = $saludo_general . "\n\n";
            
            // Add each service with better formatting
            foreach ($mensajes_individuales as $index => $mensaje_individual) {
                if ($index > 0) {
                    $mensaje_final_consolidado .= "\n" . str_repeat('─', 40) . "\n\n";
                }
                $mensaje_final_consolidado .= $mensaje_individual;
            }
            
            $mensaje_final_consolidado .= "\n\n" . $despedida_general;
            
            // Replace the client name in the consolidated message (taken from the first profile)
            $mensaje_final_consolidado = str_replace('[CLIENTE_NOMBRE]', htmlspecialchars($primer_cliente_nombre), $mensaje_final_consolidado);

            return $mensaje_final_consolidado;

        } elseif (count($mensajes_individuales) === 1) {
            return $mensajes_individuales[0]; // If it's just one profile, return its message directly
        } else {
            return null; // No messages generated
        }
    }
}

/**
 * Obtiene el mensaje configurado para un tipo de cambio específico
 * @param PDO $pdo Conexión a la base de datos
 * @param string $tipo_cambio Tipo de cambio (email, password, plataforma, estado, general)
 * @param int $id_usuario ID del usuario (opcional, para configuraciones específicas)
 * @return string Mensaje configurado o mensaje por defecto
 */
function getMensajeCambioCuenta($pdo, $tipo_cambio, $id_usuario = null) {
    $clave_config = "mensaje_cambio_{$tipo_cambio}";
    
    try {
        // Intentar obtener configuración específica del usuario
        if ($id_usuario) {
            $stmt = $pdo->prepare("SELECT valor_config FROM configuraciones WHERE clave_config = :clave AND id_usuario = :id_usuario");
            $stmt->execute([':clave' => $clave_config, ':id_usuario' => $id_usuario]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($result) {
                return $result['valor_config'];
            }
        }
        
        // Si no hay configuración específica del usuario, obtener la global
        $stmt = $pdo->prepare("SELECT valor_config FROM configuraciones WHERE clave_config = :clave AND id_usuario IS NULL");
        $stmt->execute([':clave' => $clave_config]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            return $result['valor_config'];
        }
        
        // Mensajes por defecto si no hay configuración
        $mensajes_default = [
            'email' => 'Hola {nombre_cliente}, te informamos que hemos actualizado el email de tu cuenta {plataforma} por motivos de seguridad. El nuevo email es: {nuevo_email}. Por favor, actualiza tus credenciales de acceso.',
            'password' => 'Hola {nombre_cliente}, hemos actualizado la contraseña de tu cuenta {plataforma} por motivos de seguridad. Te enviaremos la nueva contraseña por un medio seguro.',
            'plataforma' => 'Hola {nombre_cliente}, hemos migrado tu cuenta a una nueva plataforma {nueva_plataforma} para mejorar tu experiencia. Tus datos y configuraciones se han transferido correctamente.',
            'estado' => 'Hola {nombre_cliente}, tu cuenta {plataforma} ha sido actualizada. Si tienes algún problema de acceso, por favor contáctanos inmediatamente.',
            'general' => 'Hola {nombre_cliente}, hemos realizado actualizaciones en tu cuenta {plataforma}. Si experimentas algún problema, no dudes en contactarnos.'
        ];
        
        return $mensajes_default[$tipo_cambio] ?? $mensajes_default['general'];
        
    } catch (PDOException $e) {
        error_log("Error al obtener mensaje de cambio de cuenta: " . $e->getMessage());
        return "Hola {nombre_cliente}, hemos realizado cambios en tu cuenta {plataforma}. Por favor contáctanos si tienes alguna pregunta.";
    }
}

/**
 * Procesa un mensaje reemplazando las variables con los valores reales
 * @param string $mensaje Mensaje con variables
 * @param array $datos Datos para reemplazar las variables
 * @return string Mensaje procesado
 */
function procesarMensajeCambio($mensaje, $datos) {
    $variables = [
        '{nombre_cliente}' => $datos['nombre_cliente'] ?? 'Cliente',
        '{plataforma}' => $datos['plataforma'] ?? 'la plataforma',
        '{nuevo_email}' => $datos['nuevo_email'] ?? 'el nuevo email',
        '{nueva_plataforma}' => $datos['nueva_plataforma'] ?? 'la nueva plataforma',
        '{tipo_cambio}' => $datos['tipo_cambio'] ?? 'cambios'
    ];
    
    return str_replace(array_keys($variables), array_values($variables), $mensaje);
}

// Función para obtener detalles de estado de cuenta maestra para la UI
if (!function_exists('getAccountStatusDetails')) {
    function getAccountStatusDetails(array $cuenta, int $dias_alerta): array {
        $dias_restantes = intval($cuenta['dias_restantes_sql'] ?? 0); 
        $details = [ 
            'status_key' => 'inactive', 
            'status_badge' => '<span class="badge fs-6 bg-secondary-subtle text-secondary-emphasis">Inactiva</span>', 
            'card_class' => '', 
            'progress_bar_class' => 'bg-secondary', 
            'progress_bar_value' => 0, 
            'vencimiento_class' => '' 
        ];

        switch ($cuenta['estado_cuenta']) {
            case 1: // Activa
                if ($dias_restantes <= 0) { // Vencida
                    $details = array_merge($details, ['status_key' => 'expired', 'status_badge' => '<span class="badge fs-6 bg-danger-subtle text-danger-emphasis">Vencida</span>', 'card_class' => 'card-glow-danger', 'progress_bar_class' => 'bg-danger', 'progress_bar_value' => 100, 'vencimiento_class' => 'text-danger fw-bold']);
                }
                elseif ($dias_restantes <= $dias_alerta && $dias_restantes > 0) { // Por Vencer
                    $details = array_merge($details, [
                        'status_key' => 'expiring', 
                        'status_badge' => '<span class="badge fs-6 bg-warning-subtle text-warning-emphasis">Por Vencer</span>', 
                        'card_class' => 'card-glow-warning', 
                        'progress_bar_class' => 'bg-warning',
                        'progress_bar_value' => max(0, min(100, (($dias_alerta - $dias_restantes) / $dias_alerta) * 100)) 
                    ]);
                } else { // Activa y no próxima a vencer
                    $details = array_merge($details, [
                        'status_key' => 'active', 
                        'status_badge' => '<span class="badge fs-6 bg-success-subtle text-success-emphasis">Activa</span>', 
                        'card_class' => '', 
                        'progress_bar_class' => 'bg-success', 
                        'progress_bar_value' => min(100, ($dias_restantes / 30) * 100)
                    ]); 
                }
                break;
            case 0: // Inactiva
                $details = array_merge($details, [
                    'status_key' => 'inactive',
                    'status_badge' => '<span class="badge fs-6 bg-secondary-subtle text-secondary-emphasis">Inactiva</span>', 
                    'card_class' => '', 
                    'progress_bar_class' => 'bg-secondary', 
                    'progress_bar_value' => 0
                ]);
                break;
            case 2: // Con Problemas
                $details = array_merge($details, ['status_key' => 'problem', 'status_badge' => '<span class="badge fs-6 bg-danger-subtle text-danger-emphasis">Con Problemas</span>', 'card_class' => 'card-glow-danger', 'progress_bar_class' => 'bg-danger', 'progress_bar_value' => 100, 'vencimiento_class' => 'text-danger fw-bold']); 
                break;
        }
        return $details;
    }
}

// Función para obtener icono de plataforma (Mueve esta función aquí si no está en otro helper)
if (!function_exists('getPlatformIcon')) {
    function getPlatformIcon(string $platform_name): string {
        $platform_name = strtolower($platform_name);
        if (str_contains($platform_name, 'netflix')) return 'fa-tv';
        if (str_contains($platform_name, 'disney')) return 'fa-magic-wand-sparkles';
        if (str_contains($platform_name, 'hbo') || str_contains($platform_name, 'max')) return 'fa-crown';
        if (str_contains($platform_name, 'amazon') || str_contains($platform_name, 'prime')) return 'fa-box-open';
        if (str_contains($platform_name, 'spotify')) return 'fa-music';
        if (str_contains($platform_name, 'youtube')) return 'fa-youtube';
        if (str_contains($platform_name, 'paramount')) return 'fa-mountain';
        if (str_contains($platform_name, 'crunchyroll')) return 'fa-dragon';
        if (str_contains($platform_name, 'canva')) return 'fa-paint-brush';
        if (str_contains($platform_name, 'iptv')) return 'fa-satellite-dish';
        if (str_contains($platform_name, 'capcut')) return 'fa-video';
        if (str_contains($platform_name, 'directv')) return 'fa-satellite';
        if (str_contains($platform_name, 'movistar')) return 'fa-wifi';
        if (str_contains($platform_name, 'flujo tv')) return 'fa-stream';
        if (str_contains($platform_name, 'viki')) return 'fa-heart';
        return 'fa-video'; 
    }
}