<?php
/**
 * Script de Instalación para Sistema de Notificación de Cambios
 * DigitalPremium - Cuentas Maestras
 */

// Verificar si se ejecuta desde línea de comandos o web
$is_cli = php_sapi_name() === 'cli';

function output($message, $type = 'info') {
    global $is_cli;
    
    if ($is_cli) {
        $colors = [
            'info' => "\033[36m",    // Cyan
            'success' => "\033[32m", // Green
            'warning' => "\033[33m", // Yellow
            'error' => "\033[31m",   // Red
            'reset' => "\033[0m"     // Reset
        ];
        echo $colors[$type] . $message . $colors['reset'] . PHP_EOL;
    } else {
        $classes = [
            'info' => 'alert-info',
            'success' => 'alert-success',
            'warning' => 'alert-warning',
            'error' => 'alert-danger'
        ];
        echo '<div class="alert ' . $classes[$type] . '">' . htmlspecialchars($message) . '</div>';
    }
}

function checkRequirements() {
    output("🔍 Verificando requisitos del sistema...", 'info');
    
    $requirements = [
        'PHP Version (>= 7.4)' => version_compare(PHP_VERSION, '7.4.0', '>='),
        'PDO Extension' => extension_loaded('pdo'),
        'PDO MySQL Extension' => extension_loaded('pdo_mysql'),
        'JSON Extension' => extension_loaded('json'),
        'cURL Extension' => extension_loaded('curl')
    ];
    
    $all_ok = true;
    foreach ($requirements as $requirement => $status) {
        if ($status) {
            output("✅ " . $requirement, 'success');
        } else {
            output("❌ " . $requirement, 'error');
            $all_ok = false;
        }
    }
    
    return $all_ok;
}

function checkDatabaseConnection() {
    output("🔌 Verificando conexión a la base de datos...", 'info');
    
    try {
        require_once 'config/db.php';
        global $pdo;
        
        // Verificar conexión
        $pdo->query('SELECT 1');
        output("✅ Conexión a la base de datos exitosa", 'success');
        
        // Verificar tabla configuraciones
        $stmt = $pdo->query("SHOW TABLES LIKE 'configuraciones'");
        if ($stmt->rowCount() > 0) {
            output("✅ Tabla 'configuraciones' encontrada", 'success');
            return true;
        } else {
            output("❌ Tabla 'configuraciones' no encontrada", 'error');
            return false;
        }
        
    } catch (Exception $e) {
        output("❌ Error de conexión: " . $e->getMessage(), 'error');
        return false;
    }
}

function installConfigurations() {
    output("⚙️ Instalando configuraciones de mensajes...", 'info');
    
    try {
        global $pdo;
        
        // Leer archivo SQL
        $sql_file = 'config/insert_mensajes_config.sql';
        if (!file_exists($sql_file)) {
            output("❌ Archivo SQL no encontrado: " . $sql_file, 'error');
            return false;
        }
        
        $sql_content = file_get_contents($sql_file);
        $statements = explode(';', $sql_content);
        
        $pdo->beginTransaction();
        
        foreach ($statements as $statement) {
            $statement = trim($statement);
            if (!empty($statement)) {
                $pdo->exec($statement);
            }
        }
        
        $pdo->commit();
        output("✅ Configuraciones instaladas correctamente", 'success');
        return true;
        
    } catch (Exception $e) {
        if (isset($pdo)) {
            $pdo->rollBack();
        }
        output("❌ Error al instalar configuraciones: " . $e->getMessage(), 'error');
        return false;
    }
}

function verifyInstallation() {
    output("🔍 Verificando instalación...", 'info');
    
    try {
        global $pdo;
        
        // Verificar que las configuraciones estén instaladas
        $stmt = $pdo->query("SELECT COUNT(*) FROM configuraciones WHERE clave_config LIKE 'mensaje_cambio_%'");
        $count = $stmt->fetchColumn();
        
        if ($count >= 5) {
            output("✅ Se encontraron {$count} configuraciones de mensajes", 'success');
            
            // Mostrar configuraciones instaladas
            $stmt = $pdo->query("SELECT clave_config, nombre_visible FROM configuraciones WHERE clave_config LIKE 'mensaje_cambio_%' ORDER BY clave_config");
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                output("   - " . $row['nombre_visible'], 'info');
            }
            
            return true;
        } else {
            output("❌ Solo se encontraron {$count} configuraciones (se esperaban al menos 5)", 'error');
            return false;
        }
        
    } catch (Exception $e) {
        output("❌ Error al verificar instalación: " . $e->getMessage(), 'error');
        return false;
    }
}

function checkFiles() {
    output("📁 Verificando archivos del sistema...", 'info');
    
    $required_files = [
        'actions/cuentas_maestras_controlador.php',
        'includes/message_helper.php',
        'pages/cuentas_maestras.php',
        'js/notificacion_cambios.js'
    ];
    
    $all_ok = true;
    foreach ($required_files as $file) {
        if (file_exists($file)) {
            output("✅ " . $file, 'success');
        } else {
            output("❌ " . $file, 'error');
            $all_ok = false;
        }
    }
    
    return $all_ok;
}

function main() {
    output("🚀 Instalador del Sistema de Notificación de Cambios", 'info');
    output("DigitalPremium - Cuentas Maestras", 'info');
    output("", 'info');
    
    // Verificar requisitos
    if (!checkRequirements()) {
        output("❌ No se cumplen los requisitos del sistema", 'error');
        return false;
    }
    
    // Verificar conexión a BD
    if (!checkDatabaseConnection()) {
        output("❌ No se puede conectar a la base de datos", 'error');
        return false;
    }
    
    // Verificar archivos
    if (!checkFiles()) {
        output("❌ Faltan archivos del sistema", 'error');
        return false;
    }
    
    // Instalar configuraciones
    if (!installConfigurations()) {
        output("❌ Error al instalar configuraciones", 'error');
        return false;
    }
    
    // Verificar instalación
    if (!verifyInstallation()) {
        output("❌ Error en la verificación de instalación", 'error');
        return false;
    }
    
    output("", 'info');
    output("🎉 ¡Instalación completada exitosamente!", 'success');
    output("", 'info');
    output("📋 Próximos pasos:", 'info');
    output("1. Ir a Configuración del Sistema", 'info');
    output("2. Buscar la sección 'Mensajes de Cambio'", 'info');
    output("3. Personalizar los mensajes según tus necesidades", 'info');
    output("4. Probar editando una cuenta maestra", 'info');
    output("", 'info');
    output("📚 Documentación: README_NOTIFICACION_CAMBIOS.md", 'info');
    
    return true;
}

// Ejecutar instalación
if ($is_cli) {
    main();
} else {
    // Interfaz web
    echo '<!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Instalador - Sistema de Notificación de Cambios</title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
        <style>
            body { padding: 2rem; background-color: #f8f9fa; }
            .container { max-width: 800px; }
            .alert { margin-bottom: 1rem; }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h3 class="mb-0">🚀 Instalador del Sistema de Notificación de Cambios</h3>
                    <p class="mb-0">DigitalPremium - Cuentas Maestras</p>
                </div>
                <div class="card-body">';
    
    main();
    
    echo '</div>
            </div>
        </div>
    </body>
    </html>';
}
?>
