<?php
// FILE: /digitalpremium/pages/cuentas_maestras.php
// VERSION CON MEJORASf DE DISEÑO AVANZADAS: NOTIFICACIONES, MODALES INTELIGENTES Y MÁS

// MODO DE DEPURACIÓN: Cámbialo a 'true' para ver el estado detallado de todas tus cuentas en la parte superior.
$debug_mode = false;

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['id_usuario'])) {
    header("Location: /digitalpremium/login.php");
    exit();
}

require_once '../config/db.php';
require_once '../includes/message_helper.php';

// --- FUNCIONES PHP (REVISADAS Y OPTIMIZADAS) ---
if (!function_exists('getConfigValue')) {
    function getConfigValue($key, $id_usuario, $default = null) {
        global $pdo;
        try {
            $stmt = $pdo->prepare("SELECT valor_config FROM configuraciones WHERE clave_config = :key AND id_usuario = :id_usuario");
            $stmt->execute([':key' => $key, ':id_usuario' => $id_usuario]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$result) {
                $stmt = $pdo->prepare("SELECT valor_config FROM configuraciones WHERE clave_config = :key AND id_usuario IS NULL");
                $stmt->execute([':key' => $key]);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
            }
            
            return $result ? $result['valor_config'] : $default;
        } catch (PDOException $e) { error_log("Error al obtener config value {$key}: " . $e->getMessage()); return $default; }
    }
}

// Function to get the user filter clause dynamically for a given alias
if (!function_exists('getUserFilterClause')) {
    function getUserFilterClause(string $alias, string $rol_usuario_sesion): string {
        return ($rol_usuario_sesion !== 'superadmin') ? " AND {$alias}.id_usuario = :id_usuario_sesion" : "";
    }
}

// Esta función es crucial para el diseño de las tarjetas, se movió aquí
function getAccountStatusDetails(array $cuenta, int $dias_alerta): array {
    $dias_restantes = intval($cuenta['dias_restantes_sql']);
    $details = [    
        'status_key' => 'inactive',    
        'status_badge' => '<span class="badge status-badge badge-inactive">Inactiva</span>',    
        'card_class' => '',    
        'progress_bar_class' => 'bg-secondary',    
        'progress_bar_value' => 0,    
        'vencimiento_class' => '',
        'dias_restantes_text' => ''
    ];

    switch ($cuenta['estado_cuenta']) {
        case 1: // Activa
            if ($dias_restantes < 0) {    
                $details = array_merge($details, [
                    'status_key' => 'expired',    
                    'status_badge' => '<span class="badge status-badge badge-expired">Vencida</span>',    
                    'card_class' => 'account-card-expired',    
                    'progress_bar_class' => 'bg-danger',    
                    'progress_bar_value' => 100,    
                    'vencimiento_class' => 'text-danger fw-bold',
                    'dias_restantes_text' => 'Vencida hace ' . abs($dias_restantes) . ' día' . (abs($dias_restantes) != 1 ? 's' : '')
                ]);
            } elseif ($dias_restantes == 0) {
                    $details = array_merge($details, [
                    'status_key' => 'expiring',    
                    'status_badge' => '<span class="badge status-badge badge-expiring">Vence Hoy</span>',    
                    'card_class' => 'account-card-expiring',    
                    'progress_bar_class' => 'bg-warning',
                    'progress_bar_value' => 100, // Vence hoy, barra llena de warning
                    'dias_restantes_text' => 'Vence HOY'
                ]);
            }
            elseif ($dias_restantes <= $dias_alerta) {    
                $details = array_merge($details, [
                    'status_key' => 'expiring',    
                    'status_badge' => '<span class="badge status-badge badge-expiring">Por Vencer</span>',    
                    'card_class' => 'account-card-expiring',    
                    'progress_bar_class' => 'bg-warning',
                    'progress_bar_value' => max(0, min(100, ($dias_restantes / $dias_alerta) * 100)), // Escala el progreso a los días de alerta
                    'dias_restantes_text' => 'Vence en ' . $dias_restantes . ' día' . ($dias_restantes != 1 ? 's' : '')
                ]);
            } else {    
                $details = array_merge($details, [
                    'status_key' => 'active',    
                    'status_badge' => '<span class="badge status-badge badge-active">Activa</span>',    
                    'card_class' => 'account-card-active',    
                    'progress_bar_class' => 'bg-success',    
                    'progress_bar_value' => min(100, ($dias_restantes / 30) * 100), // Asume base de 30 días para progreso general
                    'dias_restantes_text' => $dias_restantes . ' día' . ($dias_restantes != 1 ? 's' : '') . ' restantes'
                ]);    
            }
            break;
        case 0: // Inactiva
            $details = array_merge($details, [
                'status_key' => 'inactive',
                'status_badge' => '<span class="badge status-badge badge-inactive">Inactiva</span>',    
                'card_class' => 'account-card-inactive',    
                'progress_bar_class' => 'bg-secondary',    
                'progress_bar_value' => 0,
                'dias_restantes_text' => 'Inactiva'
            ]);
            break;
        case 2: // Con Problemas
            $details = array_merge($details, [
                'status_key' => 'problem',    
                'status_badge' => '<span class="badge status-badge badge-problem">Problema</span>',    
                'card_class' => 'account-card-problem',    
                'progress_bar_class' => 'bg-danger',    
                'progress_bar_value' => 100,    
                'vencimiento_class' => 'text-danger fw-bold',
                'dias_restantes_text' => 'Con Problemas'
            ]);    
            break;
    }
    return $details;
}

// Iconos para plataformas (usando fab para marcas, fas para genéricos)
function getPlatformIcon(string $platform_name): string {
    $platform_name = strtolower($platform_name);
    if (str_contains($platform_name, 'netflix')) return 'fab fa-netflix';
    if (str_contains($platform_name, 'disney')) return 'fas fa-mouse-pointer'; // No hay un fab para Disney universal
    if (str_contains($platform_name, 'hbo') || str_contains($platform_name, 'max')) return 'fas fa-tv'; // O 'fa-crown' si es para MAX
    if (str_contains($platform_name, 'prime') || str_contains($platform_name, 'amazon')) return 'fab fa-amazon';
    if (str_contains($platform_name, 'spotify')) return 'fab fa-spotify';
    if (str_contains($platform_name, 'youtube')) return 'fab fa-youtube';
    if (str_contains($platform_name, 'paramount')) return 'fas fa-mountain';
    if (str_contains($platform_name, 'crunchyroll')) return 'fas fa-star';
    if (str_contains($platform_name, 'canva')) return 'fas fa-paint-brush';
    if (str_contains($platform_name, 'iptv')) return 'fas fa-satellite-dish';
    if (str_contains($platform_name, 'capcut')) return 'fas fa-video';
    if (str_contains($platform_name, 'directv')) return 'fas fa-satellite';
    if (str_contains($platform_name, 'movistar')) return 'fas fa-wifi';
    if (str_contains($platform_name, 'flujo tv')) return 'fas fa-stream';
    if (str_contains($platform_name, 'viki')) return 'fas fa-heart';
    if (str_contains($platform_name, 'eset')) return 'fas fa-shield-alt';
    if (str_contains($platform_name, 'blackbox')) return 'fas fa-box-open';
    if (str_contains($platform_name, 'vix')) return 'fas fa-film';
    if (str_contains($platform_name, 'gemini')) return 'fas fa-gem';
    if (str_contains($platform_name, 'chat gpt')) return 'fas fa-robot';
    
    return 'fas fa-globe'; // Icono por defecto más genérico
}


// --- OBTENCIÓN Y PROCESAMIENTO DE DATOS ---
$page_title = "Dashboard de Cuentas - DigitalPremium";
$id_usuario_sesion = $_SESSION['id_usuario'];
$rol_usuario_sesion = $_SESSION['rol'];

// DUEÑO de la data (todos los usuarios de un tenant ven lo mismo)
$OWNER_ID = (int)($_SESSION['data_owner_id'] ?? $_SESSION['id_usuario']);

$moneda_simbolo = getConfigValue('moneda_simbolo', $OWNER_ID, 'S/');
$dias_alerta_maestras_global = (int)getConfigValue('dias_alerta_maestras', $OWNER_ID, 7);

$plataformas_activas = $pdo->query("SELECT id_plataforma, nombre_plataforma, max_perfiles_cuenta FROM plataformas WHERE estado = 1 ORDER BY nombre_plataforma ASC")->fetchAll(PDO::FETCH_ASSOC);

$sql_metodos = "SELECT id_proveedor, nombre_proveedor
                     FROM proveedores
                     WHERE estado_proveedor = 1
                     ORDER BY nombre_proveedor ASC";
$stmt_metodos = $pdo->prepare($sql_metodos);
$stmt_metodos->execute();

$metodos_activos = $stmt_metodos->fetchAll(PDO::FETCH_ASSOC);

$where_clauses = [];
$params = [];
// TODOS los roles pueden ver TODAS las cuentas maestras
// NO se aplica filtro de owner


// Filtros para la consulta principal (para persistencia en la URL y recargas)
$filter_platform_id = $_GET['filtro_plataforma'] ?? 'all';
$filter_status = $_GET['filtro_status'] ?? 'all';
$filter_search_text = $_GET['filtro_search_text'] ?? '';

if (isset($_GET['filtro_plataforma']) && $filter_platform_id !== '' && $filter_platform_id !== 'all') {
    $where_clauses[] = "cm.id_plataforma = :filtro_plataforma";
    $params[':filtro_plataforma'] = $filter_platform_id;
}
if (isset($_GET['filtro_search_text']) && $filter_search_text !== '') {
    $where_clauses[] = "(cm.email_cuenta LIKE :search_text OR cm.nombre_proveedor LIKE :search_text OR p.nombre_plataforma LIKE :search_text)";
    $params[':search_text'] = '%' . $filter_search_text . '%';
}

// Manejo de filtros de estado desde la URL para la consulta SQL
if ($filter_status !== 'all') {
    if ($filter_status === '1') {
        $where_clauses[] = "cm.estado_cuenta = 1";
    } elseif ($filter_status === '0') {
        $where_clauses[] = "cm.estado_cuenta = 0";
    } elseif ($filter_status === '2') {
        $where_clauses[] = "cm.estado_cuenta = 2";
    }
}


$sql_where = !empty($where_clauses) ? " WHERE " . implode(" AND ", $where_clauses) : "";
$sql_cuentas = "SELECT cm.*, p.nombre_plataforma, p.max_perfiles_cuenta, 
                        DATEDIFF(cm.fecha_vencimiento_proveedor, CURDATE()) as dias_restantes_sql,
                        COALESCE(perfil_stats.perfiles_vendidos, 0) as perfiles_vendidos_reales
                 FROM cuentas_maestras cm 
                 JOIN plataformas p ON cm.id_plataforma = p.id_plataforma 
                 LEFT JOIN (
                     SELECT 
                         id_cuenta_maestra,
                         COUNT(*) as perfiles_vendidos
                     FROM perfiles_vendidos 
                     WHERE estado_suscripcion = 1
                     GROUP BY id_cuenta_maestra
                 ) perfil_stats ON cm.id_cuenta_maestra = perfil_stats.id_cuenta_maestra
                 $sql_where 
                 ORDER BY p.nombre_plataforma ASC, cm.estado_cuenta DESC, cm.perfiles_disponibles DESC, cm.fecha_vencimiento_proveedor ASC";
$stmt_cuentas = $pdo->prepare($sql_cuentas);
// Recorta $params a los placeholders reales del SQL para evitar HY093
preg_match_all('/:\w+/', $sql_cuentas, $m);
$placeholders = array_unique($m[0] ?? []);
$final_params = [];
foreach ($params as $k => $v) {
    $kk = (strpos($k, ':') === 0) ? $k : ':' . $k;
    if (in_array($kk, $placeholders, true)) {
        $final_params[$kk] = $v;
    }
}
$stmt_cuentas->execute($final_params);

$cuentas_maestras_raw = $stmt_cuentas->fetchAll(PDO::FETCH_ASSOC);


// --- CÁLCULO DE ESTADÍSTICAS ---
$total_cuentas = count($cuentas_maestras_raw);
$total_perfiles_disponibles = 0;
$cuentas_activas_count = 0;    
$alert_accounts = [];
$grouped_accounts = [];    

foreach ($cuentas_maestras_raw as $cuenta) {
    // FIX: Ensure id_plataforma is set before using it as a key for grouping
    if (!isset($cuenta['id_plataforma']) || $cuenta['id_plataforma'] === null) {
        // Log an error if a platform ID is missing, and skip this account
        error_log("Missing or NULL id_plataforma for account ID: " . $cuenta['id_cuenta_maestra'] . " - Skipping grouping.");
        continue;
    }

    $platform_name = $cuenta['nombre_plataforma'];
    $grouped_accounts[$platform_name][] = $cuenta;

    $status_details = getAccountStatusDetails($cuenta, $dias_alerta_maestras_global);
    
    // Solo contar perfiles de cuentas activas
    if ($cuenta['estado_cuenta'] == 1) {
        $cuentas_activas_count++;
        $total_perfiles_disponibles += $cuenta['perfiles_disponibles'];
    }

    if (in_array($status_details['status_key'], ['expired', 'expiring', 'problem'])) {
        $alert_accounts[] = array_merge($cuenta, ['status_details' => $status_details]);
    }
}

// Ordenar las cuentas en alerta para que las más urgentes aparezcan primero
usort($alert_accounts, function($a, $b) {
    // Las cuentas con problemas van primero
    if ($a['status_details']['status_key'] === 'problem' && $b['status_details']['status_key'] !== 'problem') return -1;
    if ($b['status_details']['status_key'] === 'problem' && $a['status_details']['status_key'] !== 'problem') return 1;
    
    // Luego, las expiradas (días restantes más negativos)
    if ($a['status_details']['status_key'] === 'expired' && $b['status_details']['status_key'] !== 'expired') return -1;
    if ($b['status_details']['status_key'] === 'expired' && $a['status_details']['status_key'] !== 'expired') return 1;

    // Finalmente, por días restantes (expiring, menor es más urgente)
    return $a['dias_restantes_sql'] <=> $b['dias_restantes_sql'];
});


include_once '../includes/header.php'; // Asegúrate de que este archivo incluya Bootstrap CSS y JS, y FontAwesome.
?>

<style>
    /* VARIABLES DE COLOR Y DISEÑO */
    :root {
        --font-primary: 'Poppins', sans-serif;
        --body-bg: #f5f8ff;
        --card-bg: #ffffff;
        --card-border: #e0e5f0;
        --text-dark: #212529;
        --text-muted: #6c757d;
        --text-secondary: #607d8b;
        --primary-color: #6a1b9a;
        --primary-light: #8e24aa;
        --primary-dark: #4a148c;
        --primary-color-rgb: 106, 27, 154;

        --success-color: #28a745;
        --success-color-rgb: 40, 167, 69;
        --warning-color: #ffc107;
        --danger-color: #dc3545;
        --info-color: #17a2b8;

        --shadow-sm: 0 1px 3px rgba(0,0,0,0.06);
        --shadow-md: 0 5px 15px rgba(0,0,0,0.08);
        --shadow-lg: 0 10px 30px rgba(0,0,0,0.12);
        --border-radius-sm: 0.5rem;
        --border-radius-md: 0.85rem;
        --border-radius-lg: 1.25rem;
    }

    /* GENERALES */
    body {
        background-color: var(--body-bg);
        font-family: var(--font-primary);
        color: var(--text-dark);
        line-height: 1.6;
    }

    .page-container {
        padding-top: 2.5rem;
        padding-bottom: 2.5rem;
    }

    /* ENCABEZADO DE PÁGINA */
    .page-header {
        margin-bottom: 2.5rem;
        border-bottom: 1px solid var(--card-border);
        padding-bottom: 1.5rem;
        display: flex;
        align-items: center;
        justify-content: space-between;
        flex-wrap: wrap;
    }
    .page-header h1 {
        font-weight: 800;
        font-size: 2.5rem;
        color: var(--primary-color);
        margin-bottom: 0.5rem;
    }
    .page-header h1 i {
        color: var(--primary-color);
        font-size: 1.1em;
    }
    .page-header p {
        font-size: 1rem;
        color: var(--text-muted) !important;
    }
    .page-header .btn-primary {
        background: linear-gradient(45deg, var(--primary-color), var(--primary-light));
        border: none;
        box-shadow: var(--shadow-md);
        transition: all 0.2s ease-in-out;
        font-weight: 600;
        padding: 0.75rem 1.5rem;
        border-radius: var(--border-radius-md);
        margin-left: 1rem;
        white-space: nowrap;
    }
    .page-header .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: var(--shadow-lg);
        filter: brightness(1.1);
    }
    .page-header .btn-primary i {
        margin-right: 0.5rem;
    }

    /* TARJETAS DE ESTADÍSTICAS */
    .stat-card {
        background: var(--card-bg);
        border: none;
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow-md);
        padding: 1.5rem;
        display: flex;
        align-items: center;
        transition: all 0.2s ease-in-out;
        overflow: hidden;
        position: relative;
        z-index: 1;
        height: 100%;
    }
    .stat-card::before {
        content: '';
        position: absolute;
        top: 0; left: 0; right: 0; bottom: 0;
        background: linear-gradient(135deg, rgba(var(--primary-color-rgb),0.05), transparent 60%);
        border-radius: var(--border-radius-lg);
        z-index: -1;
    }
    .stat-card:hover {
        transform: translateY(-5px);
        box-shadow: var(--shadow-lg);
    }
    .stat-card .stat-icon {
        font-size: 1.6rem;
        width: 60px; height: 60px;
        border-radius: 50%;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        margin-right: 1.25rem;
        flex-shrink: 0;
        box-shadow: 0 5px 10px rgba(0,0,0,0.1);
    }
    .stat-icon.bg-soft-primary { background: linear-gradient(45deg, var(--primary-color), var(--primary-light)); color: white; }
    .stat-icon.bg-soft-success { background: linear-gradient(45deg, var(--success-color), #69F0AE); color: white; }
    .stat-icon.bg-soft-warning { background: linear-gradient(45deg, var(--warning-color), #FFD54F); color: var(--text-dark); }
    
    .stat-card .stat-title {
        font-size: 0.95rem;
        font-weight: 500;
        color: var(--text-secondary);
        margin-bottom: 0.2rem;
        text-transform: uppercase;
        letter-spacing: 0.05em;
    }
    .stat-card .stat-value {
        font-size: 2rem;
        font-weight: 800;
        color: var(--text-dark);
    }

    /* SECCIÓN DE ALERTAS */
    .alerts-section {
        background: var(--card-bg);
        border: none;
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow-lg);
        padding: 2rem;
        position: relative;
        overflow: hidden;
        margin-bottom: 2.5rem;
    }
    .alerts-section::before {
        content: '';
        position: absolute;
        top: 0; left: 0; bottom: 0;
        width: 8px;
        background: linear-gradient(to bottom, var(--warning-color), #FFA000);
        border-radius: var(--border-radius-lg) 0 0 var(--border-radius-lg);
        z-index: 1;
    }
    .alerts-header {
        color: var(--text-dark);
        font-weight: 700;
        font-size: 1.6rem;
        display: flex;
        align-items: center;
        margin-bottom: 1.5rem;
        padding-left: 1rem;
    }
    .alerts-header i {
        margin-right: 0.8rem;
        color: var(--warning-color);
        font-size: 1.8rem;
    }
    .alerts-header .badge {
        background-color: var(--warning-color) !important;
        color: var(--text-dark) !important;
        padding: 0.5em 0.9em;
        border-radius: 50rem;
        font-size: 0.9rem;
        margin-left: 0.8rem;
        font-weight: 700;
    }
    .list-group-item {
        background-color: var(--body-bg);
        border: 1px solid var(--card-border);
        border-radius: var(--border-radius-md);
        padding: 1rem 1.25rem;
        margin-bottom: 0.75rem;
        transition: all 0.2s ease;
        box-shadow: var(--shadow-sm);
        display: flex;
        align-items: center;
        justify-content: space-between;
        flex-wrap: wrap;
    }
    .list-group-item:last-child { margin-bottom: 0; }
    .list-group-item:hover {
        border-color: var(--primary-color);
        transform: translateX(5px);
        box-shadow: var(--shadow-md);
    }
    .alert-platform-name {
        font-weight: 700;
        color: var(--text-dark);
        font-size: 1rem;
        display: flex;
        align-items: center;
        margin-bottom: 0.1rem;
    }
    .alert-platform-name .fas, .alert-platform-name .fab {
        font-size: 1.1rem;
        margin-right: 0.6rem;
        color: var(--primary-color);
    }
    .alert-account-email {
        color: var(--text-muted);
        font-size: 0.85rem;
        word-break: break-all;
    }
    .alert-detail-text {
        font-size: 0.95rem;
        font-weight: 600;
        margin-left: 1rem;
        flex-shrink: 0;
    }
    .alert-detail-text.text-danger { color: var(--danger-color) !important; }
    .alert-detail-text.text-warning-emphasis { color: var(--warning-color) !important; }

    /* BARRA DE CONTROLES (FILTROS) */
    .controls-bar {
        background-color: var(--card-bg);
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow-md);
        padding: 1.25rem 1.75rem;
        margin-bottom: 2.5rem;
    }
    .input-icon-group {
        position: relative;
    }
    .input-icon-group .input-icon {
        position: absolute;
        top: 50%; left: 1rem;
        transform: translateY(-50%);
        color: var(--text-secondary);
        pointer-events: none;
        font-size: 0.95rem;
        z-index: 2;
    }
    .input-icon-group .form-control,
    .input-icon-group .form-select {
        padding-left: 2.5rem;
        border-radius: 0.6rem;
        height: 45px;
        font-size: 0.95rem;
        border-color: var(--card-border);
        box-shadow: none;
        transition: border-color 0.2s ease, box-shadow 0.2s ease;
    }
    .input-icon-group .form-control:focus,
    .input-icon-group .form-select:focus {
        border-color: var(--primary-color);
        box-shadow: 0 0 0 0.2rem rgba(var(--primary-color-rgb), 0.2);
    }
    .controls-bar .btn-outline-secondary {
        border-radius: 0.6rem;
        height: 45px;
        font-size: 0.95rem;
        font-weight: 500;
        color: var(--text-secondary);
        border-color: var(--card-border);
        transition: all 0.2s ease-in-out;
    }
    .controls-bar .btn-outline-secondary:hover {
        color: var(--primary-color);
        border-color: var(--primary-color);
        box-shadow: var(--shadow-sm);
    }
    
    /* SECCIONES DE PLATAFORMAS (CONTENEDORES) */
    .platform-container {
        background-color: var(--card-bg);
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow-md);
        padding: 2rem;
        margin-bottom: 2.5rem;
        position: relative;
        overflow: hidden;
    }
    .platform-container::before {
        content: '';
        position: absolute;
        top: 0; left: 0; right: 0; bottom: 0;
        background: linear-gradient(135deg, rgba(var(--primary-color-rgb),0.02), transparent 70%);
        border-radius: var(--border-radius-lg);
        z-index: -1;
    }
    .platform-header {
        border-bottom: 2px solid var(--card-border);
        padding-bottom: 1rem;
        margin-bottom: 1.5rem;
        display: flex;
        align-items: center;
        gap: 1rem;
        justify-content: space-between;
        flex-wrap: wrap;
        background-color: transparent;
        color: var(--text-dark);
    }

    /* Colores para encabezados de plataforma (ejemplos) */
    .platform-header.header-netflix i, .platform-header.header-netflix h2 { color: #E50914; }
    .platform-header.header-disney i, .platform-header.header-disney h2 { color: #011E59; }
    .platform-header.header-prime i, .platform-header.header-prime h2 { color: #00A8E1; }
    .platform-header.header-hbo i, .platform-header.header-hbo h2 { color: #6C2B80; }
    .platform-header.header-spotify i, .platform-header.header-spotify h2 { color: #1DB954; }
    .platform-header.header-crunchyroll i, .platform-header.header-crunchyroll h2 { color: #F47521; }
    .platform-header.header-canva i, .platform-header.header-canva h2 { color: #00C4CC; }
    .platform-header.header-paramount i, .platform-header.header-paramount h2 { color: #0066FF; }
    .platform-header.header-iptv i, .platform-header.header-iptv h2 { color: #6a1b9a; }
    .platform-header.header-capcut i, .platform-header.header-capcut h2 { color: #000000; }
    .platform-header.header-directv i, .platform-header.header-directv h2 { color: #006097; }
    .platform-header.header-movistar i, .platform-header.header-movistar h2 { color: #79c000; }
    .platform-header.header-flujo i, .platform-header.header-flujo h2 { color: #007bff; }
    .platform-header.header-viki i, .platform-header.header-viki h2 { color: #FF0077; }
    .platform-header.header-youtube i, .platform-header.header-youtube h2 { color: #FF0000; }
    .platform-header.header-eset i, .platform-header.header-eset h2 { color: #2B76C1; }
    .platform-header.header-blackbox i, .platform-header.header-blackbox h2 { color: #333333; }
    .platform-header.header-vix i, .platform-header.header-vix h2 { color: #9A1B9A; }
    .platform-header.header-gemini i, .platform-header.header-gemini h2 { color: #795548; }
    .platform-header.header-chatgpt i, .platform-header.header-chatgpt h2 { color: #10a37f; }


    .platform-header .fas, .platform-header .fab {
        font-size: 1.6rem;
        margin-right: 0.8rem;
    }
    .platform-header .platform-name {
        font-size: 1.6rem;
        font-weight: 700;
        flex-grow: 1;
        min-width: 0;
    }
    .platform-header .badge {
        background-color: var(--primary-color) !important;
        color: white !important;
        padding: 0.5em 0.9em;
        border-radius: 50rem;
        font-size: 0.9rem;
        font-weight: 700;
        flex-shrink: 0;
    }
    
    /* TARJETAS DE CUENTA INDIVIDUAL */
    .account-card {
        border: 1px solid var(--card-border);
        border-radius: var(--border-radius-md);
        transition: all 0.2s ease;
        overflow: hidden;
        box-shadow: var(--shadow-md);
        background-color: var(--card-bg);
    }
    .account-card:hover {
        box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        transform: translateY(-3px);
    }
    /* Estilos de borde de tarjeta por estado */
    .account-card {
        border-left-width: 5px;
    }
    .account-card-active { border-color: rgba(var(--success-color-rgb), 0.5); }
    .account-card-expiring { border-color: var(--warning-color); }
    .account-card-expired { border-color: var(--danger-color); }
    .account-card-problem { border-color: var(--danger-color); }
    .account-card-inactive { border-color: var(--text-secondary); }


    /* NUEVO DISEÑO DE TARJETA DE CUENTA */
    .account-card .card-body {
        padding: 1.25rem;
        display: flex;
        flex-direction: column;
        gap: 1rem;
    }
    .account-details-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 0.85rem 1.5rem;
    }
    .detail-pair {
        font-size: 0.9rem;
    }
    .detail-pair .label {
        font-weight: 500;
        color: var(--text-secondary);
        display: block;
        font-size: 0.75rem;
        margin-bottom: 0.1rem;
        text-transform: uppercase;
    }
    .detail-pair .value {
        font-weight: 600;
        color: var(--text-dark);
    }
    .detail-pair .value.password-field {
        display: flex;
        align-items: center;
        justify-content: space-between;
    }
    .detail-pair .value input {
        background: transparent; border: none; padding: 0;
        font-size: inherit; color: inherit; width: 100px;
        font-family: 'Roboto Mono', monospace;
    }
    .detail-pair .value .fas { cursor: pointer; color: var(--text-secondary); transition: color 0.2s; }
    .detail-pair .value .fas:hover { color: var(--primary-color); }

    .profiles-info { display: flex; align-items: center; justify-content: space-between; background-color: var(--body-bg); padding: 0.75rem 1rem; border-radius: var(--border-radius-sm); border: 1px solid var(--card-border); }
    .profiles-info .label { font-weight: 600; color: var(--text-dark); }
    .profiles-info .value { font-size: 1.2rem; font-weight: 800; color: var(--primary-color); }

    .account-card .card-header {
        background-color: var(--body-bg);
        border-bottom: 1px solid var(--card-border);
        font-weight: 600;
        padding: 0.8rem 1.25rem;
        font-size: 1rem;
        color: var(--text-dark);
        display: flex;
        justify-content: space-between;
        align-items: center;
        flex-wrap: wrap;
        gap: 0.5rem;
    }
    .account-email {
        font-weight: 700;
        word-break: break-all;
        font-size: 1rem;
        color: var(--primary-color);
        flex-grow: 1;
        min-width: 0;
    }
    .info-item {
        display: flex;
        align-items: center;
        padding: 0.75rem 1.25rem;
        border-bottom: 1px dashed var(--card-border);
        flex-wrap: wrap;
        gap: 0.5rem;
    }
    .info-item:last-of-type { border-bottom: none; }
    .info-item .info-icon {
        width: 30px;
        text-align: center;
        color: var(--text-secondary);
        margin-right: 0.6rem;
        font-size: 1rem;
        flex-shrink: 0;
    }
    .info-item .info-label {
        color: var(--text-dark);
        font-size: 0.9rem;
        font-weight: 500;
        flex-grow: 1;
    }
    .info-item .info-value {
        font-weight: 600;
        margin-left: auto;
        color: var(--text-dark);
        font-size: 0.95rem;
        display: flex;
        align-items: center;
        flex-shrink: 0;
    }
    .info-item .info-value input[type="password"],
    .info-item .info-value input[type="text"] {
        background: none;
        border: none;
        padding: 0;
        margin: 0;
        font-size: inherit;
        color: inherit;
        width: 100px;
        max-width: 100%;
        text-align: right;
        cursor: default;
        font-family: 'Roboto Mono', monospace;
    }
    .info-item .info-value .fas {
        font-size: 0.9rem;
        margin-left: 0.5rem;
        color: var(--text-secondary);
        cursor: pointer;
    }
    .info-item .info-value .fas:hover {
        color: var(--primary-color);
    }

    .status-badge {
        font-size: 0.75rem;
        font-weight: 600;
        padding: 0.3em 0.7em;
        border-radius: 20px;
        text-transform: uppercase;
        letter-spacing: 0.7px;
        flex-shrink: 0;
    }
    .badge-active { background-color: var(--success-color); color: white; }
    .badge-expiring { background-color: var(--warning-color); color: var(--text-dark); }
    .badge-expired { background-color: var(--danger-color); color: white; }
    .badge-problem { background-color: var(--danger-color); color: white; }
    .badge-inactive { background-color: var(--text-secondary); color: white; }

    /* Barra de Progreso */
    .progress-section {
        padding: 0.75rem 1.25rem;
        padding-top: 0;
    }
    .progress-section .progress-info {
        display: flex;
        justify-content: space-between;
        align-items: baseline;
        margin-bottom: 0.4rem;
        flex-wrap: wrap;
    }
    .progress-section .progress-label {
        font-size: 0.9rem;
        color: var(--text-dark);
        font-weight: 500;
    }
    .progress-section .progress-value-text {
        font-size: 0.85rem;
        font-weight: 600;
        text-align: right;
    }
    .progress { height: 6px; border-radius: 6px; background-color: var(--card-border); }
    .progress-bar { transition: width 0.6s ease; }
    .progress-bar-animated { animation: none; }

    /* Footer de la tarjeta con botones */
    .card-footer.bg-light {
        background-color: var(--body-bg) !important;
        display: flex;
        justify-content: center;
        padding: 1rem 1.25rem;
        border-top: 1px solid var(--card-border);
        flex-wrap: wrap;
        gap: 0.4rem;
    }
    .card-footer .btn {
        padding: 0.6rem 1rem;
        font-size: 0.85rem;
        min-width: 40px;
        flex-grow: 1;
        max-width: 140px;
        border-radius: 0.6rem;
        transition: all 0.2s ease;
        box-shadow: var(--shadow-sm);
        font-weight: 500;
        color: var(--text-secondary);
        border-color: var(--card-border);
    }
    .card-footer .btn:hover {
        transform: translateY(-2px);
        box-shadow: var(--shadow-md);
    }
    .card-footer .btn .fas {
        margin-right: 0.3rem;
    }
    /* Colores específicos para los botones */
    .card-footer .btn-outline-secondary { color: var(--text-secondary); border-color: var(--text-secondary); }
    .card-footer .btn-outline-secondary:hover { background-color: var(--text-secondary); color: white; }

    .card-footer .btn-outline-success { color: var(--success-color); border-color: var(--success-color); }
    .card-footer .btn-outline-success:hover { background-color: var(--success-color); color: white; }

    .card-footer .btn-outline-danger { color: var(--danger-color); border-color: var(--danger-color); }
    .card-footer .btn-outline-danger:hover { background-color: var(--danger-color); color: white; }

    /* Modales - Aseguran un buen aspecto en diferentes tamaños */
    .modal-content { border-radius: var(--border-radius-lg); box-shadow: var(--shadow-lg); }
    .modal-header { border-bottom: none; padding: 1.25rem 1.75rem; }
    .modal-title { font-size: 1.4rem; font-weight: 700; }
    .modal-body { padding: 1.25rem 1.75rem; }
    .modal-footer { border-top: none; padding: 0.75rem 1.75rem; }
    .modal .form-label { font-weight: 600; color: var(--text-dark); font-size: 0.9rem; }
    .modal .form-control, .modal .form-select { border-radius: var(--border-radius-sm); border-color: var(--card-border); font-size: 0.9rem; }
    .modal .input-group .form-control { border-top-right-radius: 0; border-bottom-right-radius: 0; }
    .modal .input-group .btn { border-top-left-radius: 0; border-bottom-left-radius: 0; }
    .btn-close-white { filter: invert(1); }

    /* Notas de Suscripcion colapsables */
    .notes-toggle {
        cursor: pointer;
        color: var(--text-muted);
        font-size: 0.85rem;
        transition: all 0.2s ease;
        margin-top: 0.5rem;
        display: block;
        text-align: right;
    }
    .notes-toggle:hover {
        color: var(--primary-color);
        text-decoration: underline;
    }
    .notes-toggle .fas {
        font-size: 0.7em;
        transition: transform 0.2s ease;
    }
    .notes-toggle[aria-expanded="true"] .fas {
        transform: rotate(180deg);
    }
    .notes-collapse-content {
        margin-top: 0.5rem;
        padding: 0.75rem;
        background-color: var(--body-bg);
        border-radius: var(--border-radius-sm);
        font-size: 0.85rem;
        color: var(--text-dark);
        border: 1px solid var(--card-border);
        box-shadow: inset 0 1px 3px rgba(0,0,0,0.05);
    }

    /* Estilos para las notificaciones flash (nuevos) */
    .alert-custom {
        padding: 1rem 1.5rem;
        border-radius: var(--border-radius-md);
        box-shadow: var(--shadow-md);
        border: none;
        display: flex;
        align-items: center;
        margin-bottom: 1.5rem;
        position: relative; /* Para el icono */
    }
    .alert-custom .alert-icon {
        font-size: 1.5rem;
        margin-right: 1rem;
    }
    .alert-custom.alert-success {
        background-color: #e6f7ea;
        color: #1f7d37;
    }
    .alert-custom.alert-success .alert-icon { color: var(--success-color); }
    
    .alert-custom.alert-danger {
        background-color: #fcebeb;
        color: #9f3a41;
    }
    .alert-custom.alert-danger .alert-icon { color: var(--danger-color); }

    .alert-custom.alert-warning {
        background-color: #fff3e0;
        color: #a07a09;
    }
    .alert-custom.alert-warning .alert-icon { color: var(--warning-color); }

    .alert-custom.alert-info {
        background-color: #e8f7fa;
        color: #1a6d7c;
    }
    .alert-custom.alert-info .alert-icon { color: var(--info-color); }
    .alert-custom .btn-close { font-size: 0.9rem; }

    /* Media queries para responsividad */
    @media (max-width: 991.98px) {
        .page-header {
            flex-direction: column;
            align-items: flex-start;
            text-align: left;
            padding-bottom: 1rem;
            margin-bottom: 2rem;
        }
        .page-header h1 { font-size: 2.2rem; margin-bottom: 0.25rem; }
        .page-header p { font-size: 0.9rem; }
        .page-header .btn-primary { 
            width: 100%;
            margin-left: 0;    
            margin-top: 1rem;    
            padding: 0.6rem 1.2rem;
            font-size: 0.9rem;
        }

        .stat-card { padding: 1.2rem; }
        .stat-card .stat-icon { width: 50px; height: 50px; font-size: 1.4rem; margin-right: 1rem; }
        .stat-card .stat-title { font-size: 0.85rem; }
        .stat-card .stat-value { font-size: 1.8rem; }

        .alerts-section { padding: 1.5rem; margin-bottom: 2rem; }
        .alerts-section::before { width: 6px; }
        .alerts-header { font-size: 1.4rem; margin-bottom: 1rem; padding-left: 0.8rem; }
        .alerts-header i { font-size: 1.5rem; margin-right: 0.6rem; }
        .alerts-header .badge { font-size: 0.8rem; padding: 0.4em 0.8em; }
        .list-group-item { padding: 0.8rem 1rem; margin-bottom: 0.6rem; flex-direction: column; align-items: flex-start; }
        .alert-platform-name { font-size: 0.95rem; margin-bottom: 0.1rem; }
        .alert-platform-name .fas, .alert-platform-name .fab { font-size: 1rem; }
        .alert-account-email { font-size: 0.8rem; margin-bottom: 0.2rem; }
        .alert-detail-text { margin-left: 0; text-align: left; font-size: 0.9rem; width: 100%; }
        .list-group-item > div:last-child {
            width: 100%;
            display: flex;
            justify-content: flex-start;
            margin-top: 0.5rem;
            gap: 0.4rem;
        }
        .list-group-item button.btn {
            padding: 0.4rem 0.8rem;
            font-size: 0.8rem;
        }


        .controls-bar { padding: 1rem 1.25rem; margin-bottom: 2rem; }
        .input-icon-group .form-control, .input-icon-group .form-select { height: 40px; font-size: 0.85rem; padding-left: 2.2rem; }
        .input-icon-group .input-icon { font-size: 0.85rem; left: 0.8rem; }
        .controls-bar .btn-outline-secondary { height: 40px; font-size: 0.85rem; }

        .platform-container { padding: 1.5rem; margin-bottom: 2rem; }
        .platform-header { padding-bottom: 0.8rem; margin-bottom: 1.2rem; }
        .platform-header .platform-name { font-size: 1.4rem; }
        .platform-header .fas, .platform-header .fab { font-size: 1.4rem; margin-right: 0.5rem;}
        .platform-header .badge { font-size: 0.8rem; padding: 0.4em 0.8em; margin-left: 0.5rem;}

        .account-card .card-header { padding: 0.75rem 1rem; font-size: 0.95rem; }
        .account-email { font-size: 0.95rem; }
        .info-item { padding: 0.6rem 1rem; font-size: 0.85rem; }
        .info-item .info-icon { width: 25px; margin-right: 0.5rem; font-size: 0.9rem; }
        .info-item .info-label { font-size: 0.85rem; }
        .info-item .info-value { font-size: 0.9rem; }
        .info-item .info-value input[type="password"], .info-item .info-value input[type="text"] { width: 70px; font-size: 0.85rem; }
        .info-item .info-value .fas { font-size: 0.8rem; margin-left: 0.4rem; }
        .status-badge { font-size: 0.7rem; padding: 0.25em 0.6em; }

        .progress-section { padding: 0.6rem 1rem; }
        .progress-section .progress-label, .progress-section .progress-value-text { font-size: 0.8rem; }
        .progress { height: 5px; }

        .card-footer.bg-light { padding: 1rem; gap: 0.3rem; }
        .card-footer .btn { padding: 0.5rem 0.8rem; font-size: 0.8rem; max-width: none; }
        .card-footer .btn .fas { margin-right: 0.3rem; }
    }

    @media (max-width: 767.98px) {
        .page-container { padding-left: 0.8rem; padding-right: 0.8rem; }
        .page-header .btn-primary { width: 100%; margin-left: 0; }
        .alerts-header { flex-direction: column; align-items: flex-start; padding-left: 0.5rem; }
        .alerts-header .badge { margin-left: 0; margin-top: 0.5rem; }
        .list-group-item > div:first-child { width: 100%; margin-bottom: 0.5rem; }
        .list-group-item > div:last-child { justify-content: flex-start; }
    }
    
</style>

<div class="container-fluid px-lg-5 px-md-4 py-4 page-container">
    <header class="page-header">
        <div>
            <h1><i class="fas fa-server me-3"></i>Dashboard de Cuentas Maestras</h1>
            <p class="text-muted mb-0">Visión general y gestión de tus cuentas maestras.</p>
        </div>
        <div class="d-flex gap-2">
            <?php if (!empty($alert_accounts)): ?>
            <button type="button" class="btn btn-warning" data-bs-toggle="modal" data-bs-target="#modalAvisos">
                <i class="fas fa-exclamation-triangle me-2"></i>Avisos 
                <span class="badge bg-white text-warning ms-1"><?php echo count($alert_accounts); ?></span>
            </button>
            <?php endif; ?>
                            <?php if ($rol_usuario_sesion !== 'cortes'): ?>
                <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalAgregarCuenta">
                    <i class="fas fa-plus-circle"></i> Agregar Cuenta
                </button>
                <?php endif; ?>
        </div>
    </header>

    <?php if (isset($_GET['mensaje'])): ?>
    <?php
        $alert_type = htmlspecialchars($_GET['tipo'] ?? 'info');
        $icon_class = 'fa-info-circle';
        if ($alert_type === 'success') $icon_class = 'fa-check-circle';
        if ($alert_type === 'danger') $icon_class = 'fa-exclamation-circle';
        if ($alert_type === 'warning') $icon_class = 'fa-exclamation-triangle';
    ?>
    <div class="alert alert-<?php echo $alert_type; ?> alert-dismissible fade show alert-custom"><i class="alert-icon fas <?php echo $icon_class; ?>"></i>
        <div><?php echo htmlspecialchars(urldecode($_GET['mensaje'])); ?></div>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <?php if ($debug_mode): ?>
    <div class="alert alert-info mb-4">
        <h5 class="alert-heading"><i class="fas fa-bug"></i> Modo de Depuración Activado</h5>
        <p>Revisa que el "Estado de Cuenta" y los "Días Restantes" sean correctos para que las alertas funcionen como esperas.</p><hr>
        <p><strong>Días configurados para alerta:</strong> <?php echo $dias_alerta_maestras_global; ?> días.</p>
        <div class="table-responsive">
            <table class="table table-sm table-bordered">
                <thead>
                    <tr><th>Email</th><th>Estado</th><th>Fecha Vence</th><th>Días Rest.</th><th>Es Alerta?</th></tr>
                </thead>
                <tbody>
                    <?php foreach($cuentas_maestras_raw as $c): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($c['email_cuenta']); ?></td>
                            <td><?php echo htmlspecialchars($c['estado_cuenta']); echo $c['estado_cuenta'] == 1 ? " (Activa)" : ($c['estado_cuenta'] == 0 ? " (Inactiva)" : " (Problema)"); ?></td>
                            <td><?php echo htmlspecialchars($c['fecha_vencimiento_proveedor']); ?></td>
                            <td><?php    
                                $debug_status = getAccountStatusDetails($c, $dias_alerta_maestras_global);
                                echo in_array($debug_status['status_key'], ['expired', 'expiring', 'problem']) ? '<span class="text-success fw-bold">SÍ</span>' : 'No';    
                            ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php endif; ?>

    <div class="row g-4 mb-4">
        <div class="col-lg-4 col-md-6">
            <div class="stat-card">
                <div class="stat-icon bg-soft-primary"><i class="fas fa-list-ol"></i></div>
                <div>
                    <div class="stat-title">Total de Cuentas</div>
                    <div class="stat-value"><?php echo $total_cuentas; ?></div>
                </div>
            </div>
        </div>
        <div class="col-lg-4 col-md-6">
            <div class="stat-card">
                <div class="stat-icon bg-soft-success"><i class="fas fa-check-circle"></i></div>
                <div>
                    <div class="stat-title">Cuentas Activas</div>
                    <div class="stat-value"><?php echo $cuentas_activas_count; ?> / <?php echo $total_cuentas; ?></div>
                </div>
            </div>
        </div>
        <div class="col-lg-4 col-md-12">
            <div class="stat-card">
                <div class="stat-icon bg-soft-warning"><i class="fas fa-users"></i></div>
                <div>
                    <div class="stat-title">Perfiles Disponibles</div>
                    <div class="stat-value"><?php echo $total_perfiles_disponibles; ?></div>
                </div>
            </div>
        </div>
    </div>



    <div class="controls-bar mb-4">
        <div class="row g-3 align-items-center">
            <div class="col-lg-3 col-md-12">
                <label for="quickSearchInput" class="form-label">Búsqueda Rápida</label>
                <div class="input-icon-group">
                    <i class="fas fa-search input-icon"></i>
                                         <input type="text" id="quickSearchInput" class="form-control" placeholder="Buscar por email, método..." value="<?php echo htmlspecialchars($filter_search_text); ?>">
                </div>
            </div>
            <div class="col-lg-2 col-md-6">
                <label for="platformFilter" class="form-label">Plataforma</label>
                <div class="input-icon-group">
                    <i class="fas fa-tv input-icon"></i>
                    <select id="platformFilter" class="form-select">
                        <option value="all">Todas las Plataformas</option>
                        <?php foreach ($plataformas_activas as $p): ?>
                            <option value="<?php echo htmlspecialchars($p['id_plataforma']); ?>" <?php echo ($filter_platform_id == $p['id_plataforma'] ? 'selected' : ''); ?>>
                                <?php echo htmlspecialchars($p['nombre_plataforma']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            <div class="col-lg-2 col-md-6">
                <label for="statusFilter" class="form-label">Estado</label>
                <div class="input-icon-group">
                    <i class="fas fa-check-circle input-icon"></i>
                    <select id="statusFilter" class="form-select">
                        <option value="all" <?php echo ($filter_status == 'all' ? 'selected' : ''); ?>>Todos los Estados</option>
                        <option value="active" <?php echo ($filter_status == 'active' ? 'selected' : ''); ?>>Activa</option>
                        <option value="expiring" <?php echo ($filter_status == 'expiring' ? 'selected' : ''); ?>>Por Vencer</option>
                        <option value="expired" <?php echo ($filter_status == 'expired' ? 'selected' : ''); ?>>Vencida</option>
                        <option value="problem" <?php echo ($filter_status == 'problem' ? 'selected' : ''); ?>>Con Problemas</option>
                        <option value="inactive" <?php echo ($filter_status == 'inactive' ? 'selected' : ''); ?>>Inactiva</option>
                    </select>
                </div>
            </div>
            <div class="col-lg-2 col-md-6">
              <label for="stockFilter" class="form-label">Stock</label>
              <div class="input-icon-group">
                <i class="fas fa-boxes input-icon"></i>
                <select id="stockFilter" class="form-select">
                  <option value="all" selected>Todo stock</option>
                  <option value="with">Con stock (&gt; 0)</option>
                  <option value="without">Sin stock (= 0)</option>
                </select>
              </div>
            </div>
            <div class="col-lg-2 col-md-6">
              <label for="minStockFilter" class="form-label">Mín. Perfiles</label>
              <div class="input-icon-group">
                <i class="fas fa-sort-numeric-up input-icon"></i>
                <input type="number" id="minStockFilter" class="form-control" min="0" placeholder="Mín. perfiles disp.">
              </div>
            </div>
            <div class="col-lg-1 col-md-12">
                <button id="resetFiltersBtn" class="btn btn-outline-secondary w-100"><i class="fas fa-undo me-1"></i></button>
            </div>
        </div>
    </div>

    <div id="accounts-container">
        <?php if ($total_cuentas > 0): ?>
            <?php foreach ($grouped_accounts as $platform_name => $accounts): ?>
                <section class="platform-container platform-group" data-platform-id="<?php echo htmlspecialchars($accounts[0]['id_plataforma']); ?>">
                    
                    <?php
                        // dentro del foreach ($grouped_accounts as $platform_name => $accounts)
                        // calcula stock de esa plataforma:
                        $stock_disp = 0;
                        $stock_total = 0;
                        foreach ($accounts as $acc) {
                            if($acc['estado_cuenta'] == 1) { // Solo si la cuenta está Activa
                                $stock_disp  += (int)$acc['perfiles_disponibles'];
                                $stock_total += (int)$acc['perfiles_total'];
                            }
                        }
                    ?>
                    <header class="platform-header">
                        <i class="<?php echo getPlatformIcon($platform_name); ?>"></i>
                        <h2 class="platform-name mb-0"><?php echo htmlspecialchars($platform_name); ?></h2>
                        <div class="d-flex align-items-center gap-2 flex-wrap">
                            <span class="badge rounded-pill"><?php echo count($accounts); ?> Cuentas</span>
                            <span class="badge rounded-pill bg-info">Stock: <?php echo $stock_disp; ?>/<?php echo $stock_total; ?></span>
                        </div>
                    </header>
                    <div class="row g-4">
                        <?php foreach ($accounts as $cuenta): ?>
                            <?php    
                                $status_details = getAccountStatusDetails($cuenta, $dias_alerta_maestras_global);    
                                $searchable_text = strtolower($cuenta['email_cuenta'] . ' ' . $cuenta['nombre_plataforma'] . ' ' . ($cuenta['nombre_proveedor'] ?? ''));    
                            ?>
                            
                            <div class="col-xl-4 col-lg-6 account-wrapper"
                                 data-id="<?php echo (int)$cuenta['id_cuenta_maestra']; ?>"
                                 data-platform-id="<?php echo (int)$cuenta['id_plataforma']; ?>"
                                 data-perfiles-disponibles="<?php echo (int)$cuenta['perfiles_disponibles']; ?>"
                                 data-perfiles-total="<?php echo (int)$cuenta['perfiles_total']; ?>"
                                 data-search-text="<?php echo htmlspecialchars($searchable_text); ?>"
                                 data-status-key="<?php echo $status_details['status_key']; ?>">
                            <div class="card h-100 account-card <?php echo $status_details['card_class']; ?>">
                                     <div class="card-header">
                                         <span id="email-text-<?php echo $cuenta['id_cuenta_maestra']; ?>" class="account-email"><?php echo htmlspecialchars($cuenta['email_cuenta']); ?></span>
                                         <i class="fas fa-copy text-secondary" style="cursor: pointer;" title="Copiar Email" data-bs-toggle="tooltip" onclick="copyToClipboard('#email-text-<?php echo $cuenta['id_cuenta_maestra']; ?>', this)"></i>
                                     </div>
                                     <div class="card-body d-flex flex-column">
                                         <div class="profiles-info">
                                             <span class="label">Perfiles Disponibles</span>
                                             <span class="value"><?php echo htmlspecialchars($cuenta['perfiles_disponibles']); ?> / <?php echo htmlspecialchars($cuenta['perfiles_total']); ?></span>
                                         </div>
                                         <div class="account-details-grid mt-3">
                                             <div class="detail-pair">
                                                 <span class="label">Estado</span>
                                                 <span class="value"><?php echo $status_details['status_badge']; ?></span>
                                             </div>
                                             <div class="detail-pair">
                                                 <span class="label">Contraseña</span>
                                                 <div class="value password-field">
                                                     <input type="password" readonly id="password-<?php echo $cuenta['id_cuenta_maestra']; ?>" value="<?php echo htmlspecialchars($cuenta['password_cuenta']); ?>">
                                                     <span>
                                                         <i class="fas fa-eye toggle-password-btn" title="Mostrar/Ocultar" data-bs-toggle="tooltip" onclick="togglePasswordVisibility('password-<?php echo $cuenta['id_cuenta_maestra']; ?>', this)"></i>
                                                         <i class="fas fa-copy copy-btn ms-2" title="Copiar" data-bs-toggle="tooltip" onclick="copyToClipboard('#password-<?php echo $cuenta['id_cuenta_maestra']; ?>', this)"></i>
                                                     </span>
                                                 </div>
                                             </div>
                                                                                           <div class="detail-pair"><span class="label">Método</span><span class="value"><?php echo htmlspecialchars($cuenta['nombre_proveedor'] ?? 'N/A'); ?></span></div>
                                             <div class="detail-pair"><span class="label">Costo</span><span class="value"><?php echo htmlspecialchars($moneda_simbolo . number_format($cuenta['costo_compra'], 2)); ?></span></div>
                                             <div class="detail-pair"><span class="label">Fecha Compra</span><span class="value"><?php echo htmlspecialchars(date("d/m/Y", strtotime($cuenta['fecha_compra_proveedor']))); ?></span></div>
                                         </div>
                                         
                                         <?php if (!empty($cuenta['notas_cuenta'])): ?>
                                         <div class="px-3">
                                             <a class="notes-toggle text-decoration-none" data-bs-toggle="collapse" href="#notes_<?php echo $cuenta['id_cuenta_maestra']; ?>" role="button" aria-expanded="false" aria-controls="notes_<?php echo $cuenta['id_cuenta_maestra']; ?>">
                                                 <i class="fas fa-sticky-note me-1"></i> Notas <i class="fas fa-chevron-down ms-1 fa-xs"></i>
                                             </a>
                                             <div class="collapse notes-collapse-content" id="notes_<?php echo $cuenta['id_cuenta_maestra']; ?>">
                                                 <?php echo nl2br(htmlspecialchars($cuenta['notas_cuenta'])); ?>
                                             </div>
                                         </div>
                                         <?php endif; ?>

                                         <div class="progress-section mt-auto">
                                             <div class="progress-info">
                                                 <span class="progress-label">Vencimiento:</span>
                                                 <span class="progress-value-text <?php echo $status_details['vencimiento_class']; ?>"><?php echo htmlspecialchars(date("d/m/Y", strtotime($cuenta['fecha_vencimiento_proveedor']))); ?> (<?php echo $status_details['dias_restantes_text']; ?>)</span>
                                             </div>
                                             <div class="progress" title="Días restantes: <?php echo intval($cuenta['dias_restantes_sql']); ?>" data-bs-toggle="tooltip">
                                                 <div class="progress-bar <?php echo $status_details['progress_bar_class']; ?>" role="progressbar" style="width: <?php echo $status_details['progress_bar_value']; ?>%" aria-valuenow="<?php echo $status_details['progress_bar_value']; ?>" aria-valuemin="0" aria-valuemax="100"></div>
                                             </div>
                                             <?php if (isset($cuenta['contador_renovaciones']) && $cuenta['contador_renovaciones'] > 0): ?>
                                             <div class="text-center mt-2">
                                                 <small class="text-muted">
                                                     <i class="fas fa-sync-alt me-1"></i>
                                                     Renovada <?php echo $cuenta['contador_renovaciones']; ?> vez<?php echo $cuenta['contador_renovaciones'] != 1 ? 'es' : ''; ?>
                                                     <?php if (isset($cuenta['fecha_ultima_renovacion'])): ?>
                                                     (última: <?php echo date("d/m/Y", strtotime($cuenta['fecha_ultima_renovacion'])); ?>)
                                                     <?php endif; ?>
                                                 </small>
                                             </div>
                                             <?php endif; ?>
                                         </div>

                                     </div>
                                     <div class="card-footer bg-light">
                                         <div class="d-flex justify-content-center gap-2 flex-wrap">
                                             <!-- Botón Editar - Disponible para todos los roles -->
                                             <button class="btn btn-sm btn-outline-secondary" data-bs-toggle="modal" data-bs-target="#modalEditarCuenta" title="Editar Cuenta" data-id="<?php echo htmlspecialchars($cuenta['id_cuenta_maestra']); ?>" data-id_plataforma="<?php echo htmlspecialchars($cuenta['id_plataforma']); ?>" data-email_cuenta="<?php echo htmlspecialchars($cuenta['email_cuenta']); ?>" data-password_cuenta="<?php echo htmlspecialchars($cuenta['password_cuenta']); ?>" data-nombre_proveedor="<?php echo htmlspecialchars($cuenta['nombre_proveedor'] ?? ''); ?>" data-fecha_compra_proveedor="<?php echo htmlspecialchars($cuenta['fecha_compra_proveedor'] ?? ''); ?>" data-fecha_vencimiento_proveedor="<?php echo htmlspecialchars($cuenta['fecha_vencimiento_proveedor']); ?>" data-costo_compra="<?php echo htmlspecialchars($cuenta['costo_compra']); ?>" data-perfiles_total="<?php echo htmlspecialchars($cuenta['perfiles_total']); ?>" data-notas_cuenta="<?php echo htmlspecialchars($cuenta['notas_cuenta'] ?? ''); ?>" data-estado_cuenta="<?php echo htmlspecialchars($cuenta['estado_cuenta']); ?>" data-perfiles-vendidos="<?php echo htmlspecialchars($cuenta['perfiles_vendidos_reales']); ?>"><i class="fas fa-pencil-alt"></i> Editar</button>
                                             
                                             <!-- Botón Renovar - NO disponible para rol 'cortes' -->
                                             <?php if ($rol_usuario_sesion !== 'cortes'): ?>
                                             <button class="btn btn-sm btn-outline-success" data-bs-toggle="modal" data-bs-target="#modalRenovarCuenta" 
                                                     data-id="<?php echo $cuenta['id_cuenta_maestra']; ?>" 
                                                     data-info="<?php echo htmlspecialchars($cuenta['email_cuenta']); ?>" 
                                                     data-costo="<?php echo htmlspecialchars($cuenta['costo_compra']); ?>"
                                                     data-plataforma="<?php echo htmlspecialchars($cuenta['nombre_plataforma']); ?>"
                                                     data-vencimiento="<?php echo htmlspecialchars(date("d/m/Y", strtotime($cuenta['fecha_vencimiento_proveedor']))); ?>"
                                                     data-fecha-vencimiento="<?php echo htmlspecialchars($cuenta['fecha_vencimiento_proveedor']); ?>"
                                                     data-estado="<?php echo htmlspecialchars($cuenta['estado_cuenta']); ?>"
                                                     title="Renovar Cuenta">
                                                 <i class="fas fa-sync-alt"></i> Renovar
                                             </button>
                                             <?php endif; ?>
                                             
                                             <!-- Botón Eliminar - NO disponible para rol 'cortes' -->
                                             <?php if ($rol_usuario_sesion !== 'cortes'): ?>
                                             <button class="btn btn-sm btn-outline-danger" data-bs-toggle="modal" data-bs-target="#modalEliminarCuenta" data-id="<?php echo $cuenta['id_cuenta_maestra']; ?>" data-email="<?php echo htmlspecialchars($cuenta['email_cuenta']); ?>" title="Eliminar Cuenta Permanentemente"><i class="fas fa-trash-alt"></i> Eliminar</button>
                                             <?php endif; ?>
                                         </div>
                                     </div>
                                 </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </section>
            <?php endforeach; ?>
        <?php else: ?>
            <div class="text-center py-5"><i class="fas fa-folder-open fa-4x text-muted mb-3"></i><h3 class="text-muted">No hay cuentas registradas</h3><p class="text-muted"><?php echo ($rol_usuario_sesion === 'cortes') ? 'No hay cuentas disponibles para gestionar.' : 'Usa el botón "Agregar Cuenta" para empezar a gestionar.'; ?></p></div>
        <?php endif; ?>
        <div id="no-results-message" class="text-center py-5" style="display: none;"><i class="fas fa-search-minus fa-4x text-muted mb-3"></i><h3 class="text-muted">No se encontraron resultados</h3><p class="text-muted">Intenta ajustar los filtros de búsqueda.</p></div>
    </div>
</div>

<div class="modal fade" id="modalAgregarCuenta" tabindex="-1" aria-labelledby="modalAgregarCuentaLabel" aria-hidden="true" data-bs-backdrop="static">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title" id="modalAgregarCuentaLabel"><i class="fas fa-plus-circle"></i> Agregar Nueva Cuenta Maestra</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="/digitalpremium/actions/cuentas_maestras_controlador.php" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="accion" value="agregar">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Plataforma <span class="text-danger">*</span></label>
                            <select class="form-select" id="id_plataforma_add" name="id_plataforma" required>
                                <option value="">Seleccione...</option>
                                <?php foreach ($plataformas_activas as $p): ?>
                                    <option value="<?php echo htmlspecialchars($p['id_plataforma']); ?>" data-max_perfiles="<?php echo htmlspecialchars($p['max_perfiles_cuenta']); ?>">
                                        <?php echo htmlspecialchars($p['nombre_plataforma']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Email/Usuario <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="email_cuenta" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Contraseña <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <input type="password" class="form-control" id="password_cuenta_add" name="password_cuenta" required>
                                <button class="btn btn-outline-secondary" type="button" id="togglePasswordAdd"><i class="fas fa-eye"></i></button>
                            </div>
                        </div>
                        <div class="col-md-6"> 
                                                         <label class="form-label">Método</label>
                             <input list="metodos_list_modal_add" class="form-control" name="nombre_proveedor" placeholder="Escriba o seleccione">
                             <datalist id="metodos_list_modal_add">
                                 <?php foreach ($metodos_activos as $prov): ?><option value="<?php echo htmlspecialchars($prov['nombre_proveedor']); ?>"><?php endforeach; ?>
                             </datalist>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Fecha Compra</label>
                            <input type="date" class="form-control" id="fecha_compra_proveedor_add" name="fecha_compra_proveedor" value="<?php echo date('Y-m-d'); ?>">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Días de Vigencia</label>
                            <input type="number" class="form-control" id="dias_membresia_add" min="0" placeholder="Ej: 30" value="30">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Vence el <span class="text-danger">*</span></label>
                            <input type="date" class="form-control" id="fecha_vencimiento_proveedor_add" name="fecha_vencimiento_proveedor" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Costo (<?php echo htmlspecialchars($moneda_simbolo); ?>)</label>
                            <input type="number" step="0.01" class="form-control" name="costo_compra" placeholder="0.00">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Perfiles Totales <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="perfiles_total_add" name="perfiles_total" min="1" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Estado <span class="text-danger">*</span></label>
                            <select class="form-select" name="estado_cuenta" required>
                                <option value="1" selected>Activa</option>
                                <option value="0">Inactiva</option>
                                <option value="2">Con Problemas</option>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Notas Adicionales</label>
                            <textarea class="form-control" name="notas_cuenta" rows="2"></textarea>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cerrar</button>
                    <button type="submit" class="btn btn-primary"><i class="fas fa-save me-2"></i>Guardar Cuenta</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="modalEditarCuenta" tabindex="-1" aria-labelledby="modalEditarCuentaLabel" aria-hidden="true" data-bs-backdrop="static">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-warning text-dark">
                <h5 class="modal-title" id="modalEditarCuentaLabel"><i class="fas fa-edit"></i> Editar Cuenta Maestra</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form action="/digitalpremium/actions/cuentas_maestras_controlador.php" method="POST" id="form-edit-account">
                <div class="modal-body">
                    <input type="hidden" name="accion" value="editar">
                    <input type="hidden" name="id_cuenta_maestra" id="id_cuenta_maestra_edit">
                    <input type="hidden" name="__dias_alerta" value="<?php echo $dias_alerta_maestras_global; ?>">
                    <input type="hidden" id="perfiles_vendidos" name="perfiles_vendidos">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Plataforma <span class="text-danger">*</span></label>
                            <select class="form-select" id="id_plataforma_edit" name="id_plataforma" required>
                                <option value="">Seleccione...</option>
                                <?php foreach ($plataformas_activas as $p): ?>
                                    <option value="<?php echo htmlspecialchars($p['id_plataforma']); ?>" data-max_perfiles="<?php echo htmlspecialchars($p['max_perfiles_cuenta']); ?>">
                                        <?php echo htmlspecialchars($p['nombre_plataforma']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Email/Usuario <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="email_cuenta_edit" name="email_cuenta" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Contraseña</label>
                            <div class="input-group">
                                <input type="password" class="form-control" id="password_cuenta_edit" name="password_cuenta">
                                <button class="btn btn-outline-secondary" type="button" id="togglePasswordEdit"><i class="fas fa-eye"></i></button>
                            </div>
                            <small class="form-text text-muted">Dejar en blanco para no cambiar.</small>
                        </div>
                        <div class="col-md-6">
                                                         <label class="form-label">Método</label>
                             <input list="metodos_list_modal_edit" class="form-control" id="nombre_proveedor_edit" name="nombre_proveedor" placeholder="Escriba o seleccione">
                             <datalist id="metodos_list_modal_edit">
                                 <?php foreach ($metodos_activos as $prov): ?><option value="<?php echo htmlspecialchars($prov['nombre_proveedor']); ?>"><?php endforeach; ?>
                             </datalist>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Fecha Compra</label>
                            <input type="date" class="form-control" id="fecha_compra_proveedor_edit" name="fecha_compra_proveedor">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Días de Vigencia</label>
                            <input type="number" class="form-control" id="dias_membresia_edit" min="0" placeholder="Calcular...">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Fecha Vencimiento <span class="text-danger">*</span></label>
                            <input type="date" class="form-control" id="fecha_vencimiento_proveedor_edit" name="fecha_vencimiento_proveedor" required>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Costo (<?php echo htmlspecialchars($moneda_simbolo); ?>)</label>
                            <input type="number" step="0.01" class="form-control" id="costo_compra_edit" name="costo_compra">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Perfiles Totales <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="perfiles_total_edit" name="perfiles_total" min="1" required>
                            <small class="text-danger d-none" id="perfiles-vendidos-warning">No puede ser menor que los perfiles vendidos: <span id="vendidos-count">0</span></small>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Estado <span class="text-danger">*</span></label>
                            <select class="form-select" id="estado_cuenta_edit" name="estado_cuenta" required>
                                <option value="1">Activa</option>
                                <option value="0">Inactiva</option>
                                <option value="2">Con Problemas</option>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Notas Adicionales</label>
                            <textarea class="form-control" id="notas_cuenta_edit" name="notas_cuenta" rows="2"></textarea>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cerrar</button>
                    <button type="submit" class="btn btn-warning"><i class="fas fa-save me-2"></i>Actualizar Cuenta</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="modalRenovarCuenta" tabindex="-1" aria-labelledby="modalRenovarCuentaLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content rounded-4 shadow-lg">
            <div class="modal-header bg-primary text-white border-0 rounded-top-4">
                <h5 class="modal-title" id="modalRenovarCuentaLabel">
                    <i class="fas fa-sync-alt me-2"></i>Renovar Cuenta Maestra
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="/digitalpremium/actions/cuentas_maestras_controlador.php" method="POST">
                <div class="modal-body p-4">
                    <input type="hidden" name="accion" value="renovar">
                    <input type="hidden" name="id_cuenta_maestra" id="id_cuenta_maestra_renew">
                    
                    <!-- Información de la cuenta -->
                    <div class="alert alert-info border-0 rounded-3 mb-4">
                        <div class="d-flex align-items-center">
                            <i class="fas fa-server fa-2x text-primary me-3"></i>
                            <div>
                                <h6 class="mb-1 fw-bold text-dark">Cuenta a Renovar</h6>
                                <p id="info_cuenta_renew" class="mb-0 text-dark fw-semibold"></p>
                                <small id="plataforma_cuenta_renew" class="text-muted"></small>
                                <div class="mt-2">
                                    <span class="badge bg-warning text-dark">
                                        <i class="fas fa-calendar-times me-1"></i>
                                        Vence: <span id="vencimiento_actual_renew"></span>
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Fecha de vencimiento -->
                    <div class="mb-4">
                        <label class="form-label fw-bold text-dark mb-3">
                            <i class="fas fa-calendar-alt me-2 text-primary"></i>
                            Nueva Fecha de Vencimiento <span class="text-danger">*</span>
                        </label>
                        

                        
                        <!-- Botones de días rápidos -->
                        <div class="d-flex gap-2 mb-3 flex-wrap">
                            <button type="button" class="btn btn-outline-primary rounded-pill preset-renew" data-days="30">
                                <i class="fas fa-plus me-1"></i>+30 días
                            </button>
                            <button type="button" class="btn btn-outline-primary rounded-pill preset-renew" data-days="60">
                                <i class="fas fa-plus me-1"></i>+60 días
                            </button>
                            <button type="button" class="btn btn-outline-primary rounded-pill preset-renew" data-days="90">
                                <i class="fas fa-plus me-1"></i>+90 días
                            </button>
                        </div>
                        
                        <input type="date" class="form-control form-control-lg rounded-3 border-2" 
                               id="fecha_vencimiento_renew" name="fecha_vencimiento_proveedor" required>
                        <small class="text-muted mt-2 d-block">
                            <i class="fas fa-info-circle me-1"></i>
                            Los botones rápidos suman días a la fecha de vencimiento actual. También puedes elegir una fecha manual.
                        </small>
                    </div>

                    <!-- Costo -->
                    <div class="mb-4">
                        <label for="costo_compra_renew" class="form-label fw-bold text-dark">
                            <i class="fas fa-dollar-sign me-2 text-primary"></i>
                            Costo de Renovación (<?php echo htmlspecialchars($moneda_simbolo); ?>)
                        </label>
                        <div class="input-group input-group-lg">
                            <span class="input-group-text bg-light border-2"><?php echo htmlspecialchars($moneda_simbolo); ?></span>
                            <input type="number" step="0.01" class="form-control border-2" 
                                   id="costo_compra_renew" name="costo_compra" placeholder="0.00">
                        </div>
                        <small class="form-text text-muted mt-2 d-block">
                            <i class="fas fa-info-circle me-1"></i>
                            Este costo se registrará como un egreso en tu sistema de finanzas.
                        </small>
                    </div>

                    <!-- Notas -->
                    <div class="mb-4">
                        <label for="notas_renovacion_renew" class="form-label fw-bold text-dark">
                            <i class="fas fa-sticky-note me-2 text-primary"></i>
                            Notas (opcional)
                        </label>
                        <textarea class="form-control border-2" id="notas_renovacion_renew" name="notas_renovacion" 
                                  rows="2" placeholder="Observaciones sobre la renovación"></textarea>
                    </div>
                    
                    <!-- Resumen de la renovación -->
                    <div class="alert alert-success border-0 rounded-3 mb-0">
                        <div class="d-flex align-items-center">
                            <i class="fas fa-check-circle fa-2x text-success me-3"></i>
                            <div>
                                <h6 class="mb-1 fw-bold text-success">Resumen de Renovación</h6>
                                <p class="mb-0 text-success">
                                    ✅ Esta renovación se registrará automáticamente en tu sistema.<br>
                                    💰 El costo se registrará como <strong>egreso</strong> en tu sistema de finanzas.<br>
                                    📊 Los egresos se calculan sumando todos los costos de cuentas maestras.
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer d-flex justify-content-between p-4 border-top-0">
                    <button type="button" class="btn btn-outline-secondary rounded-pill px-4" data-bs-dismiss="modal">
                        <i class="fas fa-times me-2"></i>Cerrar
                    </button>
                    <button type="submit" class="btn btn-primary rounded-pill shadow-sm px-4">
                        <i class="fas fa-check-circle me-2"></i>Confirmar Renovación
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="modalEliminarCuenta" tabindex="-1" aria-labelledby="modalEliminarCuentaLabel" aria-hidden="true" data-bs-backdrop="static">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title" id="modalEliminarCuentaLabel"><i class="fas fa-trash-alt"></i> Eliminar Cuenta Maestra</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form action="/digitalpremium/actions/cuentas_maestras_controlador.php" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="accion" value="eliminar_permanente">
                    <input type="hidden" name="id_cuenta_maestra" id="id_cuenta_maestra_delete_permanent">
                    <p>¡Atención! Estás a punto de **eliminar permanentemente** la cuenta:</p>
                    <p class="fs-5 fw-bold text-danger mb-3" id="email_cuenta_delete_permanent"></p>
                    <p>Esta acción **también eliminará todos los perfiles vendidos asociados** a esta cuenta maestra.</p>
                    <p class="text-danger fw-bold">Esta acción no se puede deshacer.</p>
                    <div class="form-check mt-3">
                        <input class="form-check-input" type="checkbox" value="" id="confirmDeletionCheck">
                        <label class="form-check-label" for="confirmDeletionCheck">
                            Entiendo que esta acción es permanente y eliminará la cuenta y sus perfiles asociados.
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-danger" id="confirmDeletePermanentBtn" disabled><i class="fas fa-trash-alt me-2"></i>Eliminar Permanentemente</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal de Avisos -->
<div class="modal fade" id="modalAvisos" tabindex="-1" aria-labelledby="modalAvisosLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-warning text-dark">
                <h5 class="modal-title" id="modalAvisosLabel">
                    <i class="fas fa-exclamation-triangle me-2"></i>Cuentas que Requieren Atención
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <?php if (!empty($alert_accounts)): ?>
                    <div class="alert alert-warning border-0 rounded-3 mb-3">
                        <div class="d-flex align-items-center">
                            <i class="fas fa-info-circle fa-2x text-warning me-3"></i>
                            <div>
                                <strong><?php echo count($alert_accounts); ?> cuentas</strong> requieren tu atención inmediata.
                                <br><small class="text-muted">Revisa las cuentas vencidas, por vencer o con problemas.</small>
                            </div>
                        </div>
                    </div>
                    
                    <div class="list-group">
                        <?php foreach ($alert_accounts as $alert_cuenta): ?>
                            <?php    
                                $text_class = $alert_cuenta['status_details']['status_key'] === 'expired' || $alert_cuenta['status_details']['status_key'] === 'problem' ? 'text-danger' : 'text-warning-emphasis';    
                            ?>
                            <div class="list-group-item border-0 shadow-sm mb-2 rounded-3">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div class="flex-grow-1">
                                        <div class="d-flex align-items-center mb-2">
                                            <i class="<?php echo getPlatformIcon($alert_cuenta['nombre_plataforma']); ?> me-2 text-primary"></i>
                                            <strong><?php echo htmlspecialchars($alert_cuenta['nombre_plataforma']); ?></strong>
                                        </div>
                                        <div class="text-muted small mb-2">
                                            <i class="fas fa-envelope me-1"></i>
                                            <?php echo htmlspecialchars($alert_cuenta['email_cuenta']); ?>
                                        </div>
                                        <div class="d-flex align-items-center gap-2">
                                            <span class="fw-bold <?php echo $text_class; ?> small">
                                                <?php echo $alert_cuenta['status_details']['dias_restantes_text']; ?>
                                            </span>
                                            <?php echo $alert_cuenta['status_details']['status_badge']; ?>
                                        </div>
                                    </div>
                                    <button class="btn btn-sm btn-outline-success" type="button" 
                                            data-bs-dismiss="modal" 
                                            data-bs-toggle="modal" 
                                            data-bs-target="#modalRenovarCuenta" 
                                            data-id="<?php echo $alert_cuenta['id_cuenta_maestra']; ?>" 
                                            data-info="<?php echo htmlspecialchars($alert_cuenta['email_cuenta']); ?>" 
                                            data-costo="<?php echo htmlspecialchars($alert_cuenta['costo_compra']); ?>"
                                            data-plataforma="<?php echo htmlspecialchars($alert_cuenta['nombre_plataforma']); ?>"
                                            data-vencimiento="<?php echo htmlspecialchars(date("d/m/Y", strtotime($alert_cuenta['fecha_vencimiento_proveedor']))); ?>"
                                            data-estado="<?php echo htmlspecialchars($alert_cuenta['estado_cuenta']); ?>"
                                            title="Renovar Cuenta">
                                        <i class="fas fa-sync-alt me-1"></i> Renovar
                                    </button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="text-center py-4">
                        <i class="fas fa-check-circle fa-3x text-success mb-3"></i>
                        <h5>¡Todo en orden!</h5>
                        <p class="text-muted">No hay cuentas que requieran atención en este momento.</p>
                    </div>
                <?php endif; ?>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
            </div>
        </div>
    </div>
</div>

<!-- Modal de Notificación de Cambios -->
<div class="modal fade" id="modalNotificacionCambios" tabindex="-1" aria-labelledby="modalNotificacionCambiosLabel" aria-hidden="true">
    <div class="modal-dialog modal-fullscreen-lg-down modal-xl">
        <div class="modal-content">
            <div class="modal-header bg-gradient text-white" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
                <h5 class="modal-title" id="modalNotificacionCambiosLabel">
                    <i class="fas fa-bell me-2"></i>🔔 Notificación de Cambios en Cuenta Maestra
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <!-- Resumen de cambios -->
                <div class="alert alert-warning border-0 shadow-sm mb-4">
                    <h6 class="alert-heading fw-bold">
                        <i class="fas fa-exclamation-triangle me-2 text-warning"></i>🚨 Cambios Detectados
                    </h6>
                    <p class="mb-0">Se han detectado los siguientes cambios en la cuenta maestra:</p>
                    <ul class="mb-0 mt-2" id="lista-cambios">
                        <!-- Se llena dinámicamente -->
                    </ul>
                    <div class="mt-3 p-3 bg-light rounded">
                        <small class="text-muted">
                            <i class="fas fa-info-circle me-1"></i>
                            <strong>💡 Nota:</strong> El mensaje se genera automáticamente según el tipo de cambio detectado con emojis y formato atractivo. 
                            <br>🔐 <strong>Importante:</strong> Los mensajes incluyen automáticamente las nuevas credenciales de acceso para que los clientes puedan conectarse sin problemas.
                            <br>✨ Puedes personalizarlo antes de enviarlo a los clientes para una mejor experiencia de comunicación.
                            <br>📱 <strong>Seguridad:</strong> Se recomienda notificar a los clientes inmediatamente para evitar problemas de acceso.
                        </small>
                    </div>
                </div>

                <!-- Clientes afectados -->
                <div class="mb-4">
                    <h6 class="fw-bold text-dark mb-3">
                        <i class="fas fa-users me-2 text-primary"></i>👥 Clientes Afectados
                    </h6>
                    
                    <!-- Vista de escritorio -->
                    <div class="d-none d-lg-block">
                        <div class="table-responsive">
                            <table class="table table-hover table-striped">
                                <thead class="table-primary">
                                    <tr>
                                        <th><i class="fas fa-user me-1"></i>Cliente</th>
                                        <th><i class="fas fa-phone me-1"></i>Teléfono</th>
                                        <th><i class="fas fa-tag me-1"></i>Perfil</th>
                                        <th><i class="fas fa-calendar me-1"></i>Vencimiento</th>
                                        <th><i class="fas fa-cogs me-1"></i>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody id="tabla-clientes">
                                    <!-- Se llena dinámicamente -->
                                </tbody>
                            </table>
                        </div>
                    </div>
                    
                    <!-- Vista móvil -->
                    <div class="d-lg-none">
                        <div id="clientes-mobile" class="row g-3">
                            <!-- Se llena dinámicamente -->
                        </div>
                    </div>
                </div>

                <!-- Mensaje personalizado -->
                <div class="mb-4">
                    <h6 class="fw-bold text-dark mb-3">
                        <i class="fas fa-comment me-2 text-primary"></i>💬 Mensaje Personalizado
                    </h6>
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-filter me-1"></i>Tipo de Cambio:
                            </label>
                            <select class="form-select form-select-lg" id="tipo-cambio-select">
                                <option value="general">🔄 Cambio General</option>
                                <option value="email">📧 Cambio de Email</option>
                                <option value="password">🔐 Cambio de Contraseña</option>
                                <option value="correo_password">🔑 Cambio de Correo + Contraseña</option>
                                <option value="plataforma">🖥️ Cambio de Plataforma</option>
                                <option value="estado">⚡ Cambio de Estado</option>
                            </select>
                            <small class="text-muted mt-1 d-block">
                                <i class="fas fa-magic me-1"></i>Se selecciona automáticamente según el cambio detectado
                            </small>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label fw-semibold">
                                <i class="fas fa-info-circle me-1"></i>Variables Disponibles:
                            </label>
                            <div class="alert alert-info py-2 mb-0">
                                <small>
                                    <code>{nombre_cliente}</code> • <code>{plataforma}</code> • <code>{nuevo_email}</code> • <code>{nueva_plataforma}</code>
                                </small>
                            </div>
                        </div>
                    </div>
                    <div class="mt-3">
                        <label class="form-label fw-semibold">
                            <i class="fas fa-edit me-1"></i>💬 Mensaje:
                        </label>
                        <textarea class="form-control form-control-lg" id="mensaje-personalizado" rows="6" 
                                  placeholder="✨ El mensaje se genera automáticamente con emojis y formato atractivo. Puedes personalizarlo aquí..."></textarea>
                        <small class="text-muted mt-1 d-block">
                            <i class="fas fa-lightbulb me-1"></i>🌟 El mensaje incluye automáticamente los datos del cambio:
                            <br>• 📧 <strong>Cambio de Email:</strong> Email anterior → nuevo + contraseña actual
                            <br>• 🔐 <strong>Cambio de Contraseña:</strong> Nueva contraseña + email actual
                            <br>• 🖥️ <strong>Cambio de Plataforma:</strong> Plataforma anterior → nueva + credenciales actuales
                            <br>• ⚡ <strong>Otros cambios:</strong> Credenciales actuales de acceso
                        </small>
                    </div>
                    <div class="d-flex gap-2 mt-3 flex-wrap">
                        <button type="button" class="btn btn-outline-secondary btn-lg" id="btn-cargar-mensaje">
                            <i class="fas fa-download me-2"></i>📥 Cargar Mensaje Predefinido
                        </button>
                        <button type="button" class="btn btn-outline-primary btn-lg" id="btn-copiar-mensaje">
                            <i class="fas fa-copy me-2"></i>📋 Copiar Mensaje
                        </button>
                    </div>
                </div>
            </div>
            <div class="modal-footer bg-light border-top">
                <div class="d-flex justify-content-between w-100 flex-wrap gap-2">
                    <div class="d-flex align-items-center">
                        <small class="text-muted me-3">
                            <i class="fas fa-info-circle me-1"></i>🔄 La página se actualizará automáticamente al cerrar
                        </small>
                    </div>
                    <div class="d-flex gap-2">
                        <button type="button" class="btn btn-secondary btn-lg" data-bs-dismiss="modal">
                            <i class="fas fa-times me-2"></i>❌ Cerrar
                        </button>
                        <button type="button" class="btn btn-warning btn-lg" id="btn-preview-mensaje">
                            <i class="fas fa-eye me-2"></i>👁️ Vista Previa
                        </button>
                        <button type="button" class="btn btn-success btn-lg" id="btn-enviar-todos">
                            <i class="fas fa-paper-plane me-2"></i>📤 Enviar a Todos
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Incluir CSS y JavaScript de notificación -->
<link rel="stylesheet" href="../css/notificacion_cambios.css">
<script src="../js/notificacion_cambios.js"></script>

<?php include_once '../includes/footer.php'; ?>

<script>
document.addEventListener('DOMContentLoaded', function () {
    // Inicializar tooltips de Bootstrap
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(el => new bootstrap.Tooltip(el));

    // --- Lógica de Filtros ---
    const searchInput = document.getElementById('quickSearchInput');
    const platformFilter = document.getElementById('platformFilter');
    const statusFilter = document.getElementById('statusFilter');
    const stockFilter = document.getElementById('stockFilter');
    const minStockInput = document.getElementById('minStockFilter');
    const resetBtn = document.getElementById('resetFiltersBtn');
    const searchForm = document.querySelector('.controls-bar');

    const platformGroups = Array.from(document.querySelectorAll('.platform-group'));
    const noResultsMessage = document.getElementById('no-results-message');

    const updateView = () => {
        const searchTerm = (searchInput?.value || '').toLowerCase().trim();
        const selectedPlatformId = platformFilter?.value || 'all';
        const selectedStatus = statusFilter?.value || 'all';
        const stockMode = stockFilter?.value || 'all';
        const minStock = parseInt(minStockInput?.value || 'NaN', 10);

        let totalVisible = 0;

        platformGroups.forEach(group => {
            const groupId = group.dataset.platformId;
            const wrappers = Array.from(group.querySelectorAll('.account-wrapper'));
            let visibleInGroup = 0;

            wrappers.forEach(w => {
                const text = (w.dataset.searchText || '');
                const status = (w.dataset.statusKey || '');
                const disp = parseInt(w.dataset.perfilesDisponibles || '0', 10);

                const matchSearch = (searchTerm === '' || text.includes(searchTerm));
                const matchPlatform = (selectedPlatformId === 'all' || selectedPlatformId === groupId);
                const matchStatus = (selectedStatus === 'all' || selectedStatus === status);
                const matchStockSel = (stockMode === 'all') ? true
                    : (stockMode === 'with' ? disp > 0 : disp === 0);
                const matchMinStock = isNaN(minStock) ? true : (disp >= minStock);

                const visible = matchSearch && matchPlatform && matchStatus && matchStockSel && matchMinStock;
                w.style.display = visible ? '' : 'none';
                if (visible) visibleInGroup++;
            });

            group.style.display = (visibleInGroup > 0) ? '' : 'none';
            totalVisible += visibleInGroup;
        });

        noResultsMessage.style.display = (totalVisible === 0) ? 'block' : 'none';

        // No actualizar URL en cada input para evitar sobrecarga, solo en 'change' o 'submit'
    };

    const updateUrlState = () => {
        const url = new URL(window.location.href);
        url.searchParams.set('filtro_search_text', (searchInput?.value || '').trim());
        url.searchParams.set('filtro_plataforma', selectedPlatformId);
        url.searchParams.set('filtro_status', selectedStatus);
        window.history.replaceState({}, document.title, url.toString());
    };

    updateView();

    [searchInput, platformFilter, statusFilter, stockFilter, minStockInput].forEach(el => {
        if (!el) return;
        el.addEventListener('input', updateView); // Vista en tiempo real
        el.addEventListener('change', updateUrlState); // URL solo al confirmar
    });

    searchForm?.addEventListener('submit', (e) => {
        e.preventDefault();
        updateUrlState();
    });


    resetBtn?.addEventListener('click', () => {
        if (searchInput) searchInput.value = '';
        if (platformFilter) platformFilter.value = 'all';
        if (statusFilter) statusFilter.value = 'all';
        if (stockFilter) stockFilter.value = 'all';
        if (minStockInput) minStockInput.value = '';
        updateView();
        updateUrlState();
    });

    // --- Lógica de Modales y Formularios ---
    const diasAlerta = <?php echo (int)$dias_alerta_maestras_global; ?>;

    function computeStatus(estadoCuenta, fechaVenceISO) {
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        const fv = new Date(fechaVenceISO + 'T00:00:00');
        const diffDays = Math.floor((fv - today) / (1000 * 60 * 60 * 24));

        let key = 'inactive',
            badge = '<span class="badge status-badge badge-inactive">Inactiva</span>';
        let pbClass = 'bg-secondary',
            pbVal = 0,
            vencClass = '',
            diasTxt = 'Inactiva';

        if (String(estadoCuenta) === '2') {
            key = 'problem';
            badge = '<span class="badge status-badge badge-problem">Problema</span>';
            pbClass = 'bg-danger';
            pbVal = 100;
            diasTxt = 'Con Problemas';
            vencClass = 'text-danger fw-bold';
        } else if (String(estadoCuenta) === '1') {
            if (diffDays < 0) {
                key = 'expired';
                badge = '<span class="badge status-badge badge-expired">Vencida</span>';
                pbClass = 'bg-danger';
                pbVal = 100;
                diasTxt = `Vencida hace ${Math.abs(diffDays)} día${Math.abs(diffDays) === 1 ? '' : 's'}`;
                vencClass = 'text-danger fw-bold';
            } else if (diffDays === 0) {
                key = 'expiring';
                badge = '<span class="badge status-badge badge-expiring">Vence Hoy</span>';
                pbClass = 'bg-warning';
                pbVal = 100;
                diasTxt = 'Vence HOY';
            } else if (diffDays <= diasAlerta) {
                key = 'expiring';
                badge = '<span class="badge status-badge badge-expiring">Por Vencer</span>';
                pbClass = 'bg-warning';
                pbVal = Math.max(0, Math.min(100, (diffDays / diasAlerta) * 100));
                diasTxt = `Vence en ${diffDays} día${diffDays === 1 ? '' : 's'}`;
            } else {
                key = 'active';
                badge = '<span class="badge status-badge badge-active">Activa</span>';
                pbClass = 'bg-success';
                pbVal = Math.min(100, (diffDays / 30) * 100);
                diasTxt = `${diffDays} día${diffDays === 1 ? '' : 's'} restantes`;
            }
        }
        return { key, badge, pbClass, pbVal, vencClass, diasTxt, diffDays };
    }

    function recalcPlatformStockBadge(platformId) {
        const section = document.querySelector(`.platform-group[data-platform-id="${platformId}"]`);
        if (!section) return;
        const wrappers = section.querySelectorAll('.account-wrapper');
        let disp = 0,
            tot = 0,
            count = 0;
        wrappers.forEach(w => {
            const statusKey = w.dataset.statusKey;
            // Solo se suman perfiles si la cuenta está activa
            if (statusKey === 'active' || statusKey === 'expiring') {
                disp += parseInt(w.dataset.perfilesDisponibles || '0', 10);
                tot += parseInt(w.dataset.perfilesTotal || '0', 10);
            }
            count++;
        });
        const header = section.querySelector('.platform-header .badge.bg-info');
        if (header) header.textContent = `Stock: ${disp}/${tot}`;
        const countBadge = section.querySelector('.platform-header .badge:not(.bg-info)');
        if (countBadge) countBadge.textContent = `${count} Cuentas`;
    }

    // Modal de Edición
    const modalEditar = document.getElementById('modalEditarCuenta');
    const formEdit = document.getElementById('form-edit-account');
    if (modalEditar && formEdit) {
        const perfilesTotalEdit = document.getElementById('perfiles_total_edit');
        const perfilesVendidosInput = document.getElementById('perfiles_vendidos');
        const vendidosWarning = document.getElementById('perfiles-vendidos-warning');
        const vendidosCountSpan = document.getElementById('vendidos-count');

        modalEditar.addEventListener('show.bs.modal', function(event) {
            const button = event.relatedTarget;
            const id = button.getAttribute('data-id');
            const perfilesVendidos = parseInt(button.getAttribute('data-perfiles-vendidos'), 10);

            document.getElementById('id_cuenta_maestra_edit').value = id;
            document.getElementById('id_plataforma_edit').value = button.getAttribute('data-id_plataforma');
            document.getElementById('email_cuenta_edit').value = button.getAttribute('data-email_cuenta');
            document.getElementById('password_cuenta_edit').value = '';
            document.getElementById('password_cuenta_edit').placeholder = "•••••••• (dejar en blanco para no cambiar)";
            document.getElementById('nombre_proveedor_edit').value = button.getAttribute('data-nombre_proveedor');
            document.getElementById('fecha_compra_proveedor_edit').value = button.getAttribute('data-fecha_compra_proveedor');
            document.getElementById('fecha_vencimiento_proveedor_edit').value = button.getAttribute('data-fecha_vencimiento_proveedor');
            document.getElementById('costo_compra_edit').value = button.getAttribute('data-costo_compra');
            document.getElementById('perfiles_total_edit').value = button.getAttribute('data-perfiles_total');
            document.getElementById('notas_cuenta_edit').value = button.getAttribute('data-notas_cuenta');
            document.getElementById('estado_cuenta_edit').value = button.getAttribute('data-estado_cuenta');

            // Setear valor de perfiles vendidos y actualizar el mensaje
            perfilesVendidosInput.value = perfilesVendidos;
            vendidosCountSpan.textContent = perfilesVendidos;
            perfilesTotalEdit.min = perfilesVendidos; // Limitar el mínimo
            vendidosWarning.classList.toggle('d-none', perfilesVendidos === 0);
        });

        formEdit.addEventListener('submit', async (e) => {
            e.preventDefault();
            const perfilesTotales = parseInt(perfilesTotalEdit.value, 10);
            const perfilesVendidos = parseInt(perfilesVendidosInput.value, 10);

            if (perfilesTotales < perfilesVendidos) {
                alert('El total de perfiles no puede ser menor que los perfiles ya vendidos.');
                return;
            }
            
            const fd = new FormData(formEdit);
            fd.append('ajax_request', '1');

            try {
                const resp = await fetch('/digitalpremium/actions/cuentas_maestras_controlador.php', { method:'POST', body: fd });
                const data = await resp.json();

                if (!data || !data.success) {
                    alert((data && data.message) || 'Error al guardar.');
                    return;
                }

                const c = data.data || data.cuenta;
                const wrap = document.querySelector(`.account-wrapper[data-id="${c.id_cuenta_maestra}"]`);
                if (!wrap) { location.reload(); return; }

                // Si hay cambios SENSIBLES (email o contraseña) y clientes vinculados, mostrar modal de notificación
                // NOTA: Solo se muestra el modal para cambios de credenciales, no para otros campos como fechas, notas, etc.
                console.log('DEBUG JS - Data recibida:', data);
                console.log('DEBUG JS - Cambios detectados:', data.data?.cambios_detectados);
                console.log('DEBUG JS - Clientes vinculados:', data.data?.clientes_vinculados);
                
                if (data.data && data.data.cambios_detectados && data.data.cambios_detectados.length > 0 && 
                    data.data.clientes_vinculados && data.data.clientes_vinculados.length > 0) {
                    
                    // Verificar que solo sean cambios sensibles (email, contraseña o ambos)
                    const cambiosSensibles = data.data.cambios_detectados.filter(cambio => 
                        cambio === 'email' || cambio === 'password' || cambio === 'correo_password'
                    );
                    
                    console.log('DEBUG JS - Cambios sensibles filtrados:', cambiosSensibles);
                    
                    if (cambiosSensibles.length > 0) {
                        console.log('DEBUG JS - Mostrando modal de notificación');
                        // Cerrar modal de edición
                        bootstrap.Modal.getInstance(modalEditar)?.hide();
                        
                        // Mostrar modal de notificación solo para cambios sensibles
                        mostrarModalNotificacion(data.data);
                        return;
                    } else {
                        console.log('DEBUG JS - No hay cambios sensibles, no se muestra modal');
                    }
                } else {
                    console.log('DEBUG JS - No se cumplen las condiciones para mostrar modal');
                }

                if (String(wrap.dataset.platformId) !== String(c.id_plataforma)) {
                    alert('✅ Actualización aplicada correctamente');
                    location.reload();
                    return;
                }

                const st = computeStatus(String(c.estado_cuenta), c.fecha_vencimiento_proveedor);

                wrap.dataset.perfilesTotal = c.perfiles_total;
                wrap.dataset.perfilesDisponibles = c.perfiles_disponibles;
                wrap.dataset.statusKey = st.key;

                const card = wrap.querySelector('.account-card');
                if (card) {
                    ['account-card-active','account-card-expiring','account-card-expired','account-card-problem','account-card-inactive']
                      .forEach(cls => card.classList.remove(cls));
                    const mapCls = { active:'account-card-active', expiring:'account-card-expiring', expired:'account-card-expired', problem:'account-card-problem', inactive:'account-card-inactive' };
                    if (mapCls[st.key]) card.classList.add(mapCls[st.key]);
                }

                wrap.querySelector('.account-email').textContent = c.email_cuenta || '';
                wrap.querySelector('.info-item:nth-child(1) .info-value').innerHTML = st.badge;
                wrap.querySelector('.info-item:nth-child(2) .info-value').textContent = `${c.perfiles_disponibles} / ${c.perfiles_total}`;
                wrap.querySelector('.info-item:nth-child(4) .info-value').textContent = c.nombre_proveedor || 'N/A';
                wrap.querySelector('.info-item:nth-child(5) .info-value').textContent = c.fecha_compra_proveedor_fmt || '';
                wrap.querySelector('.progress-section .progress-value-text').innerHTML = `${c.fecha_vencimiento_proveedor_fmt || ''} (${st.diasTxt})`;
                
                const costoValue = wrap.querySelector('.info-item:nth-child(6) .info-value');
                costoValue.textContent = c.costo_compra_fmt || (c.costo_compra != null ? c.costo_compra : '');

                const notesContainer = wrap.querySelector('.notes-collapse-content');
                if (notesContainer) notesContainer.innerHTML = c.notas_cuenta ? c.notas_cuenta.replace(/\n/g, '<br>') : '';

                const pbar = wrap.querySelector('.progress .progress-bar');
                if (pbar) {
                    pbar.className = `progress-bar ${st.pbClass}`;
                    pbar.style.width = `${st.pbVal}%`;
                    pbar.setAttribute('aria-valuenow', String(st.pbVal));
                }

                                 recalcPlatformStockBadge(c.id_plataforma);

                 // Mostrar mensaje de éxito
                 alert('✅ Actualización aplicada correctamente');

                 bootstrap.Modal.getInstance(modalEditar)?.hide();
                 updateView();

            } catch (err) {
                console.error("Fetch error:", err);
                alert('Error de red. Intenta nuevamente.');
            }
        });
    }

    // Lógica para el modal de Agregar Cuenta
    const modalAdd = document.getElementById('modalAgregarCuenta');
    if (modalAdd) {
        const plataformaSelect = document.getElementById('id_plataforma_add');
        const perfilesTotalInput = document.getElementById('perfiles_total_add');
        const fechaCompraInput = document.getElementById('fecha_compra_proveedor_add');
        const diasMembresiaInput = document.getElementById('dias_membresia_add');
        const fechaVencimientoInput = document.getElementById('fecha_vencimiento_proveedor_add');

        plataformaSelect.addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const maxPerfiles = selectedOption.getAttribute('data-max_perfiles');
            if (maxPerfiles) { perfilesTotalInput.value = maxPerfiles; }
        });

        const calculateVencimiento = () => {
            const fechaCompra = fechaCompraInput.value;
            const dias = parseInt(diasMembresiaInput.value, 10);
            if (fechaCompra && !isNaN(dias) && dias >= 0) {
                const startDate = new Date(fechaCompra + 'T00:00:00');
                startDate.setDate(startDate.getDate() + dias);
                fechaVencimientoInput.value = startDate.toISOString().split('T')[0];
            }
        };

        const calculateDias = () => {
            const fechaCompra = fechaCompraInput.value;
            const fechaVencimiento = fechaVencimientoInput.value;
            if (fechaCompra && fechaVencimiento) {
                const startDate = new Date(fechaCompra + 'T00:00:00');
                const endDate = new Date(fechaVencimiento + 'T00:00:00');
                if (endDate >= startDate) {
                    const diffTime = Math.abs(endDate - startDate);
                    diasMembresiaInput.value = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
                } else {
                    diasMembresiaInput.value = '';
                }
            }
        };

        diasMembresiaInput.addEventListener('input', calculateVencimiento);
        fechaCompraInput.addEventListener('change', calculateVencimiento);
        fechaVencimientoInput.addEventListener('change', calculateDias);
        
        // Calcular al abrir el modal
        modalAdd.addEventListener('shown.bs.modal', calculateVencimiento);
    }

    // Lógica para el modal de Editar Cuenta (similar al de agregar)
    const modalEdit = document.getElementById('modalEditarCuenta');
    if (modalEdit) {
        const fechaCompraEdit = document.getElementById('fecha_compra_proveedor_edit');
        const diasMembresiaEdit = document.getElementById('dias_membresia_edit');
        const fechaVencimientoEdit = document.getElementById('fecha_vencimiento_proveedor_edit');

        const calculateVencimientoEdit = () => { if (fechaCompraEdit.value && !isNaN(parseInt(diasMembresiaEdit.value, 10))) { const d = new Date(fechaCompraEdit.value + 'T00:00:00'); d.setDate(d.getDate() + parseInt(diasMembresiaEdit.value, 10)); fechaVencimientoEdit.value = d.toISOString().split('T')[0]; } };
        const calculateDiasEdit = () => { if (fechaCompraEdit.value && fechaVencimientoEdit.value) { const s = new Date(fechaCompraEdit.value + 'T00:00:00'), e = new Date(fechaVencimientoEdit.value + 'T00:00:00'); if (e >= s) { diasMembresiaEdit.value = Math.ceil(Math.abs(e - s) / (1000 * 60 * 60 * 24)); } else { diasMembresiaEdit.value = ''; } } };
        diasMembresiaEdit.addEventListener('input', calculateVencimientoEdit);
        fechaCompraEdit.addEventListener('change', calculateVencimientoEdit);
        fechaVencimientoEdit.addEventListener('change', calculateDiasEdit);
    }

    // Toggles de contraseña
    function setupPasswordToggle(inputId, btnId) {
        const input = document.getElementById(inputId);
        const btn = document.getElementById(btnId);
        if (!input || !btn) return;
        btn.addEventListener('click', () => {
            const isPwd = input.type === 'password';
            input.type = isPwd ? 'text' : 'password';
            const icon = btn.querySelector('i');
            if (icon) { icon.classList.toggle('fa-eye'); icon.classList.toggle('fa-eye-slash'); }
        });
    }
    setupPasswordToggle('password_cuenta_add', 'togglePasswordAdd');
    setupPasswordToggle('password_cuenta_edit', 'togglePasswordEdit');

    // Modal de Eliminación
    const modalEliminarPermanente = document.getElementById('modalEliminarCuenta');
    if (modalEliminarPermanente) {
        modalEliminarPermanente.addEventListener('show.bs.modal', function(event) {
            const button = event.relatedTarget;
            document.getElementById('id_cuenta_maestra_delete_permanent').value = button.getAttribute('data-id');
            document.getElementById('email_cuenta_delete_permanent').textContent = button.getAttribute('data-email');
            const confirmCheck = document.getElementById('confirmDeletionCheck');
            const deleteButton = document.getElementById('confirmDeletePermanentBtn');
            confirmCheck.checked = false;
            deleteButton.disabled = true;
            confirmCheck.onchange = function() { deleteButton.disabled = !this.checked; };
        });
    }

    // --- Modal Renovar Cuenta Maestra: fecha por defecto + presets 30/60/90 --- 
    const modalRenovar = document.getElementById('modalRenovarCuenta'); 
    if (modalRenovar) { 
        modalRenovar.addEventListener('show.bs.modal', function(event) { 
            const button = event.relatedTarget; 
            const id = button.getAttribute('data-id');
            const email = button.getAttribute('data-info');
            const costo = button.getAttribute('data-costo');
            const plataforma = button.getAttribute('data-plataforma') || 'N/A';
            const vencimiento = button.getAttribute('data-vencimiento') || 'N/A';
            
            // Llenar información básica
            document.getElementById('id_cuenta_maestra_renew').value = id;
            document.getElementById('info_cuenta_renew').textContent = email;
            document.getElementById('plataforma_cuenta_renew').textContent = plataforma;
            document.getElementById('costo_compra_renew').value = costo;
            document.getElementById('vencimiento_actual_renew').textContent = vencimiento;
            
            // Obtener la fecha de vencimiento actual del perfil
            const fechaVencimientoActual = button.getAttribute('data-fecha-vencimiento');
            let fechaBase;
            
            if (fechaVencimientoActual) {
                // Usar la fecha de vencimiento actual como base
                fechaBase = new Date(fechaVencimientoActual);
            } else {
                // Si no hay fecha, usar hoy + 30 días como fallback
                const today = new Date();
                fechaBase = new Date();
                fechaBase.setDate(today.getDate() + 30);
            }
            
            // Establecer la fecha base en el input
            document.getElementById('fecha_vencimiento_renew').value = fechaBase.toISOString().split('T')[0];
            
            // Resetear todos los botones a estado no seleccionado
            document.querySelectorAll('.preset-renew').forEach(btn => {
                btn.classList.remove('btn-primary');
                btn.classList.add('btn-outline-primary');
            });
        }); 

        // Botones preset - LÓGICA CORREGIDA: selección/deselección
        modalRenovar.addEventListener('click', function(event) { 
            const t = event.target.closest('.preset-renew'); 
            if (!t) return; 
            
            const days = parseInt(t.getAttribute('data-days'), 10) || 30; 
            const fechaInput = document.getElementById('fecha_vencimiento_renew');
            
            // Verificar si este botón ya está seleccionado
            const isSelected = t.classList.contains('btn-primary');
            
            // Si ya está seleccionado, deseleccionarlo y restaurar fecha base
            if (isSelected) {
                // Obtener la fecha base del botón que abrió el modal
                const modalTrigger = document.querySelector('[data-bs-target="#modalRenovarCuenta"][data-id="' + document.getElementById('id_cuenta_maestra_renew').value + '"]');
                const fechaVencimientoActual = modalTrigger.getAttribute('data-fecha-vencimiento');
                
                if (fechaVencimientoActual) {
                    fechaInput.value = fechaVencimientoActual;
                } else {
                    // Fallback: hoy + 30 días
                    const today = new Date();
                    const defaultDate = new Date();
                    defaultDate.setDate(today.getDate() + 30);
                    fechaInput.value = defaultDate.toISOString().split('T')[0];
                }
                
                // Deseleccionar botón
                t.classList.remove('btn-primary');
                t.classList.add('btn-outline-primary');
            } else {
                // Deseleccionar todos los botones primero
                document.querySelectorAll('.preset-renew').forEach(btn => {
                    btn.classList.remove('btn-primary');
                    btn.classList.add('btn-outline-primary');
                });
                
                // Seleccionar botón actual
                t.classList.add('btn-primary');
                t.classList.remove('btn-outline-primary');
                
                // Obtener la fecha base original (fecha de vencimiento del perfil)
                const modalTrigger = document.querySelector('[data-bs-target="#modalRenovarCuenta"][data-id="' + document.getElementById('id_cuenta_maestra_renew').value + '"]');
                const fechaVencimientoActual = modalTrigger.getAttribute('data-fecha-vencimiento');
                
                if (fechaVencimientoActual) {
                    // Calcular nueva fecha: fecha de vencimiento original + días del botón
                    const fechaBase = new Date(fechaVencimientoActual);
                    const newDate = new Date(fechaBase);
                    newDate.setDate(fechaBase.getDate() + days);
                    fechaInput.value = newDate.toISOString().split('T')[0];
                } else {
                    // Fallback: hoy + días del botón
                    const today = new Date();
                    const newDate = new Date();
                    newDate.setDate(today.getDate() + days);
                    fechaInput.value = newDate.toISOString().split('T')[0];
                }
            }
        }); 
    }
});

// Funciones globales
function copyToClipboard(selector, element) {
    const target = document.querySelector(selector);
    if (!target) return;
    const textToCopy = target.tagName === 'INPUT' ? target.value : target.innerText;
    navigator.clipboard.writeText(textToCopy).then(() => {
        const tooltip = bootstrap.Tooltip.getInstance(element);
        const originalTitle = element.getAttribute('data-bs-original-title');
        if (tooltip) {
            element.setAttribute('data-bs-original-title', '¡Copiado!');
            tooltip.show();
            setTimeout(() => {
                tooltip.hide();
                setTimeout(() => { element.setAttribute('data-bs-original-title', originalTitle); }, 150);
            }, 1000);
        }
    }).catch(err => console.error('Error al copiar:', err));
}

function togglePasswordVisibility(inputId, iconElement) {
    const input = document.getElementById(inputId);
    if (!input) return;
    const isPwd = input.type === 'password';
    input.type = isPwd ? 'text' : 'password';
    iconElement.classList.toggle('fa-eye', !isPwd);
    iconElement.classList.toggle('fa-eye-slash', isPwd);
    const newTitle = isPwd ? 'Ocultar' : 'Mostrar';
    iconElement.setAttribute('data-bs-original-title', newTitle);
    bootstrap.Tooltip.getInstance(iconElement)?.update();
}
</script>
</body>
</html>