<?php
// -----------------------------------------------------------------------------
// Dashboard de Usuarios (solo SUPERADMIN) - Hecho para tu base de datos real
// Tablas: auditoria_acciones, usuarios, perfiles_vendidos, cuentas_maestras,
//          plataformas, clientes
// -----------------------------------------------------------------------------
declare(strict_types=1);
ini_set('display_errors','1');
ini_set('display_startup_errors','1');
error_reporting(E_ALL);

if (session_status() === PHP_SESSION_NONE) session_start();
if (!isset($_SESSION['id_usuario'])) { header('Location: /digitalpremium/login.php'); exit; }
if (($_SESSION['rol'] ?? '') !== 'superadmin') { header('Location: /digitalpremium/index.php'); exit; }

require_once __DIR__.'/../config/db.php'; // Debe definir $pdo (PDO)
date_default_timezone_set('America/Lima');

// -------------------------- Parámetros / Filtros ------------------------------
$hoy = date('Y-m-d');
$desde = $_GET['desde'] ?? date('Y-m-01');
$hasta = $_GET['hasta'] ?? $hoy;

$desde_dt = DateTime::createFromFormat('Y-m-d', $desde) ?: new DateTime(date('Y-m-01'));
$hasta_dt = DateTime::createFromFormat('Y-m-d', $hasta) ?: new DateTime($hoy);
$hasta_dt->setTime(23,59,59);

$usuario_f = isset($_GET['usuario']) ? (int)$_GET['usuario'] : 0;
$rol_f     = isset($_GET['rol']) ? trim($_GET['rol']) : '';
$accion_f  = isset($_GET['accion']) ? trim($_GET['accion']) : '';
$entidad_f = isset($_GET['entidad']) ? trim($_GET['entidad']) : '';
$plataf_f  = isset($_GET['plataforma']) ? (int)$_GET['plataforma'] : 0;
$q         = isset($_GET['q']) ? trim($_GET['q']) : '';
$lim_act   = (int)($_GET['limite_actividad'] ?? 50);
$incluir_sessions = isset($_GET['incluir_sessions']) ? 1 : 0;
$mostrar_desglose = isset($_GET['mostrar_desglose']) ? 1 : 0;
$mostrar_estadisticas = isset($_GET['mostrar_estadisticas']) ? 1 : 0;
$mostrar_actividad_hora = isset($_GET['mostrar_actividad_hora']) ? 1 : 0;
$export = $_GET['export'] ?? ''; // '', 'resumen', 'actividad'

// Moneda (ajusta si tienes config):
$MON = 'S/';

// -------------------------- Helpers de datos ---------------------------------
function fetchUsuarios(PDO $pdo): array {
    $sql = "SELECT id_usuario, nombre_completo AS nombre, rol, estado FROM usuarios ORDER BY nombre_completo";
    return $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC) ?: [];
}
function fetchPlataformasMap(PDO $pdo): array {
    $m = [];
    $rs = $pdo->query("SELECT id_plataforma, nombre_plataforma FROM plataformas");
    foreach ($rs->fetchAll(PDO::FETCH_ASSOC) as $r) $m[(int)$r['id_plataforma']] = $r['nombre_plataforma'];
    return $m;
}

$filtros = [
    'usuario_id'      => $usuario_f ?: 0,
    'rol'             => $rol_f,
    'accion'          => $accion_f,
    'entidad'         => $entidad_f,
    'plataforma_id'   => $plataf_f ?: 0,
];

// -------------------------- Consultas: KPIs -----------------------------------
function kpisGlobales(PDO $pdo, DateTime $d1, DateTime $d2, array $f, int $incl_sess): array {
    $P = [':d1'=>$d1->format('Y-m-d 00:00:00'), ':d2'=>$d2->format('Y-m-d 23:59:59')];

    // Ventas & monto desde perfiles_vendidos
    $sv = "SELECT COUNT(*) ventas, COALESCE(SUM(pv.precio_venta),0) monto
           FROM perfiles_vendidos pv
           JOIN cuentas_maestras cm ON cm.id_cuenta_maestra=pv.id_cuenta_maestra";
    $wv = " WHERE pv.fecha_inicio_servicio BETWEEN :d1 AND :d2 ";
    if (!empty($f['usuario_id']))  { $wv .= " AND pv.id_usuario=:u "; $P[':u']=(int)$f['usuario_id']; }
    if (!empty($f['plataforma_id'])) { $wv .= " AND cm.id_plataforma=:p "; $P[':p']=(int)$f['plataforma_id']; }
    $st = $pdo->prepare($sv.$wv); $st->execute($P);
    $ventas = $st->fetch(PDO::FETCH_ASSOC) ?: ['ventas'=>0,'monto'=>0];

    // Auditoría
    $Pa = [':d1'=>$P[':d1'], ':d2'=>$P[':d2']];
    $wa = " WHERE a.creado_en BETWEEN :d1 AND :d2 ";
    if (!$incl_sess) $wa .= " AND a.accion NOT IN ('login','logout') ";
    if (!empty($f['usuario_id'])) { $wa.=" AND a.id_usuario=:u "; $Pa[':u']=(int)$f['usuario_id']; }
    if (!empty($f['rol']))       { $wa.=" AND a.rol_usuario=:r "; $Pa[':r']=$f['rol']; }
    if (!empty($f['accion']))    { $wa.=" AND a.accion=:ac "; $Pa[':ac']=$f['accion']; }
    if (!empty($f['entidad']))   { $wa.=" AND a.entidad=:en "; $Pa[':en']=$f['entidad']; }

    $sqlE = "SELECT COUNT(*) FROM auditoria_acciones a".$wa;
    $e = $pdo->prepare($sqlE); $e->execute($Pa); $eventos = (int)$e->fetchColumn();

    $sqlX = "SELECT 
                 SUM(a.entidad='perfiles_vendidos' AND a.accion IN ('renovar','renew')) AS renov,
                 SUM(a.entidad='perfiles_vendidos' AND a.accion='update') AS edits,
                 SUM(a.entidad='perfiles_vendidos' AND a.accion='delete') AS cortes
               FROM auditoria_acciones a".$wa;
    $xx = $pdo->prepare($sqlX); $xx->execute($Pa);
    $x = $xx->fetch(PDO::FETCH_ASSOC) ?: ['renov'=>0,'edits'=>0,'cortes'=>0];

    return [
        'eventos'=>$eventos,
        'ventas'=>(int)$ventas['ventas'],
        'monto'=>(float)$ventas['monto'],
        'renovaciones'=>(int)$x['renov'],
        'ediciones'=>(int)$x['edits'],
        'eliminaciones'=>(int)$x['cortes'], // cortes = delete en perfiles_vendidos
    ];
}

// -------------------- Rendimiento por usuario (período) ----------------------
function rendimientoPorUsuario(PDO $pdo, DateTime $d1, DateTime $d2, array $f): array {
    $users = fetchUsuarios($pdo);
    $out = [];
    foreach ($users as $u) {
        if (!empty($f['rol']) && $f['rol'] !== $u['rol']) continue;
        if (!empty($f['usuario_id']) && (int)$f['usuario_id'] !== (int)$u['id_usuario']) continue;

        // Ventas
        $P = [':d1'=>$d1->format('Y-m-d 00:00:00'), ':d2'=>$d2->format('Y-m-d 23:59:59'), ':u'=>(int)$u['id_usuario']];
        $sqlV = "SELECT COUNT(*) ventas, COALESCE(SUM(pv.precio_venta),0) monto
                 FROM perfiles_vendidos pv
                 JOIN cuentas_maestras cm ON cm.id_cuenta_maestra=pv.id_cuenta_maestra
                 WHERE pv.id_usuario=:u AND pv.fecha_inicio_servicio BETWEEN :d1 AND :d2";
        if (!empty($f['plataforma_id'])) { $sqlV .= " AND cm.id_plataforma=:p "; $P[':p']=(int)$f['plataforma_id']; }
        $st = $pdo->prepare($sqlV); $st->execute($P);
        $v = $st->fetch(PDO::FETCH_ASSOC) ?: ['ventas'=>0,'monto'=>0];

        // Auditoría por usuario
        $Pa = [':d1'=>$P[':d1'], ':d2'=>$P[':d2'], ':u'=>$P[':u']];
        $sqlA = "SELECT 
                     SUM(a.entidad='perfiles_vendidos' AND a.accion IN ('renovar','renew')) AS renov,
                     SUM(a.entidad='perfiles_vendidos' AND a.accion='delete') AS cortes,
                     SUM(a.entidad='perfiles_vendidos' AND a.accion='update') AS edits,
                     SUM(a.entidad<>'perfiles_vendidos' AND a.accion='delete') AS elim_otras
                   FROM auditoria_acciones a
                   WHERE a.id_usuario=:u AND a.creado_en BETWEEN :d1 AND :d2";
        $st2 = $pdo->prepare($sqlA); $st2->execute($Pa);
        $a = $st2->fetch(PDO::FETCH_ASSOC) ?: ['renov'=>0,'cortes'=>0,'edits'=>0,'elim_otras'=>0];

        $out[] = [
            'id_usuario'=>$u['id_usuario'],
            'usuario'=>$u['nombre'],
            'rol'=>$u['rol'],
            'ventas'=>(int)$v['ventas'],
            'monto'=>(float)$v['monto'],
            'renov'=>(int)$a['renov'],
            'cortes'=>(int)$a['cortes'],
            'ediciones'=>(int)$a['edits'],
            'elim_otras'=>(int)$a['elim_otras'],
        ];
    }
    usort($out, fn($a,$b)=>($b['monto']<=>$a['monto']) ?: ($b['ventas']<=>$a['ventas']));
    return $out;
}

// ----------------------- Desglose por día y usuario ---------------------------
function desgloseDiarioPorUsuario(PDO $pdo, DateTime $d1, DateTime $d2, array $f): array {
    $users = fetchUsuarios($pdo);
    $rows=[];

    $period = new DatePeriod(
        new DateTime($d1->format('Y-m-d')),
        new DateInterval('P1D'),
        (new DateTime($d2->format('Y-m-d')))->modify('+1 day')
    );

    foreach ($period as $d) {
        $fday = $d->format('Y-m-d');
        $pday = [':d1'=>$fday.' 00:00:00', ':d2'=>$fday.' 23:59:59'];
        foreach($users as $u){
            if (!empty($f['rol']) && $f['rol'] !== $u['rol']) continue;
            if (!empty($f['usuario_id']) && (int)$f['usuario_id'] !== (int)$u['id_usuario']) continue;

            $P = $pday + [':u'=>(int)$u['id_usuario']];
            $sqlV = "SELECT COUNT(*) ventas, COALESCE(SUM(pv.precio_venta),0) monto
                     FROM perfiles_vendidos pv
                     JOIN cuentas_maestras cm ON cm.id_cuenta_maestra=pv.id_cuenta_maestra
                     WHERE pv.id_usuario=:u AND pv.fecha_inicio_servicio BETWEEN :d1 AND :d2";
            if (!empty($f['plataforma_id'])) { $sqlV .= " AND cm.id_plataforma=:p "; $P[':p']=(int)$f['plataforma_id']; }
            $st=$pdo->prepare($sqlV); $st->execute($P);
            $v=$st->fetch(PDO::FETCH_ASSOC) ?: ['ventas'=>0,'monto'=>0];

            $sqlA="SELECT 
                        SUM(a.entidad='perfiles_vendidos' AND a.accion IN ('renovar','renew')) AS renov,
                        SUM(a.entidad='perfiles_vendidos' AND a.accion='delete') AS cortes
                       FROM auditoria_acciones a
                       WHERE a.id_usuario=:u AND a.creado_en BETWEEN :d1 AND :d2";
            $st2=$pdo->prepare($sqlA); $st2->execute($P);
            $a=$st2->fetch(PDO::FETCH_ASSOC) ?: ['renov'=>0,'cortes'=>0];

            $rows[] = [
                'fecha'=>$fday,
                'usuario'=>$u['nombre'],
                'rol'=>$u['rol'],
                'ventas'=>(int)$v['ventas'],
                'monto'=>(float)$v['monto'],
                'renov'=>(int)$a['renov'],
                'cortes'=>(int)$a['cortes'],
            ];
        }
    }
    usort($rows, fn($a,$b)=>strcmp($b['fecha'],$a['fecha']) ?: strcmp($a['usuario'],$b['usuario']));
    return $rows;
}

// -------- Detalle ventas + renovaciones (últimos N del período) ---------------
function detalleVentasRenovaciones(PDO $pdo, DateTime $d1, DateTime $d2, array $f, int $limit=50): array {
    $P = [':d1'=>$d1->format('Y-m-d 00:00:00'), ':d2'=>$d2->format('Y-m-d 23:59:59')];
    $w = " WHERE pv.fecha_inicio_servicio BETWEEN :d1 AND :d2 ";
    if (!empty($f['usuario_id'])) { $w.=" AND pv.id_usuario=:u "; $P[':u']=(int)$f['usuario_id']; }
    if (!empty($f['plataforma_id'])) { $w.=" AND cm.id_plataforma=:p "; $P[':p']=(int)$f['plataforma_id']; }

    $sql = "SELECT pv.id_perfil_vendido, pv.fecha_inicio_servicio, pv.precio_venta,
                     u.nombre_completo AS usuario, u.rol,
                     c.nombre_completo AS cliente,
                     p.nombre_plataforma AS plataforma,
                     pv.nombre_perfil_cliente AS perfil
             FROM perfiles_vendidos pv
             JOIN usuarios u ON u.id_usuario=pv.id_usuario
             JOIN cuentas_maestras cm ON cm.id_cuenta_maestra=pv.id_cuenta_maestra
             JOIN plataformas p ON p.id_plataforma=cm.id_plataforma
             LEFT JOIN clientes c ON c.id_cliente=pv.id_cliente
             $w
             ORDER BY pv.fecha_inicio_servicio DESC
             LIMIT ".(int)$limit;
    $st = $pdo->prepare($sql); $st->execute($P);
    $ventas = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    // Renovaciones desde auditoría
    $Pa = [':d1'=>$P[':d1'], ':d2'=>$P[':d2']];
    $wa = " WHERE a.creado_en BETWEEN :d1 AND :d2 AND a.entidad='perfiles_vendidos' AND a.accion IN ('renovar','renew') ";
    if (!empty($f['usuario_id'])) { $wa.=" AND a.id_usuario=:u "; $Pa[':u']=(int)$f['usuario_id']; }

    $sqlA = "SELECT a.id_audit, a.creado_en AS fecha, u.nombre_completo AS usuario, u.rol,
                     a.entidad_id AS id_perfil_vendido,
                     p.nombre_plataforma AS plataforma,
                     pv.nombre_perfil_cliente AS perfil,
                     c.nombre_completo AS cliente
             FROM auditoria_acciones a
             JOIN usuarios u ON u.id_usuario=a.id_usuario
             LEFT JOIN perfiles_vendidos pv ON pv.id_perfil_vendido=a.entidad_id
             LEFT JOIN cuentas_maestras cm ON cm.id_cuenta_maestra=pv.id_cuenta_maestra
             LEFT JOIN plataformas p ON p.id_plataforma=cm.id_plataforma
             LEFT JOIN clientes c ON c.id_cliente=pv.id_cliente
             $wa
             ORDER BY a.creado_en DESC
             LIMIT ".(int)$limit;
    $st2 = $pdo->prepare($sqlA); $st2->execute($Pa);
    $renov = $st2->fetchAll(PDO::FETCH_ASSOC) ?: [];

    return ['ventas'=>$ventas, 'renovaciones'=>$renov];
}

// --------- Actividad reciente (basado en auditoría) ---------------
function actividadReciente(PDO $pdo, DateTime $d1, DateTime $d2, array $f, int $limit=50, int $incl_sess=0, string $q=''): array {
    $P = [':d1'=>$d1->format('Y-m-d 00:00:00'), ':d2'=>$d2->format('Y-m-d 23:59:59')];
    // Base WHERE clause
    $wa = " WHERE a.creado_en BETWEEN :d1 AND :d2 ";
    if (!$incl_sess) $wa .= " AND a.accion NOT IN ('login','logout') ";

    // Apply filters
    if (!empty($f['usuario_id'])) { $wa.=" AND a.id_usuario=:u "; $P[':u']=(int)$f['usuario_id']; }
    if (!empty($f['rol']))        { $wa.=" AND a.rol_usuario=:r "; $P[':r']=$f['rol']; }
    if (!empty($f['accion']))     { $wa.=" AND a.accion=:ac "; $P[':ac']=$f['accion']; }
    if (!empty($f['entidad']))    { $wa.=" AND a.entidad=:en "; $P[':en']=$f['entidad']; }
    
    // Platform filter (only applies if entity is perfiles_vendidos)
    if (!empty($f['plataforma_id'])) {
        $wa .= " AND a.entidad = 'perfiles_vendidos' AND EXISTS (
            SELECT 1 FROM perfiles_vendidos pv_f
            JOIN cuentas_maestras cm_f ON cm_f.id_cuenta_maestra = pv_f.id_cuenta_maestra
            WHERE pv_f.id_perfil_vendido = a.entidad_id AND cm_f.id_plataforma = :p
        )";
        $P[':p'] = (int)$f['plataforma_id'];
    }

    // General search query
    if ($q!=='') {
        $wa.=" AND (
            a.ip LIKE :q OR a.user_agent LIKE :q OR a.detalle_json LIKE :q
            OR u.nombre_completo LIKE :q OR u.username LIKE :q
            OR (a.entidad = 'perfiles_vendidos' AND (pv.nombre_perfil_cliente LIKE :q OR c.nombre_completo LIKE :q OR p.nombre_plataforma LIKE :q))
            OR (a.entidad = 'plataformas' AND p.nombre_plataforma LIKE :q)
            OR (a.entidad = 'clientes' AND c.nombre_completo LIKE :q)
        )";
        $P[':q'] = "%$q%";
    }

    $sqlA = "SELECT a.id_audit, a.creado_en AS fecha, a.id_usuario, u.nombre_completo AS usuario, a.rol_usuario AS rol,
                     a.accion, a.entidad, a.entidad_id, a.ip, a.user_agent, a.detalle_json,
                     pv.nombre_perfil_cliente AS perfil,
                     c.nombre_completo AS cliente,
                     p.nombre_plataforma AS plataforma,
                     pv.precio_venta,
                     u.username AS username_usuario
             FROM auditoria_acciones a
             LEFT JOIN usuarios u ON u.id_usuario=a.id_usuario
             LEFT JOIN perfiles_vendidos pv ON (a.entidad='perfiles_vendidos' AND pv.id_perfil_vendido=a.entidad_id)
             LEFT JOIN cuentas_maestras cm ON cm.id_cuenta_maestra=pv.id_cuenta_maestra
             LEFT JOIN plataformas p ON p.id_plataforma=cm.id_plataforma
             LEFT JOIN clientes c ON c.id_cliente=pv.id_cliente
             $wa
             ORDER BY a.creado_en DESC
             LIMIT ".(int)$limit;
    
    try {
        $stA = $pdo->prepare($sqlA);
        $stA->execute($P);
        $rowsA = $stA->fetchAll(PDO::FETCH_ASSOC) ?: [];
        return $rowsA;
    } catch (PDOException $e) {
        error_log("Error en actividadReciente: " . $e->getMessage());
        error_log("SQL: " . $sqlA);
        error_log("Parámetros: " . print_r($P, true));
        return [];
    }
}

// --------- Detalles de acción con información contextual ---------------
function obtenerDetallesAccion(array $accion): array {
    $detalles = [];
    
    // Parsear JSON de detalles si existe
    if (!empty($accion['detalle_json'])) {
        $json_data = json_decode($accion['detalle_json'], true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($json_data)) {
            $detalles = $json_data;
        }
    }
    
    // Información contextual según entidad y acción
    switch ($accion['entidad']) {
        case 'perfiles_vendidos':
            $detalles['tipo_entidad'] = 'Perfil Vendido';
            $detalles['descripcion'] = 'Perfil: ' . ($accion['perfil'] ?: 'N/A') . 
                                     ' | Cliente: ' . ($accion['cliente'] ?: 'N/A') . 
                                     ' | Plataforma: ' . ($accion['plataforma'] ?: 'N/A');
            if (!empty($accion['precio_venta'])) {
                $detalles['precio'] = $accion['precio_venta'];
            }
            break;
            
        case 'plataformas':
            $detalles['tipo_entidad'] = 'Plataforma';
            if (isset($detalles['nombre'])) {
                $detalles['descripcion'] = 'Plataforma: ' . $detalles['nombre'];
            }
            break;
            
        case 'clientes':
            $detalles['tipo_entidad'] = 'Cliente';
            if (isset($detalles['nombre_completo'])) {
                $detalles['descripcion'] = 'Cliente: ' . $detalles['nombre_completo'];
            }
            break;
            
        case 'usuarios':
            $detalles['tipo_entidad'] = 'Usuario';
            if (isset($detalles['username'])) {
                $detalles['descripcion'] = 'Usuario: ' . $detalles['username'];
            }
            break;
            
        default:
            $detalles['tipo_entidad'] = ucfirst($accion['entidad']);
            $detalles['descripcion'] = 'Entidad: ' . $accion['entidad'] . ' (ID: ' . $accion['entidad_id'] . ')';
    }
    
    // Información de navegador y dispositivo
    if (!empty($accion['user_agent'])) {
        $detalles['navegador'] = obtenerNavegador($accion['user_agent']);
        $detalles['dispositivo'] = obtenerDispositivo($accion['user_agent']);
    }
    
    return $detalles;
}

// --------- Helper para detectar navegador ---------------
function obtenerNavegador(string $user_agent): string {
    if (strpos($user_agent, 'Chrome') !== false) return 'Chrome';
    if (strpos($user_agent, 'Firefox') !== false) return 'Firefox';
    if (strpos($user_agent, 'Safari') !== false) return 'Safari';
    if (strpos($user_agent, 'Edge') !== false) return 'Edge';
    if (strpos($user_agent, 'Opera') !== false) return 'Opera';
    return 'Desconocido';
}

// --------- Helper para detectar dispositivo ---------------
function obtenerDispositivo(string $user_agent): string {
    if (strpos($user_agent, 'Mobile') !== false) return 'Móvil';
    if (strpos($user_agent, 'Tablet') !== false) return 'Tablet';
    if (strpos($user_agent, 'Android') !== false) return 'Android';
    if (strpos($user_agent, 'iPhone') !== false) return 'iPhone';
    if (strpos($user_agent, 'iPad') !== false) return 'iPad';
    return 'Desktop';
}

// --------- Helper para traducir acciones a español ---------------
function traducirAccion(string $accion): string {
    $traducciones = [
        'create' => 'Crear',
        'update' => 'Editar',
        'delete' => 'Eliminar',
        'renovar' => 'Renovar',
        'renew' => 'Renovar',
        'login' => 'Iniciar Sesión',
        'logout' => 'Cerrar Sesión',
        'cambiar_estado' => 'Cambiar Estado',
        'disable' => 'Deshabilitar',
        'enable' => 'Habilitar',
        'read' => 'Leer',
        'view' => 'Ver',
        'export' => 'Exportar',
        'import' => 'Importar',
        'download' => 'Descargar',
        'upload' => 'Subir',
        'restore' => 'Restaurar',
        'archive' => 'Archivar',
        'activate' => 'Activar',
        'deactivate' => 'Desactivar'
    ];
    
    return $traducciones[$accion] ?? ucfirst($accion);
}

// --------- Helper para hacer datos técnicos entendibles ---------------
function hacerDatosEntendibles(string $json_data): array {
    $datos = json_decode($json_data, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return ['error' => 'Formato de datos inválido'];
    }
    
    $explicaciones = [];
    
    foreach ($datos as $clave => $valor) {
        switch ($clave) {
            case 'cliente_id':
                $explicaciones[] = "ID del Cliente: $valor";
                break;
            case 'creacion_multiple':
                $explicaciones[] = "Creación Múltiple: " . ($valor ? 'Sí' : 'No');
                break;
            case 'nombre':
                $explicaciones[] = "Nombre: $valor";
                break;
            case 'email':
                $explicaciones[] = "Correo Electrónico: $valor";
                break;
            case 'telefono':
                $explicaciones[] = "Teléfono: $valor";
                break;
            case 'direccion':
                $explicaciones[] = "Dirección: $valor";
                break;
            case 'estado':
                $explicaciones[] = "Estado: " . ($valor ? 'Activo' : 'Inactivo');
                break;
            case 'rol':
                $explicaciones[] = "Rol: " . ucfirst($valor);
                break;
            case 'precio':
                $explicaciones[] = "Precio: S/ $valor";
                break;
            case 'fecha_inicio':
                $explicaciones[] = "Fecha de Inicio: $valor";
                break;
            case 'fecha_fin':
                $explicaciones[] = "Fecha de Fin: $valor";
                break;
            case 'plataforma_id':
                $explicaciones[] = "ID de Plataforma: $valor";
                break;
            case 'cuenta_maestra_id':
                $explicaciones[] = "ID de Cuenta Maestra: $valor";
                break;
            case 'perfil_id':
                $explicaciones[] = "ID de Perfil: $valor";
                break;
            case 'cambios':
                if (is_array($valor)) {
                    $cambios = [];
                    foreach ($valor as $campo => $val) {
                        $cambios[] = "$campo: $val";
                    }
                    $explicaciones[] = "Cambios Realizados: " . implode(', ', $cambios);
                } else {
                    $explicaciones[] = "Cambios: $valor";
                }
                break;
            case 'motivo':
                $explicaciones[] = "Motivo: $valor";
                break;
            case 'observaciones':
                $explicaciones[] = "Observaciones: $valor";
                break;
            case 'ip_anterior':
                $explicaciones[] = "IP Anterior: $valor";
                break;
            case 'user_agent_anterior':
                $explicaciones[] = "Navegador Anterior: $valor";
                break;
            default:
                // Para campos no reconocidos, mostrar de forma amigable
                if (is_bool($valor)) {
                    $explicaciones[] = ucfirst(str_replace('_', ' ', $clave)) . ": " . ($valor ? 'Sí' : 'No');
                } elseif (is_numeric($valor)) {
                    $explicaciones[] = ucfirst(str_replace('_', ' ', $clave)) . ": $valor";
                } else {
                    $explicaciones[] = ucfirst(str_replace('_', ' ', $clave)) . ": $valor";
                }
                break;
        }
    }
    
    return $explicaciones;
}

// --------- Estadísticas detalladas por acción ---------------
function estadisticasAcciones(PDO $pdo, DateTime $d1, DateTime $d2, array $f): array {
    $P = [':d1'=>$d1->format('Y-m-d 00:00:00'), ':d2'=>$d2->format('Y-m-d 23:59:59')];
    $wa = " WHERE a.creado_en BETWEEN :d1 AND :d2 ";
    
    if (!empty($f['usuario_id'])) { $wa.=" AND a.id_usuario=:u "; $P[':u']=(int)$f['usuario_id']; }
    if (!empty($f['rol']))        { $wa.=" AND a.rol_usuario=:r "; $P[':r']=$f['rol']; }
    
    $sql = "SELECT 
                a.accion,
                a.entidad,
                COUNT(*) as total,
                COUNT(DISTINCT a.id_usuario) as usuarios_unicos
            FROM auditoria_acciones a
            $wa
            GROUP BY a.accion, a.entidad
            ORDER BY total DESC";
    
    $st = $pdo->prepare($sql);
    $st->execute($P);
    return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

// --------- Actividad por hora del día ---------------
function actividadPorHora(PDO $pdo, DateTime $d1, DateTime $d2, array $f): array {
    $P = [':d1'=>$d1->format('Y-m-d 00:00:00'), ':d2'=>$d2->format('Y-m-d 23:59:59')];
    $wa = " WHERE a.creado_en BETWEEN :d1 AND :d2 ";
    
    if (!empty($f['usuario_id'])) { $wa.=" AND a.id_usuario=:u "; $P[':u']=(int)$f['usuario_id']; }
    if (!empty($f['rol']))        { $wa.=" AND a.rol_usuario=:r "; $P[':r']=$f['rol']; }
    
    $sql = "SELECT 
                HOUR(a.creado_en) as hora,
                COUNT(*) as total_acciones,
                COUNT(DISTINCT a.id_usuario) as usuarios_activos
            FROM auditoria_acciones a
            $wa
            GROUP BY HOUR(a.creado_en)
            ORDER BY hora";
    
    $st = $pdo->prepare($sql);
    $st->execute($P);
    return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

// --------- Resumen de actividad en tiempo real ---------------
function resumenActividadTiempoReal(PDO $pdo): array {
    $hoy = date('Y-m-d');
    $ayer = date('Y-m-d', strtotime('-1 day'));
    
    // Actividad de hoy
    $sql_hoy = "SELECT 
                    COUNT(*) as total_hoy,
                    COUNT(DISTINCT id_usuario) as usuarios_hoy,
                    SUM(CASE WHEN accion IN ('create', 'renovar', 'renew') AND entidad = 'perfiles_vendidos' THEN 1 ELSE 0 END) as ventas_hoy
                FROM auditoria_acciones 
                WHERE DATE(creado_en) = ?";
    $st_hoy = $pdo->prepare($sql_hoy);
    $st_hoy->execute([$hoy]);
    $hoy_data = $st_hoy->fetch(PDO::FETCH_ASSOC) ?: ['total_hoy'=>0, 'usuarios_hoy'=>0, 'ventas_hoy'=>0];
    
    // Actividad de ayer
    $sql_ayer = "SELECT 
                    COUNT(*) as total_ayer,
                    COUNT(DISTINCT id_usuario) as usuarios_ayer,
                    SUM(CASE WHEN accion IN ('create', 'renovar', 'renew') AND entidad = 'perfiles_vendidos' THEN 1 ELSE 0 END) as ventas_ayer
                FROM auditoria_acciones 
                WHERE DATE(creado_en) = ?";
    $st_ayer = $pdo->prepare($sql_ayer);
    $st_ayer->execute([$ayer]);
    $ayer_data = $st_ayer->fetch(PDO::FETCH_ASSOC) ?: ['total_ayer'=>0, 'usuarios_ayer'=>0, 'ventas_ayer'=>0];
    
    // Última actividad (últimos 30 minutos)
    $sql_ultima = "SELECT 
                    a.creado_en,
                    u.nombre_completo,
                    a.accion,
                    a.entidad
                FROM auditoria_acciones a
                LEFT JOIN usuarios u ON u.id_usuario = a.id_usuario
                WHERE a.creado_en >= DATE_SUB(NOW(), INTERVAL 30 MINUTE)
                ORDER BY a.creado_en DESC
                LIMIT 5";
    $st_ultima = $pdo->prepare($sql_ultima);
    $st_ultima->execute();
    $ultima_actividad = $st_ultima->fetchAll(PDO::FETCH_ASSOC) ?: [];
    
    return [
        'hoy' => $hoy_data,
        'ayer' => $ayer_data,
        'ultima_actividad' => $ultima_actividad
    ];
}

// ----------------------- Cards HOY por usuario --------------------------------
function cardsHoyPorUsuario(PDO $pdo): array {
    $users = fetchUsuarios($pdo);
    $d1 = date('Y-m-d 00:00:00'); $d2 = date('Y-m-d 23:59:59');
    $out=[];
    foreach($users as $u){
        $P=[':d1'=>$d1, ':d2'=>$d2, ':u'=>(int)$u['id_usuario']];
        // Ventas del día (desde perfiles_vendidos)
        $sqlV="SELECT COUNT(*) ventas, COALESCE(SUM(pv.precio_venta),0) monto
               FROM perfiles_vendidos pv
               WHERE pv.id_usuario=:u AND pv.fecha_inicio_servicio BETWEEN :d1 AND :d2";
        $st=$pdo->prepare($sqlV); $st->execute($P);
        $v=$st->fetch(PDO::FETCH_ASSOC) ?: ['ventas'=>0,'monto'=>0];

        // Ventas adicionales desde auditoría (creaciones de perfiles vendidos)
        $sqlVA="SELECT COUNT(*) ventas_audit
                FROM auditoria_acciones a
                WHERE a.id_usuario=:u AND a.creado_en BETWEEN :d1 AND :d2 
                AND a.entidad='perfiles_vendidos' AND a.accion='create'";
        $stVA=$pdo->prepare($sqlVA); $stVA->execute($P);
        $ventas_audit=(int)$stVA->fetchColumn();

        // Total de ventas = ventas de perfiles_vendidos + ventas de auditoría
        $total_ventas = (int)$v['ventas'] + $ventas_audit;

        // Para el monto, usamos solo el de perfiles_vendidos ya que es más confiable
        // Las ventas de auditoría son principalmente para conteo
        $total_monto = (float)$v['monto'];

        // Auditoría del día
        $sqlA="SELECT 
                  SUM(a.entidad='perfiles_vendidos' AND a.accion IN ('renovar','renew')) AS renov,
                  SUM(a.entidad='perfiles_vendidos' AND a.accion='update') AS edits,
                  SUM(a.entidad='perfiles_vendidos' AND a.accion IN ('delete','cancelar')) AS cortes
                FROM auditoria_acciones a
                WHERE a.id_usuario=:u AND a.creado_en BETWEEN :d1 AND :d2";
        $st2=$pdo->prepare($sqlA); $st2->execute($P);
        $a=$st2->fetch(PDO::FETCH_ASSOC) ?: ['renov'=>0,'edits'=>0,'cortes'=>0];

        $out[]=[
            'id_usuario'=>$u['id_usuario'],
            'usuario'=>$u['nombre'],
            'rol'=>$u['rol'],
            'ventas'=>$total_ventas,
            'monto'=>$total_monto,
            'renov'=>(int)$a['renov'],
            'edits'=>(int)$a['edits'],
            'cortes'=>(int)$a['cortes'],
        ];
    }
    usort($out, fn($a,$b)=>$b['monto']<=>$a['monto']);
    return $out;
}

// ----------------------------- EXPORT CSV -------------------------------------
function csvHeaders(string $filename): void {
    if (ob_get_level()) { ob_end_clean(); }
    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="'.$filename.'"');
    header('Pragma: no-cache'); header('Expires: 0');
}

if ($export === 'resumen') {
    $rows = rendimientoPorUsuario($pdo, $desde_dt, $hasta_dt, $filtros);
    csvHeaders('resumen_usuarios_'.date('Ymd_His').'.csv');
    $out = fopen('php://output','w');
    fputcsv($out,['usuario','rol','ventas','monto_vendido','renovaciones','cortes','ediciones','eliminaciones_otras']);
    foreach($rows as $r){
        fputcsv($out,[
            $r['usuario'], $r['rol'], $r['ventas'],
            number_format($r['monto'],2,'.',''),
            $r['renov'],$r['cortes'],$r['ediciones'],$r['elim_otras']
        ]);
    }
    fclose($out); exit;
}
if ($export === 'actividad') {
    $rows = actividadReciente($pdo, $desde_dt, $hasta_dt, $filtros, max(50,$lim_act), $incluir_sessions, $q);
    csvHeaders('actividad_'.date('Ymd_His').'.csv');
    $out = fopen('php://output','w');
    fputcsv($out,['fecha','usuario','rol','accion','entidad','entidad_id','plataforma','perfil','cliente','ip','user_agent']);
    foreach($rows as $r){
        fputcsv($out,[
            $r['fecha'],$r['usuario'],$r['rol'],$r['accion'],$r['entidad'],$r['entidad_id'],
            $r['plataforma'] ?: '', $r['perfil'] ?: '', $r['cliente'] ?: '', $r['ip'] ?: '', $r['user_agent'] ?: ''
        ]);
    }
    fclose($out); exit;
}

// ----------------------------- Cargar datos -----------------------------------
$usuarios_all   = fetchUsuarios($pdo);
$plataformasMap = fetchPlataformasMap($pdo);
$kpis           = kpisGlobales($pdo, $desde_dt, $hasta_dt, $filtros, $incluir_sessions);
$cardsHoy       = cardsHoyPorUsuario($pdo);
$ranking        = rendimientoPorUsuario($pdo, $desde_dt, $hasta_dt, $filtros);
$desglose       = $mostrar_desglose ? desgloseDiarioPorUsuario($pdo, $desde_dt, $hasta_dt, $filtros) : [];
try {
    $actividad = actividadReciente($pdo, $desde_dt, $hasta_dt, $filtros, $lim_act, $incluir_sessions, $q);
} catch (Exception $e) {
    error_log("Error al cargar actividad reciente: " . $e->getMessage());
    $actividad = [];
    $error_actividad = "Error al cargar la actividad reciente. Por favor, verifica los filtros aplicados.";
}
$estadisticas   = $mostrar_estadisticas ? estadisticasAcciones($pdo, $desde_dt, $hasta_dt, $filtros) : [];
$actividad_hora = $mostrar_actividad_hora ? actividadPorHora($pdo, $desde_dt, $hasta_dt, $filtros) : [];
$tiempo_real    = resumenActividadTiempoReal($pdo);

// ----------------------------- UI / HTML --------------------------------------
$page_title = "Dashboard de Usuarios";
include_once __DIR__.'/../includes/header.php';
include_once __DIR__.'/../includes/sidebar.php';
?>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
<style>
:root {
    --dp-primary: #4f46e5;
    --dp-secondary: #64748b;
    --dp-success: #10b981;
    --dp-warning: #f59e0b;
    --dp-danger: #ef4444;
    --dp-info: #06b6d4;
    --dp-dark: #1e293b;
    --dp-light: #f8fafc;
    --text-dark: #334155;
    --text-muted: #64748b;
    --border-color: #e2e8f0;
    --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
    --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
    --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
    --shadow-xl: 0 20px 25px -5px rgb(0 0 0 / 0.1), 0 8px 10px -6px rgb(0 0 0 / 0.1);
    --gradient-primary: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    --gradient-success: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
    --gradient-warning: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
    --gradient-info: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
    --gradient-dark: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
}

body {
    background: linear-gradient(135deg, #f1f5f9 0%, #e2e8f0 100%);
    font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    color: var(--text-dark);
    line-height: 1.6;
    min-height: 100vh;
}

/* Corrección de colores de texto */
.card {
    color: var(--text-dark) !important;
}

.card-title, .card-header, .card-body {
    color: var(--text-dark) !important;
}

.text-muted {
    color: var(--text-muted) !important;
}

.badge {
    color: #fff !important;
}

/* Asegurar que todos los textos sean visibles */
.activity-card .card-title,
.activity-card .action-text,
.activity-card .detail-text,
.activity-card .activity-time,
.activity-card .activity-user {
    color: var(--text-dark) !important;
}

/* Mejorar contraste de badges */
.badge-role {
    background: var(--dp-primary) !important;
    color: white !important;
}

.badge-success {
    background: var(--dp-success) !important;
    color: white !important;
}

.badge-warning {
    background: var(--dp-warning) !important;
    color: white !important;
}

.badge-danger {
    background: var(--dp-danger) !important;
    color: white !important;
}

.container-fluid {
    max-width: 1400px;
    margin: 0 auto;
}

/* Header principal mejorado */
.dashboard-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 1.5rem;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: var(--shadow-xl);
    position: relative;
    overflow: hidden;
}

.dashboard-header::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
    animation: float 8s ease-in-out infinite;
}

.dashboard-header .content {
    position: relative;
    z-index: 2;
}

.dashboard-title {
    font-size: 2.5rem;
    font-weight: 800;
    margin-bottom: 0.5rem;
    text-shadow: 0 2px 4px rgba(0,0,0,0.3);
}

.dashboard-subtitle {
    font-size: 1.1rem;
    opacity: 0.9;
    margin-bottom: 1.5rem;
}

.dashboard-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

/* Secciones mejoradas */
.section-heading {
    color: var(--dp-dark);
    font-weight: 700;
    font-size: 1.75rem;
    margin-bottom: 2rem;
    padding-bottom: 1rem;
    border-bottom: 3px solid var(--dp-primary);
    position: relative;
    display: flex;
    align-items: center;
    gap: 1rem;
}

.section-heading::before {
    content: '';
    width: 6px;
    height: 2.5rem;
    background: var(--gradient-primary);
    border-radius: 3px;
    position: absolute;
    left: -1.5rem;
}

.section-heading i {
    color: var(--dp-primary);
    font-size: 1.5rem;
}

/* Cards mejoradas */
.card {
    border: none;
    border-radius: 1.25rem;
    box-shadow: var(--shadow-md);
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    background: rgba(255, 255, 255, 0.95);
    backdrop-filter: blur(20px);
    border: 1px solid rgba(255, 255, 255, 0.2);
    overflow: hidden;
}

.card:hover {
    transform: translateY(-8px);
    box-shadow: var(--shadow-xl);
}

.card-header {
    background: var(--gradient-primary);
    color: white;
    border-radius: 1.25rem 1.25rem 0 0 !important;
    padding: 1.5rem 2rem;
    font-weight: 700;
    font-size: 1.25rem;
    border: none;
    position: relative;
}

.card-header::after {
    content: '';
    position: absolute;
    bottom: 0;
    left: 0;
    right: 0;
    height: 1px;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
}

.card-body {
    padding: 2rem;
}

/* KPI Cards Mejoradas */
.kpi-card {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 1.5rem;
    position: relative;
    overflow: hidden;
    min-height: 160px;
    display: flex;
    align-items: center;
    box-shadow: var(--shadow-lg);
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.kpi-card:hover {
    transform: translateY(-8px);
    box-shadow: var(--shadow-xl);
    background: linear-gradient(135deg, #5a67d8 0%, #6b46c1 100%);
}

.kpi-card::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
    animation: float 6s ease-in-out infinite;
}

@keyframes float {
    0%, 100% { transform: translateY(0px) rotate(0deg); }
    50% { transform: translateY(-20px) rotate(180deg); }
}

.kpi-card .card-body {
    position: relative;
    z-index: 2;
    text-align: center;
    width: 100%;
    padding: 2rem;
}

.kpi-label {
    font-size: 0.9rem;
    font-weight: 600;
    opacity: 0.9;
    text-transform: uppercase;
    letter-spacing: 1px;
    margin-bottom: 1rem;
    color: rgba(255, 255, 255, 0.9);
    text-shadow: 0 1px 2px rgba(0,0,0,0.3);
}

.kpi-value {
    font-size: 2.8rem;
    font-weight: 800;
    margin-bottom: 1rem;
    text-shadow: 0 2px 4px rgba(0,0,0,0.3);
    line-height: 1;
}

.kpi-icon {
    font-size: 3.5rem;
    opacity: 0.15;
    position: absolute;
    top: 50%;
    right: 1.5rem;
    transform: translateY(-50%);
    transition: all 0.3s ease;
}

.kpi-card:hover .kpi-icon {
    opacity: 0.25;
    transform: translateY(-50%) scale(1.1);
}

/* Filtros Mejorados */
.filters-card {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border: 1px solid var(--border-color);
    border-radius: 1.5rem;
    box-shadow: var(--shadow-md);
    overflow: hidden;
}

.filters-card .card-header {
    background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%);
    color: white;
    border: none;
    padding: 1.5rem 2rem;
}

.filters-card .card-header h5 {
    margin: 0;
    font-weight: 700;
    font-size: 1.25rem;
}

.filters-card .card-body {
    padding: 2rem;
    background: rgba(255, 255, 255, 0.8);
}

/* Estilos para los campos de filtro */
.filters-card .form-label {
    font-weight: 600;
    color: var(--dp-primary);
    margin-bottom: 0.75rem;
    font-size: 0.9rem;
}

.filters-card .form-control,
.filters-card .form-select {
    border: 2px solid #e2e8f0;
    border-radius: 0.75rem;
    padding: 0.6rem 0.8rem;
    font-size: 0.9rem;
    transition: all 0.3s ease;
    background: #ffffff;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
}

.filters-card .form-control:focus,
.filters-card .form-select:focus {
    border-color: var(--dp-primary);
    box-shadow: 0 0 0 0.2rem rgba(79, 70, 229, 0.15);
    transform: translateY(-1px);
    background: #ffffff;
}

/* Checkboxes mejorados */
.filters-card .form-check {
    margin-bottom: 0.5rem;
}

.filters-card .form-check-input {
    border: 2px solid var(--dp-primary);
    border-radius: 0.5rem;
    width: 1.2rem;
    height: 1.2rem;
}

.filters-card .form-check-input:checked {
    background-color: var(--dp-primary);
    border-color: var(--dp-primary);
}

.filters-card .form-check-label {
    font-size: 0.85rem;
    color: var(--text-dark);
    font-weight: 500;
    margin-left: 0.5rem;
}

/* Botones de filtro */
.filters-card .btn {
    border-radius: 0.75rem;
    font-weight: 600;
    font-size: 0.85rem;
    padding: 0.6rem 1rem;
    transition: all 0.3s ease;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}

.filters-card .btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.15);
}

.filters-card .btn-primary {
    background: linear-gradient(135deg, var(--dp-primary), #7c3aed);
    border: none;
}

.filters-card .btn-outline-secondary {
    border: 2px solid #6c757d;
    color: #6c757d;
}

.filters-card .btn-outline-secondary:hover {
    background: #6c757d;
    border-color: #6c757d;
    color: #ffffff;
}

/* Botones mejorados */
.btn {
    border-radius: 1rem;
    font-weight: 700;
    padding: 0.75rem 1.5rem;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    border: none;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    font-size: 0.9rem;
    box-shadow: var(--shadow-sm);
}

.btn-primary {
    background: var(--gradient-primary);
    box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
}

.btn-primary:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 20px rgba(102, 126, 234, 0.5);
}

.btn-outline-primary {
    border-width: 2px;
    font-weight: 700;
    transition: all 0.3s ease;
    background: rgba(255, 255, 255, 0.1);
    backdrop-filter: blur(10px);
}

.btn-outline-primary:hover {
    transform: translateY(-3px);
    box-shadow: 0 6px 15px rgba(13, 110, 253, 0.4);
    background: rgba(255, 255, 255, 0.2);
}

.btn-outline-success {
    border-width: 2px;
    font-weight: 700;
    transition: all 0.3s ease;
    background: rgba(255, 255, 255, 0.1);
    backdrop-filter: blur(10px);
}

.btn-outline-success:hover {
    transform: translateY(-3px);
    box-shadow: 0 6px 15px rgba(25, 135, 84, 0.4);
    background: rgba(255, 255, 255, 0.2);
}

/* Animaciones mejoradas */
.card {
    animation: fadeInUp 0.8s cubic-bezier(0.4, 0, 0.2, 1);
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(40px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Responsive */
@media (max-width: 768px) {
    .dashboard-title {
        font-size: 2rem;
    }
    
    .dashboard-actions {
        flex-direction: column;
    }
    
    .kpi-value {
        font-size: 2rem;
    }
    
    /* Responsive para los filtros del dashboard */
    .filters-card .card-body {
        padding: 1.5rem;
    }
    
    .filters-card .form-label {
        font-size: 0.8rem;
    }
    
    .filters-card .form-control,
    .filters-card .form-select {
        padding: 0.5rem 0.7rem;
        font-size: 0.85rem;
    }
    
    .filters-card .btn {
        padding: 0.5rem 0.8rem;
        font-size: 0.8rem;
    }
    
    .kpi-card {
        min-height: 140px;
    }
    
    .kpi-card .card-body {
        padding: 1.5rem;
    }
    
    .kpi-value {
        font-size: 2.2rem;
    }
    
    .kpi-icon {
        font-size: 2.8rem;
        right: 1rem;
    }
}
.card {
    border: 1px solid #e9ecef;
    border-radius: 1.25rem;
    box-shadow: 0 8px 16px rgba(0,0,0,.08), 0 2px 8px rgba(0,0,0,.05);
    transition: all 0.3s ease-in-out;
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    backdrop-filter: blur(10px);
}
.card:hover { 
    transform: translateY(-5px); 
    box-shadow: 0 16px 32px rgba(0,0,0,.12), 0 4px 16px rgba(0,0,0,.08);
}
.card-kpi .h4 { font-weight: 700; }
.badge-role { background:#f1f3f5; color:#333; border:1px solid #e9ecef; }
.badge-act-venta, .badge-act-create { background: var(--dp-success); color: white; font-weight: 600; padding: .4em .8em; }
.badge-act-update { background: var(--dp-warning); color: black; font-weight: 600; padding: .4em .8em; }
.badge-act-delete { background: var(--dp-danger); color: white; font-weight: 600; padding: .4em .8em; }
.badge-act-renovar, .badge-act-renew { background: var(--dp-info); color: white; font-weight: 600; padding: .4em .8em; }
.badge-act-login, .badge-act-logout { background: var(--dp-secondary); color: white; font-weight: 600; padding: .4em .8em; }
.text-truncate-2 { display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; overflow: hidden; }
.search-chip { background: var(--dp-chip); border-radius: 999px; padding: .25rem .8rem; font-size: .8rem; font-weight: 500; }
.section-heading {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    font-size: 1.75rem;
    font-weight: 700;
    color: var(--dp-dark);
    margin-bottom: 2rem;
    text-shadow: 0 2px 4px rgba(0,0,0,0.1);
}
.section-heading i { color: var(--dp-primary); }

.card-title-icon {
    font-size: 1.25rem;
    color: var(--dp-primary);
    margin-right: .5rem;
}
.card-value {
    font-weight: 700;
    font-size: 1.25rem;
}
.card-detail-item {
    font-size: 0.875rem;
}
.card-detail-item i {
    color: var(--dp-gray);
    width: 1rem;
    text-align: center;
}
.card-icon-lg {
    font-size: 2.5rem;
    opacity: 0.15;
}

.kpi-card {
    position: relative;
    overflow: hidden;
    border-radius: 1.25rem;
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    border: 1px solid rgba(255,255,255,0.2);
}
.kpi-card .card-body {
    position: relative;
    z-index: 2;
}
.kpi-card .kpi-icon {
    position: absolute;
    right: -10px;
    bottom: -10px;
    font-size: 4.5rem;
    color: rgba(0,0,0,0.08);
    transition: transform 0.3s ease;
}
.kpi-card:hover .kpi-icon {
    transform: scale(1.1) rotate(-5deg);
}
.kpi-card .kpi-value {
    font-size: 2.5rem;
    font-weight: 800;
    background: linear-gradient(135deg, var(--dp-primary), var(--dp-info));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}
.kpi-card .kpi-label {
    font-size: 0.9rem;
    font-weight: 500;
    text-transform: uppercase;
    color: var(--dp-muted);
}

.user-perf-card.top-seller {
    border: 2px solid gold;
    box-shadow: 0 0 20px rgba(255, 215, 0, 0.5);
}
.user-perf-card .crown-icon { color: gold; font-size: 1.2rem; }

.activity-card { border-left: 4px solid var(--dp-gray); }
.activity-card.action-create { border-color: var(--dp-success); }
.activity-card.action-update { border-color: var(--dp-warning); }
.activity-card.action-delete { border-color: var(--dp-danger); }
.activity-card.action-renovar, .activity-card.action-renew { border-color: var(--dp-info); }
.activity-card.action-login, .activity-card.action-logout { border-color: var(--dp-secondary); }

/* Estilos adicionales para el dashboard mejorado */
.border-warning { border-color: var(--dp-warning) !important; }
.border-secondary { border-color: var(--dp-secondary) !important; }
.border-info { border-color: var(--dp-info) !important; }

.text-sm { font-size: 0.875rem; }
.badge-role.text-sm { font-size: 0.75rem; }

.card-title.text-warning { color: var(--dp-warning) !important; }
.card-title.text-secondary { color: var(--dp-secondary) !important; }
.card-title.text-info { color: var(--dp-info) !important; }

/* Animación para las cards de tiempo real */
.card.border-warning:hover,
.card.border-secondary:hover,
.card.border-info:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 12px rgba(0,0,0,0.15);
}

/* Estilo para las estadísticas por acción */
.stats-card {
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border: 1px solid #dee2e6;
}

/* Estilo para actividad por hora */
.hour-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    border: 1px solid #e9ecef;
    transition: all 0.3s ease;
}

.hour-card:hover {
    transform: scale(1.05);
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}

/* Mejoras para las cards de actividad */
.activity-card {
    transition: all 0.3s ease;
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border: 1px solid var(--border-color);
    border-radius: 1rem;
    overflow: hidden;
}

.activity-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 16px rgba(0,0,0,0.1);
}

.activity-card .card-header {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid var(--border-color);
    padding: 1rem;
}

.activity-card .card-body {
    padding: 1.5rem;
    background: #ffffff;
}

.activity-card .card-title {
    color: var(--text-dark) !important;
    font-weight: 600;
    margin-bottom: 0;
}

.activity-card .action-text {
    color: var(--text-dark) !important;
    font-weight: 600;
}

.activity-card .detail-text {
    color: var(--text-dark) !important;
    font-size: 0.9rem;
}

.activity-card .activity-time,
.activity-card .activity-user {
    color: var(--text-dark) !important;
    font-weight: 500;
}

.activity-card .detail-item {
    margin-bottom: 0.75rem;
}

.activity-card .detail-item i {
    width: 16px;
    text-align: center;
    margin-right: 0.5rem;
}

/* Estilo para detalles JSON */
.json-details {
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border: 1px solid #e9ecef;
    border-radius: 0.5rem;
    padding: 0.75rem;
    box-shadow: inset 0 2px 4px rgba(0,0,0,0.1);
}

.json-header {
    background: linear-gradient(135deg, var(--dp-primary) 0%, #6366f1 100%);
    color: white;
    padding: 0.75rem;
    border-radius: 0.5rem;
    margin-bottom: 1rem;
    font-weight: 600;
    display: flex;
    align-items: center;
}

.json-content {
    background: white;
    border-radius: 0.5rem;
    padding: 1rem;
    border: 1px solid #e9ecef;
}

.datos-entendibles {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.dato-item {
    display: flex;
    align-items: center;
    padding: 0.5rem;
    background: #f8f9fa;
    border-radius: 0.5rem;
    border-left: 3px solid var(--dp-success);
    font-size: 0.9rem;
    color: var(--text-dark);
}

.dato-item:hover {
    background: #e9ecef;
    transform: translateX(2px);
    transition: all 0.2s ease;
}

/* Mejoras adicionales para el diseño profesional */
.btn-outline-primary {
    border-width: 2px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-outline-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(13, 110, 253, 0.3);
}

.btn-outline-success {
    border-width: 2px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-outline-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(25, 135, 84, 0.3);
}

/* Animaciones suaves */
.card {
    animation: fadeInUp 0.6s ease-out;
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Mejoras para los badges */
.badge-role {
    background: linear-gradient(135deg, var(--dp-primary) 0%, #6366f1 100%);
    color: white !important;
    border: 1px solid var(--dp-primary);
    font-weight: 700;
    text-shadow: 0 1px 2px rgba(0,0,0,0.1);
    padding: 0.5rem 1rem;
    border-radius: 1rem;
    box-shadow: 0 2px 4px rgba(79, 70, 229, 0.2);
}

.badge-success {
    background: linear-gradient(135deg, var(--dp-success) 0%, #059669 100%);
    color: white !important;
    border: 1px solid var(--dp-success);
    font-weight: 700;
    padding: 0.5rem 1rem;
    border-radius: 1rem;
    box-shadow: 0 2px 4px rgba(16, 185, 129, 0.2);
}

.badge-warning {
    background: linear-gradient(135deg, var(--dp-warning) 0%, #d97706 100%);
    color: white !important;
    border: 1px solid var(--dp-warning);
    font-weight: 700;
    padding: 0.5rem 1rem;
    border-radius: 1rem;
    box-shadow: 0 2px 4px rgba(245, 158, 11, 0.2);
}

.badge-danger {
    background: linear-gradient(135deg, var(--dp-danger) 0%, #dc2626 100%);
    color: white !important;
    border: 1px solid var(--dp-danger);
    font-weight: 700;
    padding: 0.5rem 1rem;
    border-radius: 1rem;
    box-shadow: 0 2px 4px rgba(239, 68, 68, 0.2);
}

/* Tarjetas de rendimiento mejoradas */
.user-perf-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border: 1px solid var(--border-color);
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
}

.user-perf-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: var(--gradient-primary);
    opacity: 0;
    transition: opacity 0.3s ease;
}

.user-perf-card:hover::before {
    opacity: 1;
}

.user-perf-card:hover {
    transform: translateY(-5px);
    box-shadow: var(--shadow-lg);
}

.card-top-seller {
    background: linear-gradient(135deg, #fff7ed 0%, #fed7aa 100%);
    border: 2px solid #f59e0b;
}

.card-top-seller::before {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    opacity: 1;
}

.crown-icon {
    color: #f59e0b;
    font-size: 1.2rem;
    animation: crownGlow 2s ease-in-out infinite alternate;
}

@keyframes crownGlow {
    from { filter: drop-shadow(0 0 2px rgba(245, 158, 11, 0.5)); }
    to { filter: drop-shadow(0 0 8px rgba(245, 158, 11, 0.8)); }
}

/* Mejoras para las listas de estadísticas */
.list-unstyled li {
    padding: 0.5rem 0;
    border-bottom: 1px solid rgba(0,0,0,0.05);
    transition: background-color 0.2s ease;
}

.list-unstyled li:hover {
    background-color: rgba(0,0,0,0.02);
    border-radius: 0.5rem;
    padding-left: 0.5rem;
    padding-right: 0.5rem;
}

.list-unstyled li:last-child {
    border-bottom: none;
}

/* Tarjetas de actividad mejoradas */
.activity-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border: 1px solid var(--border-color);
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
}

.activity-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: var(--gradient-primary);
    opacity: 0;
    transition: opacity 0.3s ease;
}

.activity-card:hover::before {
    opacity: 1;
}

.activity-card:hover {
    transform: translateX(5px);
    box-shadow: var(--shadow-lg);
}

.activity-card.action-create::before {
    background: var(--gradient-success);
}

.activity-card.action-update::before {
    background: var(--gradient-info);
}

.activity-card.action-delete::before {
    background: var(--gradient-warning);
}

.activity-card.action-renovar::before {
    background: var(--gradient-primary);
}

.activity-card.action-login::before,
.activity-card.action-logout::before {
    background: var(--gradient-dark);
}

/* Mejoras para los iconos de actividad */
.card-title-icon {
    font-size: 1.25rem;
    color: var(--dp-primary);
    margin-right: 0.75rem;
    transition: transform 0.3s ease;
}

.activity-card:hover .card-title-icon {
    transform: scale(1.1);
}

/* Mejoras para las tarjetas de hora */
.hour-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border: 1px solid var(--border-color);
    transition: all 0.3s ease;
}

.hour-card:hover {
    transform: translateY(-3px);
    box-shadow: var(--shadow-md);
}

/* Mejoras para las tarjetas de estadísticas */
.stats-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border: 1px solid var(--border-color);
    transition: all 0.3s ease;
}

.stats-card:hover {
    transform: translateY(-3px);
    box-shadow: var(--shadow-md);
}

/* Mejoras para las tarjetas de actividad */
.activity-card-header {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid var(--border-color);
    padding: 1rem 1.5rem;
}

.action-text {
    font-weight: 700;
    color: var(--text-dark);
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.activity-summary {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-radius: 0.75rem;
    padding: 1rem;
    border: 1px solid var(--border-color);
}

.activity-time {
    color: var(--text-dark);
    font-size: 0.95rem;
}

.activity-user {
    color: var(--text-dark);
    font-size: 0.95rem;
}

.activity-details {
    margin-top: 1rem;
}

.detail-item {
    display: flex;
    align-items: flex-start;
    padding: 0.5rem 0;
    border-bottom: 1px solid rgba(0,0,0,0.05);
}

.detail-item:last-child {
    border-bottom: none;
}

.detail-text {
    color: var(--text-dark);
    font-size: 0.9rem;
    line-height: 1.5;
    flex: 1;
}

.json-details {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-radius: 0.75rem;
    border: 1px solid var(--border-color);
    overflow: hidden;
}

.json-header {
    background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%);
    color: white;
    padding: 0.75rem 1rem;
    font-size: 0.9rem;
    font-weight: 600;
}

.json-content {
    padding: 1rem;
    max-height: 200px;
    overflow-y: auto;
}

.json-content pre {
    background: #1e293b;
    color: #e2e8f0;
    padding: 1rem;
    border-radius: 0.5rem;
    font-size: 0.8rem;
    line-height: 1.4;
    margin: 0;
    white-space: pre-wrap;
    word-wrap: break-word;
}

/* Mejoras para los iconos de actividad */
.activity-card .card-title-icon {
    font-size: 1.1rem;
    margin-right: 0.5rem;
}

.activity-card.action-create .card-title-icon {
    color: var(--dp-success);
}

.activity-card.action-update .card-title-icon {
    color: var(--dp-info);
}

.activity-card.action-delete .card-title-icon {
    color: var(--dp-danger);
}

.activity-card.action-renovar .card-title-icon {
    color: var(--dp-primary);
}

.activity-card.action-login .card-title-icon,
.activity-card.action-logout .card-title-icon {
    color: var(--dp-secondary);
}

/* Mejoras para las tarjetas de "Hoy por Usuario" */
.today-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    border: 1px solid var(--border-color);
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
}

.today-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: var(--gradient-success);
    opacity: 0;
    transition: opacity 0.3s ease;
}

.today-card:hover::before {
    opacity: 1;
}

.today-card:hover {
    transform: translateY(-5px);
    box-shadow: var(--shadow-lg);
}

.today-card-header {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid var(--border-color);
    padding: 1rem 1.5rem;
}

.today-sales-badge {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    padding: 0.5rem 1rem;
    border-radius: 2rem;
    font-size: 1.1rem;
    font-weight: 700;
    box-shadow: 0 2px 8px rgba(16, 185, 129, 0.3);
}

.today-summary {
    background: linear-gradient(135deg, #f0fdf4 0%, #dcfce7 100%);
    border-radius: 0.75rem;
    padding: 1rem;
    border: 1px solid #bbf7d0;
    text-align: center;
}

.today-amount {
    color: var(--text-dark);
}

.today-stats {
    margin: 1rem 0;
}

.stat-item {
    padding: 0.5rem;
    border-radius: 0.5rem;
    transition: all 0.3s ease;
}

.stat-item:hover {
    background: rgba(0,0,0,0.02);
    transform: translateY(-2px);
}

.stat-item i {
    font-size: 1.5rem;
    margin-bottom: 0.5rem;
}

.stat-value {
    font-size: 1.5rem;
    font-weight: 700;
    margin-bottom: 0.25rem;
}

.stat-label {
    font-size: 0.8rem;
    color: var(--text-muted);
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.today-efficiency {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-radius: 0.75rem;
    padding: 1rem;
    border: 1px solid var(--border-color);
}
</style>

<div class="container-fluid py-4">

    <div class="dashboard-header">
        <div class="content">
            <div class="d-flex flex-wrap gap-4 align-items-center justify-content-between">
                <div>
                    <h1 class="dashboard-title mb-2">
                        <i class="fa-solid fa-chart-line me-3"></i>Dashboard de Usuarios
                    </h1>
                    <p class="dashboard-subtitle mb-0">
                        <i class="fa-solid fa-calendar-day me-2"></i>
                        Período: <?=htmlspecialchars($desde_dt->format('d/m/Y'))?> - <?=htmlspecialchars($hasta_dt->format('d/m/Y'))?>
                    </p>
                </div>
                <div class="dashboard-actions">
                    <button id="refreshBtn" class="btn btn-outline-light me-2">
                        <i class="fa-solid fa-sync-alt me-2"></i>Actualizar Datos
                    </button>
                    <a class="btn btn-outline-primary" href="?<?=http_build_query(array_merge($_GET,['export'=>'resumen']))?>">
                        <i class="fa-solid fa-file-export me-2"></i>Exportar Resumen
                    </a>
                    <a class="btn btn-outline-success" href="?<?=http_build_query(array_merge($_GET,['export'=>'actividad']))?>">
                        <i class="fa-solid fa-file-export me-2"></i>Exportar Actividad
                    </a>
                </div>
            </div>
        </div>
    </div>

    <div class="card mb-5 filters-card">
        <div class="card-header">
            <h5 class="card-title mb-0">
                <i class="fa-solid fa-filter me-2"></i>Filtros Avanzados
            </h5>
        </div>
        <div class="card-body">
            <?php
            $filtros_activos = [];
            if ($usuario_f > 0) $filtros_activos[] = "Usuario: " . ($usuarios_all[$usuario_f]['nombre'] ?? 'N/A');
            if ($rol_f) {
                $roles_nombres = [
                    'superadmin' => 'Super Administrador',
                    'admin' => 'Administrador',
                    'ventas' => 'Ventas',
                    'cortes' => 'Cortes'
                ];
                $filtros_activos[] = "Rol: " . ($roles_nombres[$rol_f] ?? ucfirst($rol_f));
            }
            if ($accion_f) $filtros_activos[] = "Acción: " . traducirAccion($accion_f);
            if ($entidad_f) {
                $entidades_nombres = [
                    'usuarios' => 'Usuarios',
                    'plataformas' => 'Plataformas',
                    'perfiles_vendidos' => 'Perfiles Vendidos',
                    'cuentas_maestras' => 'Cuentas Maestras',
                    'clientes' => 'Clientes'
                ];
                $filtros_activos[] = "Entidad: " . ($entidades_nombres[$entidad_f] ?? ucfirst($entidad_f));
            }
            if ($plataf_f > 0) $filtros_activos[] = "Plataforma: " . ($plataformasMap[$plataf_f] ?? 'N/A');
            if ($q) $filtros_activos[] = "Búsqueda: " . $q;
            ?>
            
            <?php if (!empty($filtros_activos)): ?>
                <div class="alert alert-info alert-dismissible fade show mb-3" role="alert">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Filtros activos:</strong> <?= implode(' | ', $filtros_activos) ?>
                    <a href="?" class="btn btn-sm btn-outline-info ms-2">Limpiar Filtros</a>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>
            
            <form class="row g-4" method="get">
                <!-- Fila 1: Fechas y Usuario -->
                <div class="col-12 col-lg-3">
                    <label class="form-label fw-bold text-primary mb-2">
                        <i class="fas fa-calendar me-1"></i>Rango de Fechas
                    </label>
                    <div class="row g-2">
                        <div class="col-6">
                            <input type="date" name="desde" class="form-control form-control-sm" value="<?=htmlspecialchars($desde_dt->format('Y-m-d'))?>">
                            <small class="text-muted">Desde</small>
                        </div>
                        <div class="col-6">
                            <input type="date" name="hasta" class="form-control form-control-sm" value="<?=htmlspecialchars($hasta_dt->format('Y-m-d'))?>">
                            <small class="text-muted">Hasta</small>
                        </div>
                    </div>
                </div>
                
                <div class="col-12 col-lg-3">
                    <label class="form-label fw-bold text-primary mb-2">
                        <i class="fas fa-user me-1"></i>Usuario
                    </label>
                    <select name="usuario" class="form-select form-select-sm">
                        <option value="0">Todos los usuarios</option>
                        <?php foreach($usuarios_all as $u): ?>
                            <option value="<?=$u['id_usuario']?>" <?=$usuario_f===(int)$u['id_usuario']?'selected':''?>>
                                <?=htmlspecialchars($u['nombre'])?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-12 col-lg-3">
                    <label class="form-label fw-bold text-primary mb-2">
                        <i class="fas fa-user-tag me-1"></i>Rol
                    </label>
                    <select name="rol" class="form-select form-select-sm">
                        <option value="">Todos los roles</option>
                        <?php 
                        $roles = [
                            'superadmin' => 'Super Administrador',
                            'admin' => 'Administrador',
                            'ventas' => 'Ventas',
                            'cortes' => 'Cortes'
                        ];
                        foreach($roles as $valor => $nombre): ?>
                            <option value="<?=$valor?>" <?=$rol_f===$valor?'selected':''?>><?=$nombre?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-12 col-lg-3">
                    <label class="form-label fw-bold text-primary mb-2">
                        <i class="fas fa-cog me-1"></i>Acción
                    </label>
                    <select name="accion" class="form-select form-select-sm">
                        <option value="">Todas las acciones</option>
                        <?php foreach(['create','update','delete','renovar','renew','login','logout'] as $a): ?>
                            <option value="<?=$a?>" <?=$accion_f===$a?'selected':''?>><?=traducirAccion($a)?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <!-- Fila 2: Entidad, Plataforma y Búsqueda -->
                <div class="col-12 col-lg-4">
                    <label class="form-label fw-bold text-primary mb-2">
                        <i class="fas fa-database me-1"></i>Entidad
                    </label>
                    <select name="entidad" class="form-select form-select-sm">
                        <option value="">Todas las entidades</option>
                        <?php 
                        $entidades = [
                            'usuarios' => 'Usuarios',
                            'plataformas' => 'Plataformas',
                            'perfiles_vendidos' => 'Perfiles Vendidos',
                            'cuentas_maestras' => 'Cuentas Maestras',
                            'clientes' => 'Clientes'
                        ];
                        foreach($entidades as $valor => $nombre): ?>
                            <option value="<?=$valor?>" <?=$entidad_f===$valor?'selected':''?>><?=$nombre?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-12 col-lg-4">
                    <label class="form-label fw-bold text-primary mb-2">
                        <i class="fas fa-tv me-1"></i>Plataforma
                    </label>
                    <select name="plataforma" class="form-select form-select-sm">
                        <option value="0">Todas las plataformas</option>
                        <?php foreach($plataformasMap as $id=>$nombre): ?>
                            <option value="<?=$id?>" <?=$plataf_f===$id?'selected':''?>><?=htmlspecialchars($nombre)?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="col-12 col-lg-4">
                    <label class="form-label fw-bold text-primary mb-2">
                        <i class="fas fa-search me-1"></i>Búsqueda
                    </label>
                    <input type="text" name="q" class="form-control form-control-sm" value="<?=htmlspecialchars($q)?>" placeholder="Buscar por usuario, cliente, perfil, IP...">
                </div>
                
                <!-- Fila 3: Opciones avanzadas -->
                <div class="col-12 col-lg-6">
                    <label class="form-label fw-bold text-primary mb-2">
                        <i class="fas fa-sliders me-1"></i>Opciones Avanzadas
                    </label>
                    <div class="row g-3">
                        <div class="col-6">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="incluir_sessions" value="1" id="chks" <?=$incluir_sessions?'checked':''?>>
                                <label class="form-check-label text-sm" for="chks">Incluir sesiones</label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="mostrar_desglose" value="1" id="chk-desglose" <?=$mostrar_desglose?'checked':''?>>
                                <label class="form-check-label text-sm" for="chk-desglose">Desglose diario</label>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="mostrar_estadisticas" value="1" id="chk-estadisticas" <?=$mostrar_estadisticas?'checked':''?>>
                                <label class="form-check-label text-sm" for="chk-estadisticas">Estadísticas</label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="mostrar_actividad_hora" value="1" id="chk-actividad-hora" <?=$mostrar_actividad_hora?'checked':''?>>
                                <label class="form-check-label text-sm" for="chk-actividad-hora">Actividad por hora</label>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="col-12 col-lg-3">
                    <label class="form-label fw-bold text-primary mb-2">
                        <i class="fas fa-list-ol me-1"></i>Límite de Actividad
                    </label>
                    <input type="number" name="limite_actividad" class="form-control form-control-sm" value="<?=$lim_act?>" min="10" max="500">
                </div>
                
                <div class="col-12 col-lg-3">
                    <label class="form-label fw-bold text-primary mb-2">
                        <i class="fas fa-play me-1"></i>Acciones
                    </label>
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-primary btn-sm">
                            <i class="fas fa-filter me-1"></i>Aplicar Filtros
                        </button>
                        <a href="?" class="btn btn-outline-secondary btn-sm">
                            <i class="fas fa-eraser me-1"></i>Limpiar
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- KPIs del período seleccionado -->
    <h5 class="section-heading">
        <i class="fa-solid fa-chart-line"></i>Métricas Principales
        <small class="text-muted ms-2 fw-normal">(<?=$desde_dt->format('d/m/Y')?> a <?=$hasta_dt->format('d/m/Y')?>)</small>
    </h5>
    <div class="row g-4 mb-5">
        <div class="col-6 col-lg-3">
            <div class="card kpi-card">
                <div class="card-body">
                    <div class="kpi-label">Eventos</div>
                    <div class="kpi-value"><?=$kpis['eventos']?></div>
                    <i class="fa-solid fa-bell-concierge kpi-icon"></i>
                </div>
            </div>
        </div>
        <div class="col-6 col-lg-3">
            <div class="card kpi-card">
                <div class="card-body">
                    <div class="kpi-label">Ventas</div>
                    <div class="kpi-value"><?=$kpis['ventas']?></div>
                    <i class="fa-solid fa-sack-dollar kpi-icon"></i>
                </div>
            </div>
        </div>
        <div class="col-6 col-lg-3">
            <div class="card kpi-card">
                <div class="card-body">
                    <div class="kpi-label">Monto</div>
                    <div class="kpi-value"><?=$MON?><?=number_format($kpis['monto'],2)?></div>
                    <i class="fa-solid fa-coins kpi-icon"></i>
                </div>
            </div>
        </div>
        <div class="col-6 col-lg-3">
            <div class="card kpi-card">
                <div class="card-body">
                    <div class="kpi-label">Renovaciones</div>
                    <div class="kpi-value"><?=$kpis['renovaciones']?></div>
                    <i class="fa-solid fa-arrows-rotate kpi-icon"></i>
                </div>
            </div>
        </div>
        <div class="col-6 col-md-2">
            <div class="card kpi-card">
                <div class="card-body">
                    <div class="kpi-label">Ediciones</div>
                    <div class="kpi-value"><?=$kpis['ediciones']?></div>
                    <i class="fa-solid fa-pencil kpi-icon"></i>
                </div>
            </div>
        </div>
        <div class="col-6 col-md-2">
            <div class="card kpi-card">
                <div class="card-body">
                    <div class="kpi-label">Cortes/Elim.</div>
                    <div class="kpi-value"><?=$kpis['eliminaciones']?></div>
                    <i class="fa-solid fa-trash-can kpi-icon"></i>
                </div>
            </div>
        </div>
    </div>

    <!-- Actividad en tiempo real -->
    <h5 class="section-heading"><i class="fa-solid fa-bolt text-warning"></i>Actividad en tiempo real</h5>
    <div class="row g-3 mb-4">
        <div class="col-12 col-md-4">
            <div class="card border-warning">
                <div class="card-body text-center">
                    <h6 class="card-title text-warning"><i class="fa-solid fa-calendar-day me-2"></i>Hoy</h6>
                    <div class="row text-center">
                        <div class="col-4">
                            <div class="h4 mb-0 fw-bold text-primary"><?=$tiempo_real['hoy']['total_hoy']?></div>
                            <small class="text-muted">Eventos</small>
                        </div>
                        <div class="col-4">
                            <div class="h4 mb-0 fw-bold text-success"><?=$tiempo_real['hoy']['usuarios_hoy']?></div>
                            <small class="text-muted">Usuarios</small>
                        </div>
                        <div class="col-4">
                            <div class="h4 mb-0 fw-bold text-info"><?=$tiempo_real['hoy']['ventas_hoy']?></div>
                            <small class="text-muted">Ventas</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-12 col-md-4">
            <div class="card border-secondary">
                <div class="card-body text-center">
                    <h6 class="card-title text-secondary"><i class="fa-solid fa-calendar-minus me-2"></i>Ayer</h6>
                    <div class="row text-center">
                        <div class="col-4">
                            <div class="h4 mb-0 fw-bold text-primary"><?=$tiempo_real['ayer']['total_ayer']?></div>
                            <small class="text-muted">Eventos</small>
                        </div>
                        <div class="col-4">
                            <div class="h4 mb-0 fw-bold text-success"><?=$tiempo_real['ayer']['usuarios_ayer']?></div>
                            <small class="text-muted">Usuarios</small>
                        </div>
                        <div class="col-4">
                            <div class="h4 mb-0 fw-bold text-info"><?=$tiempo_real['ayer']['ventas_ayer']?></div>
                            <small class="text-muted">Ventas</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-12 col-md-4">
            <div class="card border-info">
                <div class="card-body">
                    <h6 class="card-title text-info"><i class="fa-solid fa-clock me-2"></i>Última actividad (30 min)</h6>
                    <?php if (empty($tiempo_real['ultima_actividad'])): ?>
                        <p class="text-muted mb-0">Sin actividad reciente</p>
                    <?php else: ?>
                        <ul class="list-unstyled mb-0">
                            <?php foreach($tiempo_real['ultima_actividad'] as $act): ?>
                            <li class="d-flex justify-content-between align-items-center mb-1">
                                <span class="text-sm"><?=htmlspecialchars($act['nombre_completo'] ?: 'Usuario')?></span>
                                <span class="badge badge-role text-sm"><?=htmlspecialchars($act['accion'])?></span>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <h5 class="section-heading">
        <i class="fa-solid fa-users"></i>Rendimiento de Hoy por Usuario
        <small class="text-muted ms-2 fw-normal">(<?=date('d/m/Y')?>)</small>
    </h5>
    <div class="row g-4 mb-5">
        <?php foreach($cardsHoy as $c): ?>
        <div class="col-12 col-md-6 col-xl-4">
            <div class="card h-100 today-card" style="background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);">
                <div class="card-header today-card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h5 class="mb-1 fw-bold text-dark"><?=htmlspecialchars($c['usuario'])?></h5>
                            <span class="badge badge-role text-capitalize"><?=$c['rol']?></span>
                        </div>
                        <div class="text-end">
                            <div class="today-sales-badge">
                                <i class="fa-solid fa-sack-dollar me-1"></i>
                                <span class="fw-bold"><?=$c['ventas']?></span>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="card-body d-flex flex-column">
                    <div class="today-summary mb-3">
                        <div class="today-amount">
                            <i class="fa-solid fa-coins me-2 text-success"></i>
                            <span class="fw-bold text-success fs-4"><?=$MON?> <?=number_format($c['monto'],2)?></span>
                            <small class="text-muted d-block">Monto total vendido hoy</small>
                        </div>
                    </div>
                    <div class="today-stats">
                        <div class="row text-center">
                            <div class="col-4">
                                <div class="stat-item">
                                    <i class="fa-solid fa-arrows-rotate text-info"></i>
                                    <div class="stat-value text-info"><?=$c['renov']?></div>
                                    <div class="stat-label">Renovaciones</div>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="stat-item">
                                    <i class="fa-solid fa-pencil text-warning"></i>
                                    <div class="stat-value text-warning"><?=$c['edits']?></div>
                                    <div class="stat-label">Ediciones</div>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="stat-item">
                                    <i class="fa-solid fa-scissors text-danger"></i>
                                    <div class="stat-value text-danger"><?=$c['cortes']?></div>
                                    <div class="stat-label">Cortes</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="mt-auto pt-3 border-top">
                        <div class="today-efficiency">
                            <div class="row text-center">
                                <div class="col-6">
                                    <div class="small text-muted">Promedio por venta</div>
                                    <div class="fw-bold text-success"><?=$MON?><?=number_format($c['monto'] > 0 ? $c['monto'] / $c['ventas'] : 0, 2)?></div>
                                </div>
                                <div class="col-6">
                                    <div class="small text-muted">Eficiencia</div>
                                    <div class="fw-bold text-primary"><?=number_format(($c['ventas'] / max(1, $c['ventas'] + $c['cortes'])) * 100, 1)?>%</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php endforeach; ?>
    </div>

    <h5 class="section-heading">
        <i class="fa-solid fa-ranking-star"></i>Rendimiento por Usuario (Período)
        <small class="text-muted ms-2 fw-normal">(<?=$desde_dt->format('d/m/Y')?> a <?=$hasta_dt->format('d/m/Y')?>)</small>
    </h5>
    <div class="row g-4 mb-5">
        <?php $topSellerId = count($ranking) > 0 ? $ranking[0]['id_usuario'] : null; ?>
        <?php foreach($ranking as $r): ?>
        <?php 
            $cardClass = 'card h-100 user-perf-card';
            $iconHtml = '';
            if ($r['id_usuario'] === $topSellerId) {
                $cardClass .= ' card-top-seller';
                $iconHtml = '<i class="fa-solid fa-crown crown-icon me-2"></i>';
            }
        ?>
        <div class="col-12 col-md-6 col-lg-4">
            <div class="<?=$cardClass?>" style="background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start mb-3">
                        <div>
                            <h5 class="mb-1 fw-bold d-flex align-items-center text-dark"><?=$iconHtml?><?=htmlspecialchars($r['usuario'])?></h5>
                            <span class="badge badge-role text-capitalize px-3 py-2" style="font-size: 0.8rem;"><?=$r['rol']?></span>
                        </div>
                        <div class="text-end">
                            <div class="small text-muted text-uppercase fw-semibold">VENTAS</div>
                            <div class="h3 mb-0 fw-bold text-success"><?=$r['ventas']?></div>
                        </div>
                    </div>
                    <ul class="list-unstyled mb-0 border-top pt-3">
                        <li class="d-flex justify-content-between align-items-center mb-2">
                            <span><i class="fa-solid fa-coins me-2 text-success"></i>Monto vendido</span>
                            <span class="fw-bold text-success"><?=$MON?> <?=number_format($r['monto'],2)?></span>
                        </li>
                        <li class="d-flex justify-content-between align-items-center mb-2">
                            <span><i class="fa-solid fa-arrows-rotate me-2 text-info"></i>Renovaciones</span>
                            <span class="fw-bold text-info"><?=$r['renov']?></span>
                        </li>
                        <li class="d-flex justify-content-between align-items-center mb-2">
                            <span><i class="fa-solid fa-scissors me-2 text-danger"></i>Cortes</span>
                            <span class="fw-bold text-danger"><?=$r['cortes']?></span>
                        </li>
                        <li class="d-flex justify-content-between align-items-center mb-2">
                            <span><i class="fa-solid fa-pencil me-2 text-warning"></i>Ediciones</span>
                            <span class="fw-bold text-warning"><?=$r['ediciones']?></span>
                        </li>
                        <li class="d-flex justify-content-between align-items-center">
                            <span><i class="fa-solid fa-trash me-2 text-secondary"></i>Elim. otras</span>
                            <span class="fw-bold text-secondary"><?=$r['elim_otras']?></span>
                        </li>
                    </ul>
                    <div class="mt-3 pt-3 border-top">
                        <div class="row text-center">
                            <div class="col-6">
                                <div class="small text-muted">Promedio por venta</div>
                                <div class="fw-bold text-success"><?=$MON?><?=number_format($r['monto'] > 0 ? $r['monto'] / $r['ventas'] : 0, 2)?></div>
                            </div>
                            <div class="col-6">
                                <div class="small text-muted">Eficiencia</div>
                                <div class="fw-bold text-primary"><?=number_format(($r['ventas'] / max(1, $r['ventas'] + $r['cortes'])) * 100, 1)?>%</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php endforeach; ?>
    </div>

    <?php if ($mostrar_desglose): ?>
        <h5 class="section-heading"><i class="fa-solid fa-calendar-day"></i>Desglose por día y usuario <small class="text-muted ms-2">(<?=$desde_dt->format('Y-m-d')?> a <?=$hasta_dt->format('Y-m-d')?>)</small></h5>
        <div class="row g-3 mb-4">
            <?php foreach($desglose as $d): ?>
            <div class="col-12 col-md-6 col-xl-4">
                <div class="card h-100">
                    <div class="card-body">
                        <h6 class="card-title fw-bold mb-3"><?=htmlspecialchars($d['fecha'])?> <span class="badge bg-secondary ms-2"><?=htmlspecialchars($d['usuario'])?></span></h6>
                        <ul class="list-unstyled mb-0">
                            <li class="d-flex justify-content-between align-items-center mb-2">
                                <span>Ventas</span>
                                <span class="fw-bold text-success"><?=$d['ventas']?></span>
                            </li>
                            <li class="d-flex justify-content-between align-items-center mb-2">
                                <span>Monto</span>
                                <span class="fw-bold text-success"><?=$MON?> <?=number_format($d['monto'],2)?></span>
                            </li>
                            <li class="d-flex justify-content-between align-items-center mb-2">
                                <span>Renovaciones</span>
                                <span class="fw-bold text-info"><?=$d['renov']?></span>
                            </li>
                            <li class="d-flex justify-content-between align-items-center">
                                <span>Cortes</span>
                                <span class="fw-bold text-danger"><?=$d['cortes']?></span>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

    <!-- Estadísticas detalladas por acción -->
    <h5 class="section-heading"><i class="fa-solid fa-chart-bar"></i>Estadísticas por acción <small class="text-muted ms-2">(<?=$desde_dt->format('Y-m-d')?> a <?=$hasta_dt->format('Y-m-d')?>)</small></h5>
    <div class="row g-3 mb-4">
        <?php if (empty($estadisticas)): ?>
            <div class="col-12"><div class="card p-4 text-center text-muted">Sin estadísticas para los filtros.</div></div>
        <?php endif; ?>
        <?php foreach($estadisticas as $stat): ?>
        <div class="col-12 col-md-6 col-lg-4">
            <div class="card h-100 stats-card">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h6 class="card-title mb-0">
                            <i class="fa-solid fa-chart-line card-title-icon"></i>
                            <?=htmlspecialchars(ucfirst($stat['accion']))?>
                        </h6>
                        <span class="badge badge-role text-capitalize"><?=htmlspecialchars($stat['entidad'])?></span>
                    </div>
                    <div class="row text-center">
                        <div class="col-6">
                            <div class="h4 mb-0 fw-bold text-primary"><?=$stat['total']?></div>
                            <small class="text-muted">Total</small>
                        </div>
                        <div class="col-6">
                            <div class="h4 mb-0 fw-bold text-success"><?=$stat['usuarios_unicos']?></div>
                            <small class="text-muted">Usuarios</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php endforeach; ?>
    </div>

    <!-- Actividad por hora del día -->
    <?php if (!empty($actividad_hora)): ?>
    <h5 class="section-heading"><i class="fa-solid fa-clock"></i>Actividad por hora del día <small class="text-muted ms-2">(<?=$desde_dt->format('Y-m-d')?> a <?=$hasta_dt->format('Y-m-d')?>)</small></h5>
    <div class="row g-3 mb-4">
        <?php foreach($actividad_hora as $hora): ?>
        <div class="col-6 col-md-3 col-lg-2">
            <div class="card text-center hour-card">
                <div class="card-body">
                    <div class="h5 mb-1 fw-bold"><?=sprintf('%02d:00', $hora['hora'])?></div>
                    <div class="text-primary fw-bold"><?=$hora['total_acciones']?></div>
                    <small class="text-muted">acciones</small>
                    <div class="text-success fw-bold mt-1"><?=$hora['usuarios_activos']?></div>
                    <small class="text-muted">usuarios</small>
                </div>
            </div>
        </div>
        <?php endforeach; ?>
    </div>
    <?php endif; ?>

    <h5 class="section-heading">
        <i class="fa-solid fa-clock-rotate-left"></i>Actividad Reciente Detallada
        <small class="text-muted ms-2 fw-normal">(<?=$desde_dt->format('d/m/Y')?> a <?=$hasta_dt->format('d/m/Y')?>)</small>
    </h5>
    
    
    
    <?php if (isset($error_actividad)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-triangle me-2"></i>
            <strong>Error:</strong> <?= htmlspecialchars($error_actividad) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    <div class="row g-4 mb-5">
        <?php if (empty($actividad)): ?>
            <div class="col-12">
                <div class="card p-5 text-center">
                    <div class="text-muted mb-3">
                        <i class="fas fa-inbox fa-3x"></i>
                    </div>
                    <h5 class="text-muted">Sin actividad para los filtros aplicados</h5>
                    <p class="text-muted mb-0">Intenta ajustar los filtros o cambiar el rango de fechas para ver más resultados.</p>
                </div>
            </div>
        <?php endif; ?>
        <?php foreach($actividad as $a):
            $cardClass = '';
            $icon = '';
            $info = '—';
            $accion_class = 'action-'.strtolower($a['accion']);
            
            switch($a['accion']) {
                case 'create':
                    $icon = 'fa-solid fa-sack-dollar';
                    if ($a['entidad'] !== 'perfiles_vendidos') {
                        $icon = 'fa-solid fa-plus-circle';
                    }
                    break;
                case 'update':
                    $icon = 'fa-solid fa-pencil-alt';
                    break;
                case 'delete':
                    $icon = 'fa-solid fa-trash-alt';
                    break;
                case 'renovar':
                case 'renew':
                    $accion_class = 'action-renovar';
                    $icon = 'fa-solid fa-arrows-rotate';
                    break;
                case 'login':
                case 'logout':
                    $icon = 'fa-solid fa-right-to-bracket';
                    break;
                default:
                    $icon = 'fa-solid fa-question-circle';
                    break;
            }

            $precio_html = '';
            if (in_array($a['accion'], ['create', 'renovar', 'renew']) && $a['entidad'] === 'perfiles_vendidos' && !empty($a['precio_venta'])) {
                $precio_html = '<span class="fw-bold text-success ms-2">'.$MON.number_format((float)$a['precio_venta'],2).'</span>';
            }

            // Obtener detalles contextuales
            $detalles = obtenerDetallesAccion($a);
            
            if ($a['entidad']==='perfiles_vendidos') {
                $info = 'Perfil: '.trim(($a['plataforma'] ?: '—').' / '.($a['perfil'] ?: '—').' / '.($a['cliente'] ?: '—'));
            } elseif ($a['entidad']==='plataformas') {
                $info = 'Plataforma: '.($a['plataforma'] ?: '—');
                if ($info==='—' && !empty($a['detalle_json'])) {
                    $j = json_decode($a['detalle_json'], true);
                    if (json_last_error()===JSON_ERROR_NONE && isset($j['nombre'])) $info = 'Plataforma: '.$j['nombre'];
                }
            } else {
                $info = 'Entidad: ' . htmlspecialchars($a['entidad']) . ' (ID: ' . htmlspecialchars((string)$a['entidad_id']) . ')';
            }
        ?>
            <div class="col-12 col-md-6 col-lg-4">
                <div class="card h-100 activity-card <?=$accion_class?>" style="background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);">
                    <div class="card-header activity-card-header">
                        <div class="d-flex justify-content-between align-items-center">
                            <h6 class="card-title mb-0 d-flex align-items-center">
                                <i class="<?=$icon?> card-title-icon"></i>
                                <span class="action-text"><?=htmlspecialchars(traducirAccion($a['accion']))?></span>
                            </h6>
                            <span class="badge badge-role text-capitalize"><?=htmlspecialchars($a['rol'] ?: '—')?></span>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="activity-summary mb-3">
                            <div class="activity-time">
                                <i class="fa-solid fa-clock me-2 text-primary"></i>
                                <span class="fw-bold"><?=htmlspecialchars($a['fecha'])?></span>
                                <?=$precio_html?>
                            </div>
                            <div class="activity-user mt-2">
                                <i class="fa-solid fa-user me-2 text-info"></i>
                                <span class="fw-bold"><?=htmlspecialchars($a['usuario'] ?: '—')?></span>
                                <?php if (!empty($a['username_usuario'])): ?>
                                    <small class="text-muted">(<?=htmlspecialchars($a['username_usuario'])?>)</small>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="activity-details">
                            <div class="detail-item mb-2">
                                <i class="fa-solid fa-info-circle me-2 text-warning"></i>
                                <span class="detail-text"><?=htmlspecialchars($info)?></span>
                            </div>
                            
                            <?php if (!empty($detalles['navegador']) || !empty($detalles['dispositivo'])): ?>
                            <div class="detail-item mb-2">
                                <i class="fa-solid fa-globe me-2 text-success"></i>
                                <span class="detail-text">
                                    <strong>Navegador:</strong> <?=htmlspecialchars($detalles['navegador'] ?: '—')?><br>
                                    <strong>Dispositivo:</strong> <?=htmlspecialchars($detalles['dispositivo'] ?: '—')?>
                                </span>
                            </div>
                            <?php endif; ?>
                            
                            <div class="detail-item mb-2">
                                <i class="fa-solid fa-network-wired me-2 text-secondary"></i>
                                <span class="detail-text"><strong>IP:</strong> <?=htmlspecialchars($a['ip'] ?: '—')?></span>
                            </div>
                            
                            <?php if (!empty($a['detalle_json'])): ?>
                            <div class="detail-item">
                                <button class="btn btn-sm btn-outline-primary w-100" type="button" 
                                        data-bs-toggle="collapse" 
                                        data-bs-target="#json-<?=$a['id_audit']?>" 
                                        aria-expanded="false">
                                    <i class="fa-solid fa-info-circle me-2"></i>Ver Información Detallada
                                </button>
                            </div>
                            <div class="collapse mt-2" id="json-<?=$a['id_audit']?>">
                                <div class="json-details">
                                    <div class="json-header">
                                        <i class="fa-solid fa-list-ul me-2"></i>
                                        <strong>Información Detallada de la Acción</strong>
                                    </div>
                                    <div class="json-content">
                                        <?php 
                                        $datos_entendibles = hacerDatosEntendibles($a['detalle_json']);
                                        if (isset($datos_entendibles['error'])): ?>
                                            <div class="alert alert-warning mb-0">
                                                <i class="fas fa-exclamation-triangle me-2"></i>
                                                <?= htmlspecialchars($datos_entendibles['error']) ?>
                                            </div>
                                        <?php else: ?>
                                            <div class="datos-entendibles">
                                                <?php foreach ($datos_entendibles as $explicacion): ?>
                                                    <div class="dato-item">
                                                        <i class="fas fa-check-circle text-success me-2"></i>
                                                        <?= htmlspecialchars($explicacion) ?>
                                                    </div>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    </div>

</div>

<?php
include_once __DIR__.'/../includes/footer.php';
?>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
// Tooltip para mostrar información adicional
document.addEventListener('DOMContentLoaded', function() {
    // Inicializar tooltips de Bootstrap si están disponibles
    if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
        var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    }
    
    // Botón de actualización manual
    const refreshBtn = document.getElementById('refreshBtn');
    if (refreshBtn) {
        refreshBtn.addEventListener('click', function() {
            this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Actualizando...';
            this.disabled = true;
            setTimeout(() => {
                location.reload();
            }, 1000);
        });
    }
});
</script>