<?php
// FILE: /digitalpremium/pages/egresos.php
// VERSION FINAL (V7 - ULTRA PREMIUM DESIGN & UX REVAMP) POR GEMINI

// Protección de la página y inicio de sesión
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['id_usuario'])) {
    header("Location: /digitalpremium/login.php");
    exit();
}

require_once '../config/db.php';
require_once '../includes/message_helper.php'; // Asegúrate de que esta ruta sea correcta y el archivo exista

// <<< INICIO DE LÓGICA PHP OPTIMIZADA >>>

// 1. OBTENER DATOS DE SESIÓN Y CONFIGURACIÓN
$id_usuario_sesion = $_SESSION['id_usuario'];
$rol_usuario_sesion = $_SESSION['rol'];

// Función getConfigValue se asume que está en message_helper.php o se ha incluido previamente
$moneda_simbolo = getConfigValue('moneda_simbolo', $id_usuario_sesion, 'S/');


// 2. LÓGICA DE FILTROS Y CONSULTA DE EGRESOS
$where_clauses = [];
$params = [];
$filtros_aplicados = false;

// Regla de Oro: Aplicar filtro de usuario si no es Superadmin
if ($rol_usuario_sesion !== 'superadmin') {
    $where_clauses[] = "cm.id_usuario = :id_usuario_sesion";
    $params[':id_usuario_sesion'] = $id_usuario_sesion;
}

// Obtener valores de los filtros desde GET (para persistencia en la URL)
$fecha_desde_eg = $_GET['filtro_fecha_desde_eg'] ?? '';
$fecha_hasta_eg = $_GET['filtro_fecha_hasta_eg'] ?? '';
$filtro_plataforma_eg = $_GET['filtro_plataforma_eg'] ?? '';
$filtro_proveedor_eg = $_GET['filtro_proveedor_eg'] ?? '';

if (!empty($fecha_desde_eg)) {
    $where_clauses[] = "cm.fecha_compra_proveedor >= :fecha_desde_eg";
    $params[':fecha_desde_eg'] = $fecha_desde_eg;
    $filtros_aplicados = true;
}
if (!empty($fecha_hasta_eg)) {
    $where_clauses[] = "cm.fecha_compra_proveedor <= :fecha_hasta_eg";
    $params[':fecha_hasta_eg'] = $fecha_hasta_eg;
    $filtros_aplicados = true;
}
if (!empty($filtro_plataforma_eg)) {
    $where_clauses[] = "cm.id_plataforma = :id_plataforma_eg";
    $params[':id_plataforma_eg'] = $filtro_plataforma_eg;
    $filtros_aplicados = true;
}
if (!empty($filtro_proveedor_eg)) {
    $where_clauses[] = "cm.nombre_proveedor LIKE :nombre_proveedor_eg";
    $params[':nombre_proveedor_eg'] = '%' . trim($filtro_proveedor_eg) . '%';
    $filtros_aplicados = true;
}

// Asegurar que solo se listen egresos válidos (costo > 0 y fecha de compra no nula)
$where_clauses[] = "cm.costo_compra > 0";
$where_clauses[] = "cm.fecha_compra_proveedor IS NOT NULL";

$sql_where = " WHERE " . implode(" AND ", $where_clauses);

// Consulta principal de Egresos
$sql_egresos = "SELECT cm.id_cuenta_maestra, cm.fecha_compra_proveedor, cm.costo_compra, cm.email_cuenta,
                         cm.nombre_proveedor, 
                         p.nombre_plataforma, p.id_plataforma
                FROM cuentas_maestras cm
                JOIN plataformas p ON cm.id_plataforma = p.id_plataforma
                $sql_where
                ORDER BY cm.fecha_compra_proveedor DESC"; // Ordenar por fecha de compra descendente

$stmt_egresos = $pdo->prepare($sql_egresos);
$stmt_egresos->execute($params);
$egresos = $stmt_egresos->fetchAll(PDO::FETCH_ASSOC);

// --- Procesamiento de datos para los Gráficos y KPIs ---
$total_egresos = 0;
$egresos_por_dia_raw = [];
$egresos_por_plataforma = [];

foreach ($egresos as $egreso) {
    $total_egresos += $egreso['costo_compra'];
    
    $fecha = date('Y-m-d', strtotime($egreso['fecha_compra_proveedor']));
    if (!isset($egresos_por_dia_raw[$fecha])) {
        $egresos_por_dia_raw[$fecha] = 0;
    }
    $egresos_por_dia_raw[$fecha] += $egreso['costo_compra'];
    
    $plataforma = $egreso['nombre_plataforma'];
    if (!isset($egresos_por_plataforma[$plataforma])) {
        $egresos_por_plataforma[$plataforma] = 0;
    }
    $egresos_por_plataforma[$plataforma] += $egreso['costo_compra'];
}

$numero_de_compras = count($egresos);
$promedio_por_compra = ($numero_de_compras > 0) ? $total_egresos / $numero_de_compras : 0;

// Asegurar que el rango de fechas para el gráfico sea continuo, incluso si no hay datos todos los días
$egresos_por_dia_completos = [];
if ($numero_de_compras > 0) {
    $all_dates = array_keys($egresos_por_dia_raw);
    $fecha_inicio_grafico = !empty($fecha_desde_eg) ? $fecha_desde_eg : (empty($all_dates) ? date('Y-m-d') : min($all_dates));
    $fecha_fin_grafico = !empty($fecha_hasta_eg) ? $fecha_hasta_eg : (empty($all_dates) ? date('Y-m-d') : max($all_dates));
    
    $period = new DatePeriod(
        new DateTime($fecha_inicio_grafico),
        new DateInterval('P1D'),
        (new DateTime($fecha_fin_grafico))->modify('+1 day')
    );

    foreach ($period as $date) {
        $egresos_por_dia_completos[$date->format('Y-m-d')] = 0;
    }

    foreach ($egresos_por_dia_raw as $fecha_compra => $monto) {
        if (isset($egresos_por_dia_completos[$fecha_compra])) {
            $egresos_por_dia_completos[$fecha_compra] = $monto;
        }
    }
}
ksort($egresos_por_dia_completos); // Ordenar por fecha para el gráfico de línea
arsort($egresos_por_plataforma); // Ordenar plataformas por el egreso más alto


// --- 3. CONSULTAS PARA DROPDOWNS DE FILTROS ---
$plataformas_filtro = $pdo->query("SELECT id_plataforma, nombre_plataforma FROM plataformas WHERE estado = 1 ORDER BY nombre_plataforma ASC")->fetchAll(PDO::FETCH_ASSOC);

$sql_proveedores_filtro = "SELECT DISTINCT nombre_proveedor FROM cuentas_maestras WHERE nombre_proveedor IS NOT NULL AND nombre_proveedor != ''";
$params_proveedores_filtro = [];

if ($rol_usuario_sesion !== 'superadmin') {
    $sql_proveedores_filtro .= " AND id_usuario = :id_usuario_sesion";
    $params_proveedores_filtro[':id_usuario_sesion'] = $id_usuario_sesion;
}
$sql_proveedores_filtro .= " ORDER BY nombre_proveedor ASC";

$stmt_prov_filtro = $pdo->prepare($sql_proveedores_filtro);
$stmt_prov_filtro->execute($params_proveedores_filtro);
$proveedores_filtro = $stmt_prov_filtro->fetchAll(PDO::FETCH_COLUMN);

$page_title = "Dashboard de Egresos - DigitalPremium";
include_once '../includes/header.php'; // Incluye tu header (con Bootstrap, FontAwesome, etc.)

// Función para obtener el color de la plataforma para el badge en la tabla
function getPlatformBadgeColor(string $platform_name): string {
    $platform_name = strtolower($platform_name);
    if (str_contains($platform_name, 'netflix')) return '#E50914';
    if (str_contains($platform_name, 'disney')) return '#1A2A4B';
    if (str_contains($platform_name, 'prime')) return '#00A8E1';
    if (str_contains($platform_name, 'hbo') || str_contains($platform_name, 'max')) return '#6C2B80';
    if (str_contains($platform_name, 'spotify')) return '#1DB954';
    if (str_contains($platform_name, 'youtube')) return '#FF0000';
    if (str_contains($platform_name, 'paramount')) return '#0064FF';
    if (str_contains($platform_name, 'crunchyroll')) return '#F47521';
    if (str_contains($platform_name, 'canva')) return '#00C4CC';
    return '#607d8b'; // Color por defecto
}

?>

<style>
    /* VARIABLES DE COLOR Y DISEÑO (REFINADAS) */
    :root {
        --font-primary: 'Inter', sans-serif;
        --body-bg: #f5f8ff; /* Un azul muy claro, casi blanco */
        --card-bg: #ffffff;
        --card-border: #e0e5f0;
        --text-dark: #212529; /* Casi negro */
        --text-secondary: #607d8b; /* Gris azulado */
        
        --primary-main: #673AB7; /* Morado profundo (primary) */
        --primary-light: #9C27B0; /* Morado más claro */
        --primary-accent: #7E57C2; /* Morado intermedio para acentos */

        /* Colores de Egresos: un rojo/naranja más fuerte */
        --expense-main: #D32F2F; /* Rojo oscuro para Egresos */
        --expense-light: #FFCDD2; /* Rojo claro para fondos suaves */
        --expense-accent: #C62828; /* Rojo más oscuro para acentos */
        
        --danger-main: #F44336; /* Rojo principal (danger) - puede usarse para alertas */
        --danger-light: #FFCDD2; /* Rojo claro para soft bg */
        --danger-accent: #D32F2F; /* Rojo más oscuro para bordes */

        --success-main: #4CAF50; /* Verde principal */
        --success-light: #C8E6C9; /* Verde claro */
        --success-accent: #388E3C; /* Verde oscuro */

        --warning-main: #FFC107; /* Amarillo principal */
        --warning-light: #FFECB3; /* Amarillo claro */
        --warning-accent: #FFA000; /* Naranja-amarillo */

        --info-main: #03A9F4; /* Azul info */
        --info-light: #B3E5FC; /* Azul info claro */

        --shadow-sm: 0 1px 3px rgba(0,0,0,0.06);
        --shadow-md: 0 5px 15px rgba(0,0,0,0.08);
        --shadow-lg: 0 10px 30px rgba(0,0,0,0.12);
        
        --border-radius-sm: 0.5rem;
        --border-radius-md: 0.85rem;
        --border-radius-lg: 1.25rem;
    }

    /* GENERALES */
    body {  
        background-color: var(--body-bg);  
        font-family: var(--font-primary);  
        color: var(--text-dark);  
        line-height: 1.6;
    }

    /* CONTENEDOR PRINCIPAL Y PADDING */
    .container-fluid {
        padding-left: 1rem; /* Más reducido */
        padding-right: 1rem; /* Más reducido */
    }
    @media (max-width: 768px) {
        .container-fluid {
            padding-left: 0.5rem; /* Máxima reducción en móvil */
            padding-right: 0.5rem;
        }
    }

    /* ENCABEZADO DE PÁGINA */
    .page-header {
        margin-top: 1rem; /* Aún más reducido */
        margin-bottom: 1rem; /* Aún más reducido */
    }
    .page-header h1 {  
        font-weight: 800;  
        font-size: 1.8rem;  /* Aún más reducido */
        color: var(--text-dark);
        display: flex;
        align-items: center;
    }
    .page-header h1 .fas {
        font-size: 1.6rem; /* Aún más reducido */
        margin-right: 0.5rem; /* Aún más reducido */
        color: var(--expense-main); /* Icono de egresos en rojo */
    }
    .page-header .btn {
        font-weight: 600;
        border-radius: 50rem;
        padding: 0.5rem 1rem; /* Más reducido */
        font-size: 0.85rem; /* Más reducido */
        transition: all 0.2s ease-in-out;
    }
    .page-header .btn-danger { /* Usamos btn-danger como base para egresos */
        background: linear-gradient(45deg, var(--expense-main), var(--expense-accent));
        border: none;
        box-shadow: var(--shadow-md);
    }
    .page-header .btn-danger:hover {
        transform: translateY(-2px);
        box-shadow: var(--shadow-lg);
        filter: brightness(1.1);
    }
    .page-header .btn-outline-secondary {
        border-color: var(--card-border);
        color: var(--text-secondary);
        box-shadow: var(--shadow-sm);
    }
    .page-header .btn-outline-secondary:hover {
        background-color: var(--body-bg);
        color: var(--primary-accent);
        border-color: var(--primary-accent);
        box-shadow: var(--shadow-md);
    }
    .page-header .dropdown-menu {
        border-radius: var(--border-radius-sm);
        box-shadow: var(--shadow-md);
        border: none;
    }
    .page-header .dropdown-item {
        padding: 0.5rem 0.8rem; /* Más reducido */
        font-size: 0.85rem; /* Más reducido */
    }
    .page-header .dropdown-item .fas {
        margin-right: 0.5rem; /* Más reducido */
    }

    /* TARJETAS DE ESTADÍSTICAS (KPIs) - AÚN MÁS PEQUEÑAS Y ALINEADAS */
    .stat-card-kpi {
        background: var(--card-bg);
        border: none;
        border-radius: var(--border-radius-md);
        box-shadow: var(--shadow-md);
        padding: 0.8rem; /* Más reducido */
        display: flex;
        flex-direction: row; /* Cambiado a fila para alinear icono y texto horizontalmente */
        align-items: center;
        text-align: left; /* Alineación de texto a la izquierda dentro del card */
        transition: all 0.2s ease-in-out;
        position: relative;
        overflow: hidden;
        margin-bottom: 0.8rem; /* Más reducido */
    }
    .stat-card-kpi::before {
        content: '';
        position: absolute;
        top: 0; left: 0; right: 0; bottom: 0;
        background: linear-gradient(135deg, rgba(244, 67, 54, 0.03), transparent 60%); /* Degradado más sutil (rojo) */
        border-radius: var(--border-radius-md);
        z-index: -1;
    }
    .stat-card-kpi:hover {  
        transform: translateY(-2px); 
        box-shadow: var(--shadow-lg);  
    }
    .stat-card-kpi .stat-icon {  
        font-size: 1.3rem; 
        width: 40px; height: 40px; /* Tamaño de icono más pequeño */
        border-radius: 50%;  
        display: inline-flex;  
        align-items: center;  
        justify-content: center;  
        margin-right: 0.8rem; /* Espacio a la derecha del icono */
        flex-shrink: 0;  
        box-shadow: 0 2px 8px rgba(0,0,0,0.1); 
        color: white;  
    }
    /* Colores específicos para los iconos de KPI (de egresos) */
    .stat-card-kpi.total .stat-icon { background: linear-gradient(45deg, var(--expense-main), var(--expense-accent)); } /* Rojo */
    .stat-card-kpi.compras .stat-icon { background: linear-gradient(45deg, var(--text-secondary), #8b9ea8); } /* Gris */
    .stat-card-kpi.promedio .stat-icon { background: linear-gradient(45deg, var(--warning-main), var(--warning-accent)); } /* Amarillo/Naranja */

    .stat-card-kpi .stat-content { /* Contenedor para título y valor */
        flex-grow: 1; /* Permite que el contenido tome el espacio restante */
        text-align: left; /* Asegura alineación a la izquierda */
    }
    .stat-card-kpi .stat-title {
        font-size: 0.8rem; 
        font-weight: 500;
        color: var(--text-secondary);
        margin-bottom: 0.1rem; 
        text-transform: uppercase;
        letter-spacing: 0.4px;
    }
    .stat-card-kpi .stat-value {
        font-size: 1.6rem;  
        font-weight: 800;  
        color: var(--text-dark);
        line-height: 1;  
    }

    /* TARJETAS DE GRÁFICOS Y TABLAS */
    .card.shadow-lg {
        border: none;
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow-lg);
        margin-bottom: 1rem; 
        overflow: hidden;
    }
    .card-header {
        background-color: var(--card-bg);
        border-bottom: 1px solid var(--card-border);
        padding: 0.8rem 1rem; 
        font-weight: 700;
        font-size: 0.95rem; 
        color: var(--text-dark);
        display: flex;
        align-items: center;
    }
    .card-header .fas {
        margin-right: 0.5rem; 
        color: var(--expense-main); /* Iconos de header en rojo para egresos */
    }

    /* GRÁFICOS (CHART.JS) */
    .chart-area, .chart-pie {
        position: relative;
        height: 250px; /* Altura de gráficos ajustada */
        width: 100%;
        margin: auto;
    }
    .card-body canvas {
        width: 100% !important;
        height: 100% !important;
    }

    /* TABLA DE EGRESOS - ESTILOS MUCHO MÁS COMPACTOS */
    .table-responsive {
        border-radius: var(--border-radius-md);  
        overflow-x: auto; /* Permite scroll horizontal si el contenido excede */
        border: 1px solid var(--card-border); 
    }
    .table {
        margin-bottom: 0;  
        border-collapse: separate; 
        border-spacing: 0; 
        width: 100%; 
    }
    .table-hover tbody tr:hover {
        background-color: rgba(244, 67, 54, 0.08); /* Hover sutil (rojo) */
    }
    .table thead {
        background-color: black; 
        color: #FFFFFF; 
        text-transform: uppercase;
        font-size: 0.7em; /* Fuente muy pequeña para el encabezado */
        letter-spacing: 0.1px; 
    }
    .table thead th {
        border-bottom: none;
        padding: 0.5rem 0.6rem; /* Padding mínimo en encabezados */
        font-weight: 700; 
        color: #FFFFFF;
        border-right: 1px solid rgba(255,255,255,0.1); 
    }
    .table thead th:last-child {
        border-right: none; 
    }
    /* Ajuste de anchos para las columnas específicas para que quepan */
    .table thead th:nth-child(1) { width: 15%; } /* Fecha Compra */
    .table thead th:nth-child(2) { width: 20%; } /* Proveedor */
    .table thead th:nth-child(3) { width: 15%; } /* Plataforma */
    .table thead th:nth-child(4) { width: 30%; } /* Cuenta (Email) */
    .table thead th:nth-child(5) { width: 20%; } /* Costo */

    .table tbody td {
        padding: 0.5rem 0.6rem; /* Menos padding en celdas de datos */
        vertical-align: middle;
        font-size: 0.78rem; /* Fuente más pequeña para los datos */
        color: var(--text-dark);
        border-bottom: 1px solid var(--card-border); 
        border-right: 1px solid var(--card-border); 
        white-space: nowrap; 
        overflow: hidden; 
        text-overflow: ellipsis; 
    }
    .table tbody td:last-child {
        border-right: none; 
    }
    .table tbody tr:last-child td {
        border-bottom: none; 
    }
    .table tbody tr:nth-of-type(even) {
        background-color: rgba(0, 0, 0, 0.02); /* Aún más sutil */
    }
    /* Badge de plataforma en la tabla */
    .table tbody .badge {
        font-size: 0.7em; 
        padding: 0.25em 0.5em; 
        border-radius: var(--border-radius-sm);
        font-weight: 700; 
        color: white;
        text-shadow: 0 1px 1px rgba(0,0,0,0.3); 
        box-shadow: var(--shadow-sm); 
    }
    .text-end {
        text-align: end !important; 
    }

    /* MODAL DE FILTROS */
    .modal-content {
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow-lg);
        border: none;
    }
    .modal-header {
        background: linear-gradient(45deg, var(--expense-main), var(--expense-accent)); 
        color: white;
        border-bottom: none;
        padding: 1rem 1.2rem; 
        border-top-left-radius: var(--border-radius-lg);
        border-top-right-radius: var(--border-radius-lg);
    }
    .modal-header .modal-title {
        color: white;
        font-size: 1.2rem; 
        font-weight: 700;
    }
    .modal-header .btn-close {
        filter: invert(1) brightness(2); 
    }
    .modal-body {
        padding: 1.2rem; 
    }
    .modal-body .form-label {
        font-weight: 600;
        color: var(--text-dark);
        margin-bottom: 0.3rem; 
        font-size: 0.85rem; 
    }
    .modal-body .form-control,
    .modal-body .form-select {
        border-radius: var(--border-radius-sm);
        border-color: var(--card-border);
        box-shadow: inset 0 1px 3px rgba(0,0,0,0.05);
        padding: 0.5rem 0.7rem; 
        font-size: 0.85rem; 
    }
    .modal-body .input-group-text {
        background-color: var(--body-bg);
        border-color: var(--card-border);
        color: var(--text-secondary);
        border-radius: var(--border-radius-sm);
    }
    .modal-footer {
        border-top: none;
        padding: 1rem 1.2rem; 
        background-color: var(--body-bg);
        border-bottom-left-radius: var(--border-radius-lg);
        border-bottom-right-radius: var(--border-radius-lg);
    }
    .modal-footer .btn {
        font-weight: 600;
        padding: 0.5rem 1rem; 
        border-radius: 50rem;
        box-shadow: var(--shadow-sm);
        transition: all 0.2s ease-in-out;
    }
    .modal-footer .btn-outline-secondary {
        color: var(--text-secondary);
        border-color: var(--text-secondary);
    }
    .modal-footer .btn-outline-secondary:hover {
        background-color: var(--text-secondary);
        color: white;
    }
    .modal-footer .btn-danger { /* Botón de acción para egresos */
        background: linear-gradient(45deg, var(--expense-main), var(--expense-accent));
        border: none;
        color: white;
    }
    .modal-footer .btn-danger:hover {
        filter: brightness(1.1);
        transform: translateY(-2px);
    }

    /* ==========================================================================
       ESTILOS PARA LAS CARDS DE EGRESOS (DISEÑO IDÉNTICO A INGRESOS)
       ========================================================================== */
    
    .egreso-card {
        background: #ffffff;
        border-radius: 1.25rem;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.12);
        overflow: hidden;
        transition: all 0.3s ease;
        border: 1px solid #e2e8f0;
        height: 100%;
        display: flex;
        flex-direction: column;
    }
    
    .egreso-card:hover {
        transform: translateY(-8px);
        box-shadow: 0 20px 40px rgba(244, 67, 54, 0.15);
        border-color: #fecaca;
    }
    
    .egreso-card-header {
        background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%);
        padding: 1.5rem;
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        border-bottom: 1px solid #fecaca;
    }
    
    .egreso-fecha {
        display: flex;
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .fecha-principal {
        font-size: 1.1rem;
        font-weight: 700;
        color: #dc2626;
        display: flex;
        align-items: center;
    }
    
    .fecha-hora {
        font-size: 0.85rem;
        color: #ef4444;
        font-weight: 500;
        display: flex;
        align-items: center;
    }
    
    .egreso-tipo {
        text-align: right;
    }
    
    .badge.tipo-compra {
        background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%);
        color: white;
        font-size: 0.75rem;
        padding: 0.5rem 1rem;
        border-radius: 20px;
        font-weight: 600;
        text-shadow: 0 1px 2px rgba(0,0,0,0.3);
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }
    
    .egreso-card-body {
        padding: 1.5rem;
        flex-grow: 1;
        display: flex;
        flex-direction: column;
        gap: 1.5rem;
    }
    
    .egreso-monto {
        text-align: center;
        padding: 1rem;
        background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%);
        border-radius: 1rem;
        border: 2px solid #fecaca;
    }
    
    .monto-label {
        font-size: 0.9rem;
        color: #dc2626;
        font-weight: 600;
        margin-bottom: 0.5rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .monto-valor {
        font-size: 1.8rem;
        font-weight: 800;
        color: #b91c1c;
        text-shadow: 0 1px 2px rgba(0,0,0,0.1);
    }
    
    .egreso-info {
        display: flex;
        flex-direction: column;
        gap: 1rem;
    }
    
    .info-section {
        border-bottom: 1px solid #e2e8f0;
        padding-bottom: 1rem;
    }
    
    .info-section:last-child {
        border-bottom: none;
        padding-bottom: 0;
    }
    
    .info-title {
        font-size: 0.85rem;
        font-weight: 700;
        color: #475569;
        margin-bottom: 0.5rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .info-content {
        display: flex;
        flex-direction: column;
        gap: 0.25rem;
    }
    
    .proveedor-nombre {
        font-size: 1rem;
        font-weight: 700;
        color: #1e293b;
        margin-bottom: 0.25rem;
    }
    
    .cuenta-email {
        font-size: 0.9rem;
        color: #1e293b;
        font-weight: 600;
        word-break: break-all;
    }
    
    .detalle-compra {
        font-size: 0.8rem;
        color: #64748b;
        display: flex;
        align-items: center;
    }
    
    .badge.plataforma-badge {
        font-size: 0.8rem;
        padding: 0.5rem 1rem;
        border-radius: 20px;
        font-weight: 600;
        color: white;
        text-shadow: 0 1px 2px rgba(0,0,0,0.3);
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }
    
    /* Responsive para las cards */
    @media (max-width: 1200px) {
        .col-xl-4 {
            flex: 0 0 50%;
            max-width: 50%;
        }
    }
    
    @media (max-width: 768px) {
        .col-lg-6,
        .col-xl-4 {
            flex: 0 0 100%;
            max-width: 100%;
        }
        
        .egreso-card-header {
            flex-direction: column;
            gap: 1rem;
            align-items: flex-start;
        }
        
        .egreso-tipo {
            text-align: left;
        }
        
        .monto-valor {
            font-size: 1.5rem;
        }
    }

</style>

<div class="container-fluid px-lg-5 px-md-4 py-4">
    <header class="page-header d-flex justify-content-between align-items-center">
        <h1><i class="fas fa-chart-line"></i>Dashboard de Egresos</h1>
        <div class="d-flex gap-3">
            <button class="btn btn-danger" type="button" data-bs-toggle="modal" data-bs-target="#modalFiltros">
                <i class="fas fa-filter me-2"></i> Filtrar Reporte
            </button>
            <div class="btn-group">
                <button type="button" class="btn btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">
                    <i class="fas fa-download me-2"></i> Exportar
                </button>
                <ul class="dropdown-menu dropdown-menu-end">
                    <li><a class="dropdown-item" href="#" id="exportExcel"><i class="fas fa-file-excel me-2 text-success"></i>Exportar a Excel</a></li>
                    <li><a class="dropdown-item" href="#" id="exportPdf"><i class="fas fa-file-pdf me-2 text-danger"></i>Exportar a PDF</a></li>
                    <li><hr class="dropdown-divider"></li>
                    <li><a class="dropdown-item" href="#" id="printTable"><i class="fas fa-print me-2 text-primary"></i>Imprimir</a></li>
                </ul>
            </div>
        </div>
    </header>

    <?php if (isset($_GET['mensaje'])): ?>
    <div class="alert alert-<?php echo htmlspecialchars($_GET['tipo'] ?? 'info'); ?> alert-dismissible fade show"><button type="button" class="btn-close" data-bs-dismiss="alert"></button><?php echo htmlspecialchars(urldecode($_GET['mensaje'])); ?></div>
    <?php endif; ?>

    <div class="row g-4 mb-4">
        <div class="col-lg-4 col-md-6">
            <div class="stat-card-kpi total">
                <div class="stat-icon"><i class="fas fa-money-bill-wave"></i></div>
                <div class="stat-content"> <div class="stat-title">Egresos Totales</div>
                    <div class="stat-value"><?php echo htmlspecialchars($moneda_simbolo); ?> <?php echo number_format($total_egresos, 2); ?></div>
                </div>
            </div>
        </div>
        <div class="col-lg-4 col-md-6">
            <div class="stat-card-kpi compras">
                <div class="stat-icon"><i class="fas fa-shopping-cart"></i></div>
                <div class="stat-content"> <div class="stat-title">Número de Compras</div>
                    <div class="stat-value"><?php echo number_format($numero_de_compras); ?></div>
                </div>
            </div>
        </div>
        <div class="col-lg-4 col-md-12">
            <div class="stat-card-kpi promedio">
                <div class="stat-icon"><i class="fas fa-calculator"></i></div>
                <div class="stat-content"> <div class="stat-title">Costo Promedio / Compra</div>
                    <div class="stat-value"><?php echo htmlspecialchars($moneda_simbolo); ?> <?php echo number_format($promedio_por_compra, 2); ?></div>
                </div>
            </div>
        </div>
    </div>

    <div class="row g-4">
        <div class="col-lg-8">
            <div class="card shadow-lg mb-4">
                <div class="card-header">
                    <h6 class="m-0 fw-bold"><i class="fas fa-chart-line me-2 text-danger"></i>Evolución de Egresos por Día</h6>
                </div>
                <div class="card-body">
                    <div class="chart-area" style="height: 250px;"> <canvas id="graficoEgresosLinea"></canvas>
                    </div>
                </div>
            </div>
            <div class="card shadow-lg">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6 class="m-0 fw-bold">
                            <i class="fas fa-cards-blank me-2 text-danger"></i>Detalle de Egresos - Vista de Cards
                            <span class="badge bg-primary ms-2"><?php echo count($egresos); ?> egresos</span>
                        </h6>
                        <?php if ($filtros_aplicados): ?>
                            <div class="filtros-activos">
                                <span class="badge bg-info me-2">
                                    <i class="fas fa-filter me-1"></i>Filtros Activos
                                </span>
                                <small class="text-muted">
                                    <?php 
                                    $filtros_texto = [];
                                    if (!empty($fecha_desde_eg)) $filtros_texto[] = "Desde: " . date('d/m/Y', strtotime($fecha_desde_eg));
                                    if (!empty($fecha_hasta_eg)) $filtros_texto[] = "Hasta: " . date('d/m/Y', strtotime($fecha_hasta_eg));
                                    if (!empty($filtro_plataforma_eg)) $filtros_texto[] = "Plataforma seleccionada";
                                    if (!empty($filtro_proveedor_eg)) $filtros_texto[] = "Proveedor: " . $filtro_proveedor_eg;
                                    echo implode(', ', $filtros_texto);
                                    ?>
                                </small>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="card-body">
                    <div class="row g-4">
                        <?php if (empty($egresos)): ?>
                            <div class="col-12">
                                <div class="text-center py-5">
                                    <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                                    <h5 class="text-muted">No se encontraron egresos</h5>
                                    <p class="text-muted">No hay egresos que coincidan con los filtros aplicados.</p>
                                </div>
                            </div>
                        <?php else: ?>
                            <?php foreach (array_reverse($egresos) as $egreso): ?>
                                <div class="col-lg-6 col-xl-4">
                                    <div class="egreso-card">
                                        <div class="egreso-card-header">
                                            <div class="egreso-fecha">
                                                <div class="fecha-principal">
                                                    <i class="fas fa-calendar-alt me-2"></i>
                                                    <?php echo htmlspecialchars(date("d/m/Y", strtotime($egreso['fecha_compra_proveedor']))); ?>
                                                </div>
                                                <div class="fecha-hora">
                                                    <i class="fas fa-clock me-1"></i>
                                                    <?php echo htmlspecialchars(date("H:i", strtotime($egreso['fecha_compra_proveedor']))); ?>
                                                </div>
                                            </div>
                                            <div class="egreso-tipo">
                                                <span class="badge tipo-compra">
                                                    <i class="fas fa-shopping-cart me-1"></i>Compra
                                                </span>
                                            </div>
                                        </div>
                                        
                                        <div class="egreso-card-body">
                                            <div class="egreso-monto">
                                                <div class="monto-label">Costo de Compra</div>
                                                <div class="monto-valor">
                                                    <?php echo htmlspecialchars($moneda_simbolo); ?> <?php echo number_format($egreso['costo_compra'], 2); ?>
                                                </div>
                                            </div>
                                            
                                            <div class="egreso-info">
                                                <div class="info-section">
                                                    <div class="info-title">
                                                        <i class="fas fa-tv me-2"></i>Plataforma
                                                    </div>
                                                    <div class="info-content">
                                                        <span class="badge plataforma-badge" style="background-color: <?php echo getPlatformBadgeColor($egreso['nombre_plataforma']); ?>;">
                                                            <?php echo htmlspecialchars($egreso['nombre_plataforma']); ?>
                                                        </span>
                                                    </div>
                                                </div>
                                                
                                                <div class="info-section">
                                                    <div class="info-title">
                                                        <i class="fas fa-building me-2"></i>Proveedor
                                                    </div>
                                                    <div class="info-content">
                                                        <div class="proveedor-nombre"><?php echo htmlspecialchars($egreso['nombre_proveedor']); ?></div>
                                                    </div>
                                                </div>
                                                
                                                <div class="info-section">
                                                    <div class="info-title">
                                                        <i class="fas fa-envelope me-2"></i>Cuenta
                                                    </div>
                                                    <div class="info-content">
                                                        <div class="cuenta-email"><?php echo htmlspecialchars($egreso['email_cuenta']); ?></div>
                                                    </div>
                                                </div>
                                                
                                                <div class="info-section">
                                                    <div class="info-title">
                                                        <i class="fas fa-info-circle me-2"></i>Detalles
                                                    </div>
                                                    <div class="info-content">
                                                        <div class="detalle-compra">
                                                            <i class="fas fa-calendar me-1"></i>Comprado el: <?php echo htmlspecialchars(date("d/m/Y", strtotime($egreso['fecha_compra_proveedor']))); ?>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-lg-4">
            <div class="card shadow-lg">
                <div class="card-header">
                    <h6 class="m-0 fw-bold"><i class="fas fa-pie-chart me-2 text-danger"></i>Distribución por Plataforma</h6>
                </div>
                <div class="card-body">
                    <div class="chart-pie pt-4" style="height: 250px;"> <canvas id="graficoPlataformasDona"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="modalFiltros" tabindex="-1" aria-labelledby="modalFiltrosLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="modalFiltrosLabel"><i class="fas fa-filter me-2"></i>Filtrar Reporte de Egresos</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="" method="GET">
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-12">
                            <label class="form-label">Rango de Fechas de Compra:</label>
                            <div class="input-group">
                                <input type="date" name="filtro_fecha_desde_eg" class="form-control" value="<?php echo htmlspecialchars($fecha_desde_eg); ?>">
                                <span class="input-group-text">a</span>
                                <input type="date" name="filtro_fecha_hasta_eg" class="form-control" value="<?php echo htmlspecialchars($fecha_hasta_eg); ?>">
                            </div>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Plataforma:</label>
                            <select name="filtro_plataforma_eg" class="form-select">
                                <option value="">Todas las plataformas</option>
                                <?php foreach ($plataformas_filtro as $plataforma_f): ?>
                                    <option value="<?php echo $plataforma_f['id_plataforma']; ?>" <?php echo ($filtro_plataforma_eg == $plataforma_f['id_plataforma']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($plataforma_f['nombre_plataforma']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Proveedor:</label>
                            <input list="proveedores_datalist" name="filtro_proveedor_eg" class="form-control" value="<?php echo htmlspecialchars($filtro_proveedor_eg); ?>" placeholder="Escriba para buscar proveedor...">
                            <datalist id="proveedores_datalist">
                                <?php foreach ($proveedores_filtro as $prov_nombre): ?>
                                    <option value="<?php echo htmlspecialchars($prov_nombre); ?>">
                                <?php endforeach; ?>
                            </datalist>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <a href="/digitalpremium/pages/egresos.php" class="btn btn-outline-secondary">Limpiar</a>
                    <button type="submit" class="btn btn-danger"><i class="fas fa-check me-1"></i> Aplicar Filtros</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php include_once '../includes/footer.php'; ?>

<script src="https://code.jquery.com/jquery-3.7.0.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap5.min.css">
<script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/dataTables.buttons.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.1.53/pdfmake.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.1.53/vfs_fonts.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.html5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.print.min.js"></script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const monedaSimbolo = '<?php echo $moneda_simbolo; ?>';

    // Gráfico de Línea: Evolución de Egresos
    const ctxLinea = document.getElementById('graficoEgresosLinea')?.getContext('2d');
    if (ctxLinea) {
        new Chart(ctxLinea, {
            type: 'line',
            data: {
                labels: <?php echo json_encode(array_keys($egresos_por_dia_completos)); ?>,
                datasets: [{
                    label: 'Egresos',
                    data: <?php echo json_encode(array_values($egresos_por_dia_completos)); ?>,
                    borderColor: '#F44336', // Usar color danger-main
                    backgroundColor: 'rgba(244, 67, 54, 0.1)', // Fondo más suave para el área
                    fill: true,
                    tension: 0.3,
                    pointRadius: 3, 
                    pointHoverRadius: 5,
                    pointBackgroundColor: '#F44336',
                    pointBorderColor: '#fff',
                }]
            },
            options: {
                responsive: true, 
                maintainAspectRatio: false,
                scales: { 
                    x: {
                        grid: {
                            display: false 
                        },
                        ticks: {
                            color: '#607d8b' 
                        }
                    },
                    y: { 
                        beginAtZero: true, 
                        grid: {
                            color: '#e0e5f0' 
                        },
                        ticks: { 
                            callback: (v) => `${monedaSimbolo} ${v.toLocaleString()}`,
                            color: '#607d8b' 
                        } 
                    } 
                },
                plugins: {
                    legend: { display: false },
                    tooltip: { 
                        callbacks: { 
                            label: (c) => ` ${c.dataset.label}: ${monedaSimbolo} ${c.parsed.y.toLocaleString(undefined, {minimumFractionDigits: 2})}` 
                        },
                        backgroundColor: 'rgba(0,0,0,0.8)', 
                        titleColor: '#fff',
                        bodyColor: '#fff',
                        borderColor: '#F44336', // Borde del tooltip
                        borderWidth: 1,
                        cornerRadius: 4,
                        displayColors: false
                    }
                }
            }
        });
    }

    // Gráfico de Dona: Distribución por Plataforma
    const ctxDona = document.getElementById('graficoPlataformasDona')?.getContext('2d');
    if (ctxDona) {
        new Chart(ctxDona, {
            type: 'doughnut',
            data: {
                labels: <?php echo json_encode(array_keys($egresos_por_plataforma)); ?>,
                datasets: [{
                    data: <?php echo json_encode(array_values($egresos_por_plataforma)); ?>,
                    backgroundColor: [
                        '#F44336', // Danger-main (red)
                        '#FFC107', // Warning-main (yellow)
                        '#4CAF50', // Success-main (green)
                        '#03A9F4', // Info-main (blue)
                        '#673AB7', // Primary-main (purple)
                        '#FF5722', // Deeper Orange
                        '#00BCD4', // Cyan
                        '#E91E63', // Pink
                        '#9E9E9E'  // Grey
                    ],
                    hoverOffset: 8, 
                    borderWidth: 2, 
                    borderColor: '#fff' 
                }]
            },
            options: {
                responsive: true, 
                maintainAspectRatio: false,
                plugins: {
                    legend: { 
                        position: 'bottom',
                        labels: {
                            boxWidth: 12, 
                            padding: 15, 
                            color: '#607d8b' 
                        }
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const label = context.label || '';
                                const value = context.parsed || 0;
                                const total = context.chart.data.datasets[0].data.reduce((a, b) => a + b, 0);
                                const percentage = total > 0 ? ((value / total) * 100).toFixed(2) : 0;
                                return ` ${label}: ${monedaSimbolo} ${value.toLocaleString(undefined, {minimumFractionDigits: 2})} (${percentage}%)`;
                            }
                        },
                        backgroundColor: 'rgba(0,0,0,0.8)',
                        titleColor: '#fff',
                        bodyColor: '#fff',
                        borderColor: '#F44336', // Borde del tooltip
                        borderWidth: 1,
                        cornerRadius: 4,
                        displayColors: true 
                    }
                }
            }
        });
    }

    // Configuración de Tabla (DataTables)
    const tabla = $('#tablaEgresos').DataTable({
        language: { url: '//cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json' },
        order: [[0, 'desc']], 
        dom: 'rtip', 
        responsive: true, // Deja que DataTables maneje la responsividad aquí
        // scrollX: true, // No es necesario si responsive: true está activo y bien configurado
        buttons: [
            { extend: 'excelHtml5', title: 'Reporte_Egresos', filename: 'reporte_egresos' },
            { extend: 'pdfHtml5', title: 'Reporte_Egresos', filename: 'reporte_egresos', orientation: 'landscape', pageSize: 'A4' },
            { extend: 'print', title: 'Reporte de Egresos' }
        ]
    });
    
    // Asignar funciones a los botones de exportación/impresión
    $('#exportExcel').on('click', (e) => { e.preventDefault(); tabla.button('.buttons-excel').trigger(); });
    $('#exportPdf').on('click', (e) => { e.preventDefault(); tabla.button('.buttons-pdf').trigger(); });
    $('#printTable').on('click', (e) => { e.preventDefault(); tabla.button('.buttons-print').trigger(); });
});
</script>