<?php
// FILE: /digitalpremium/pages/importar_excel.php

// Iniciar sesión si no está iniciada
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Redirigir si el usuario no está logueado
if (!isset($_SESSION['id_usuario'])) {
    header("Location: /digitalpremium/login.php");
    exit();
}

// Incluir archivos de configuración y helpers de la base de datos
require_once '../config/db.php';
require_once '../includes/message_helper.php'; // Se asume que aquí están getConfigValue, limpiarNumeroWhatsApp, etc.

// Cargar el autoloader de Composer para PhpSpreadsheet
require_once '../vendor/autoload.php';

// Usar las clases de PhpSpreadsheet
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Shared\Date; // Para manejar fechas de Excel

$page_title = "Importar Datos Excel - DigitalPremium";

// Incluir la cabecera HTML (asume que contiene <head>, CSS, etc.)
include_once '../includes/header.php'; 

$message = ''; // Mensaje a mostrar al usuario
$message_type = ''; // Tipo de mensaje: 'success', 'danger', 'warning', 'info'

// Asegurarse de que getConfigValue exista, si no está ya en message_helper.php
if (!function_exists('getConfigValue')) {
    function getConfigValue($key, $default = null) {
        global $pdo; // Necesita acceso a la conexión PDO
        // Obtener el ID de usuario de la sesión para la lógica multiusuario
        $id_usuario_sesion_for_config = $_SESSION['id_usuario'] ?? null; 
        try {
            // Prioridad: configuración específica del usuario
            if ($id_usuario_sesion_for_config !== null) {
                $stmt = $pdo->prepare("SELECT valor_config FROM configuraciones WHERE clave_config = :key AND id_usuario = :id_usuario");
                $stmt->execute([':key' => $key, ':id_usuario' => $id_usuario_sesion_for_config]);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($result) {
                    return $result['valor_config'];
                }
            }
            // Si no hay configuración de usuario o no se encontró, buscar la configuración global (id_usuario IS NULL)
            $stmt = $pdo->prepare("SELECT valor_config FROM configuraciones WHERE clave_config = :key AND id_usuario IS NULL");
            $stmt->execute([':key' => $key]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result ? $result['valor_config'] : $default;
        } catch (PDOException $e) {
            error_log("Error al obtener config value {$key}: " . $e->getMessage());
            return $default;
        }
    }
}

// --- FUNCIÓN ADICIONAL: safeTrim para manejo robusto de valores ---
// Esta función garantiza que trim() siempre reciba una cadena y devuelve null si el resultado es vacío.
if (!function_exists('safeTrim')) {
    function safeTrim($value) {
        // Convierte a cadena de forma segura. Si $value es null, se convierte a "".
        $stringValue = (string)$value;
        $trimmedValue = trim($stringValue);
        // Si la cadena recortada está vacía, devuelve null, de lo contrario devuelve la cadena recortada.
        return ($trimmedValue === '') ? null : $trimmedValue;
    }
}

// --- FUNCIÓN ADICIONAL: convertExcelDate mejorada ---
// Convierte un valor numérico de Excel a una fecha en formato 'YYYY-MM-DD'.
// Si la conversión falla o el valor no es válido, retorna la fecha actual (o un valor predeterminado si se especifica).
if (!function_exists('convertExcelDate')) {
    function convertExcelDate($excelDate, $defaultDate = 'now') {
        if (is_numeric($excelDate) && $excelDate > 0) {
            try {
                $phpDate = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($excelDate);
                return $phpDate->format('Y-m-d');
            } catch (Exception $e) {
                error_log("Error al convertir fecha de Excel '{$excelDate}': " . $e->getMessage());
                // Si hay un error de conversión, usa la fecha por defecto
                return ($defaultDate === 'now') ? date('Y-m-d') : $defaultDate;
            }
        }
        // Si no es un número de fecha de Excel válido, usa la fecha por defecto
        return ($defaultDate === 'now') ? date('Y-m-d') : $defaultDate;
    }
}

$id_usuario_sesion = $_SESSION['id_usuario']; // Asegurarse de tener el id_usuario de la sesión
$rol_usuario_sesion = $_SESSION['rol']; // Asegurarse de tener el rol de la sesión

// Obtener el símbolo de moneda de la configuración para limpiar los valores de costo
$moneda_simbolo = getConfigValue('moneda_simbolo', 'S/'); 

// Procesar la subida del archivo cuando se envía el formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['excel_file'])) {
    $file_info = $_FILES['excel_file'];

    // --- Validación básica del archivo subido ---
    if ($file_info['error'] !== UPLOAD_ERR_OK) {
        $message = "Error al subir el archivo. Código de error: " . $file_info['error'] . ". Posibles causas: tamaño de archivo excedido, archivo corrupto, etc.";
        $message_type = "danger";
    } elseif (!in_array($file_info['type'], ['application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'application/vnd.ms-excel'])) {
        $message = "Tipo de archivo no permitido. Sube un archivo .xlsx (Excel 2007+) o .xls (Excel 97-2003).";
        $message_type = "danger";
    } else {
        try {
            // Cargar el archivo Excel
            $spreadsheet = IOFactory::load($file_info['tmp_name']);
            $sheet = $spreadsheet->getActiveSheet(); // Obtener la hoja activa
            $highestRow = $sheet->getHighestRow(); // Última fila con datos
            $highestColumn = $sheet->getHighestColumn(); // Última columna con datos

            $total_rows_processed = 0;
            $rows_inserted_updated_cm = 0; // Cuentas Maestras procesadas
            $rows_inserted_updated_pv = 0; // Perfiles Vendidos procesados
            $rows_skipped = 0; // Filas completamente saltadas por errores
            $errors = []; // Para almacenar mensajes de error específicos por fila

            // Obtener los encabezados de la primera fila y mapearlos a sus índices de columna
            $header_row_data = $sheet->rangeToArray('A1:' . $highestColumn . '1', NULL, TRUE, FALSE)[0];
            $header_mapping = [];
            foreach ($header_row_data as $col_index => $col_name) {
                // Usar safeTrim() para asegurar que $col_name sea una cadena para trim()
                $header_mapping[safeTrim($col_name)] = $col_index; // Almacena "Nombre de Columna" => Índice
            }

            // Definir las columnas esenciales que DEBEN existir en el Excel
            // Si alguna de estas no se encuentra, se detendrá la importación
            $essential_cols_master = [
                'Plataforma', 'Email Cuenta', 'Password Cuenta', 'Fecha Vencimiento Proveedor (YYYY-MM-DD)'
            ];
            
            $missing_cols = [];
            foreach ($essential_cols_master as $col) {
                if (!isset($header_mapping[$col])) {
                    $missing_cols[] = $col;
                }
            }

            if (!empty($missing_cols)) {
                // Si faltan columnas esenciales, detener la importación y mostrar error
                throw new Exception("Columnas esenciales de Cuenta Maestra faltantes en el archivo Excel: " . implode(", ", $missing_cols) . ". <br>Asegúrate de que los encabezados coincidan exactamente con la plantilla.");
            }

            // Mapear los índices de columna para cada campo esperado (null si la columna es opcional y no existe)
            $col_platform_name = $header_mapping['Plataforma'];
            $col_email_cuenta = $header_mapping['Email Cuenta'];
            $col_password_cuenta = $header_mapping['Password Cuenta'];
            $col_nombre_proveedor = $header_mapping['Proveedor'] ?? null;
            $col_fecha_compra_proveedor = $header_mapping['Fecha Compra Proveedor (YYYY-MM-DD)'] ?? null;
            $col_fecha_vencimiento_proveedor = $header_mapping['Fecha Vencimiento Proveedor (YYYY-MM-DD)']; // Esencial

            $col_costo_compra = $header_mapping['Costo Compra'] ?? null;
            $col_perfiles_total = $header_mapping['Perfiles Total'] ?? null;
            $col_perfiles_disponibles = $header_mapping['Perfiles Disponibles'] ?? null;
            $col_notas_cuenta = $header_mapping['Notas Cuenta'] ?? null;
            $col_estado_cuenta = $header_mapping['Estado Cuenta (1=Activo, 0=Inactivo, 2=Problemas)'] ?? null;

            // Headers para perfiles vendidos (usar safeTrim para manejar valores nulos directamente desde $rowData)
            $col_cliente_nombre = $header_mapping['Nombre Cliente'] ?? null;
            $col_cliente_telefono = $header_mapping['Telefono Cliente'] ?? null;
            $col_cliente_email = $header_mapping['Email Cliente'] ?? null; // Añadido para el email del cliente
            $col_cliente_notas = $header_mapping['Notas Cliente'] ?? null; // Añadido para las notas del cliente

            $col_perfil_nombre = $header_mapping['Nombre Perfil Cliente'] ?? null;
            $col_perfil_pin = $header_mapping['PIN Perfil Cliente'] ?? null;
            $col_fecha_inicio_servicio = $header_mapping['Fecha Inicio Servicio (YYYY-MM-DD)'] ?? null;
            $col_fecha_fin_servicio = $header_mapping['Fecha Fin Servicio (YYYY-MM-DD)'] ?? null; // Esencial si hay perfil
            $col_precio_venta = $header_mapping['Precio Venta Perfil'] ?? null;
            $col_estado_suscripcion = $header_mapping['Estado Suscripcion Perfil (1=Activa, 0=Vencida, 2=Cancelada)'] ?? null;
            $col_notas_suscripcion = $header_mapping['Notas Suscripcion Perfil'] ?? null;

            // Iniciar transacción de base de datos para asegurar atomicidad
            $pdo->beginTransaction();

            // Iterar sobre cada fila de datos (empezando desde la segunda fila)
            for ($row_num = 2; $row_num <= $highestRow; $row_num++) {
                $rowData = $sheet->rangeToArray('A' . $row_num . ':' . $highestColumn . $row_num, NULL, TRUE, FALSE)[0];
                $total_rows_processed++;

                // --- Extraer y validar datos de la Cuenta Maestra con safeTrim() ---
                $platform_name = safeTrim($rowData[$col_platform_name] ?? null);
                $email_cuenta = safeTrim($rowData[$col_email_cuenta] ?? null);
                $password_cuenta = safeTrim($rowData[$col_password_cuenta] ?? null);
                $fecha_vencimiento_proveedor_excel_val = safeTrim($rowData[$col_fecha_vencimiento_proveedor] ?? null);

                // Validar que los campos esenciales de la Cuenta Maestra no estén vacíos en la fila actual
                if (empty($platform_name) || empty($email_cuenta) || empty($password_cuenta) || empty($fecha_vencimiento_proveedor_excel_val)) {
                    $errors[] = "Fila " . $row_num . ": Datos esenciales de Cuenta Maestra incompletos (Plataforma, Email Cuenta, Password Cuenta o Fecha Vencimiento Proveedor). Saltando procesamiento de esta fila.";
                    $rows_skipped++;
                    continue; // Saltar a la siguiente fila del Excel
                }

                // Obtener el ID de la plataforma (debe existir en la tabla 'plataformas')
                $stmt_platform = $pdo->prepare("SELECT id_plataforma FROM plataformas WHERE nombre_plataforma = :nombre_plataforma AND estado = 1");
                $stmt_platform->execute([':nombre_plataforma' => $platform_name]);
                $platform_id = $stmt_platform->fetchColumn();

                if (!$platform_id) {
                    $errors[] = "Fila " . $row_num . ": Plataforma '" . htmlspecialchars($platform_name) . "' no encontrada o inactiva en el sistema. Asegúrate de crearla primero. Saltando cuenta maestra.";
                    $rows_skipped++;
                    continue;
                }

                // Convertir fecha de vencimiento del proveedor (usa convertExcelDate que maneja nulls y defaults)
                $fecha_vencimiento_proveedor_db = convertExcelDate($fecha_vencimiento_proveedor_excel_val);
                
                // Convertir fecha de compra del proveedor (opcional)
                $fecha_compra_proveedor_db = null;
                if ($col_fecha_compra_proveedor !== null) {
                    $fecha_compra_proveedor_excel_val = safeTrim($rowData[$col_fecha_compra_proveedor] ?? null);
                    if ($fecha_compra_proveedor_excel_val !== null) {
                       $fecha_compra_proveedor_db = convertExcelDate($fecha_compra_proveedor_excel_val, null); // Permite null si no se puede convertir
                    }
                }
                
                // Limpiar y obtener otros datos de la Cuenta Maestra
                $nombre_proveedor_val = ($col_nombre_proveedor !== null) ? safeTrim($rowData[$col_nombre_proveedor]) : NULL;
                // Eliminar símbolo de moneda y comas para el costo
                $costo_compra_val = 0.00;
                if ($col_costo_compra !== null) {
                    $raw_costo = safeTrim($rowData[$col_costo_compra]);
                    if ($raw_costo !== null) {
                        $costo_compra_val = floatval(str_replace([$moneda_simbolo, ',', ' '], '', $raw_costo));
                    }
                }
                
                $perfiles_total_val = ($col_perfiles_total !== null) ? intval($rowData[$col_perfiles_total] ?? 1) : 1;
                $perfiles_disponibles_val = ($col_perfiles_disponibles !== null) ? intval($rowData[$col_perfiles_disponibles] ?? 1) : 1;
                $notas_cuenta_val = ($col_notas_cuenta !== null) ? safeTrim($rowData[$col_notas_cuenta]) : NULL;
                
                $estado_cuenta_val = 1; // Valor predeterminado
                if ($col_estado_cuenta !== null) {
                    $raw_estado_cuenta = safeTrim($rowData[$col_estado_cuenta]);
                    if ($raw_estado_cuenta !== null && in_array($raw_estado_cuenta, ['0', '1', '2'], true)) { 
                        $estado_cuenta_val = intval($raw_estado_cuenta);
                    }
                }

                // Buscar si la Cuenta Maestra ya existe por Email, Plataforma Y USUARIO
                $stmt_check_cm = $pdo->prepare("SELECT id_cuenta_maestra FROM cuentas_maestras WHERE email_cuenta = :email_cuenta AND id_plataforma = :id_plataforma AND id_usuario = :id_usuario");
                $stmt_check_cm->execute([
                    ':email_cuenta' => $email_cuenta, 
                    ':id_plataforma' => $platform_id, 
                    ':id_usuario' => $id_usuario_sesion // ¡Importante para multiusuario!
                ]);
                $existing_master_account_id = $stmt_check_cm->fetchColumn();

                if ($existing_master_account_id) {
                    // Actualizar Cuenta Maestra existente
                    $stmt_update_cm = $pdo->prepare("
                        UPDATE cuentas_maestras SET
                            password_cuenta = :password_cuenta,
                            nombre_proveedor = :nombre_proveedor,
                            fecha_compra_proveedor = :fecha_compra_proveedor,
                            fecha_vencimiento_proveedor = :fecha_vencimiento_proveedor,
                            costo_compra = :costo_compra,
                            perfiles_total = :perfiles_total,
                            perfiles_disponibles = :perfiles_disponibles,
                            notas_cuenta = :notas_cuenta,
                            estado_cuenta = :estado_cuenta
                        WHERE id_cuenta_maestra = :id_cuenta_maestra AND id_usuario = :id_usuario
                    ");
                    $stmt_update_cm->execute([
                        ':password_cuenta' => password_hash($password_cuenta, PASSWORD_DEFAULT), // Siempre hashear el password
                        ':nombre_proveedor' => $nombre_proveedor_val,
                        ':fecha_compra_proveedor' => $fecha_compra_proveedor_db,
                        ':fecha_vencimiento_proveedor' => $fecha_vencimiento_proveedor_db,
                        ':costo_compra' => $costo_compra_val,
                        ':perfiles_total' => $perfiles_total_val,
                        ':perfiles_disponibles' => $perfiles_disponibles_val,
                        ':notas_cuenta' => $notas_cuenta_val,
                        ':estado_cuenta' => $estado_cuenta_val,
                        ':id_cuenta_maestra' => $existing_master_account_id,
                        ':id_usuario' => $id_usuario_sesion // Incluir id_usuario en el WHERE
                    ]);
                    $master_account_id = $existing_master_account_id;
                    $rows_inserted_updated_cm++; 
                } else {
                    // Insertar nueva Cuenta Maestra
                    // FIX: Añadir id_usuario a la lista de columnas y valores en el INSERT
                    $stmt_insert_cm = $pdo->prepare("
                        INSERT INTO cuentas_maestras (
                            id_usuario, id_plataforma, email_cuenta, password_cuenta, nombre_proveedor,
                            fecha_compra_proveedor, fecha_vencimiento_proveedor, costo_compra,
                            perfiles_total, perfiles_disponibles, notas_cuenta, estado_cuenta
                        ) VALUES (
                            :id_usuario, :id_plataforma, :email_cuenta, :password_cuenta, :nombre_proveedor,
                            :fecha_compra_proveedor, :fecha_vencimiento_proveedor, :costo_compra,
                            :perfiles_total, :perfiles_disponibles, :notas_cuenta, :estado_cuenta
                        )
                    ");
                    $stmt_insert_cm->execute([
                        ':id_usuario' => $id_usuario_sesion, // Valor para la nueva columna id_usuario
                        ':id_plataforma' => $platform_id,
                        ':email_cuenta' => $email_cuenta,
                        ':password_cuenta' => password_hash($password_cuenta, PASSWORD_DEFAULT), // Siempre hashear el password
                        ':nombre_proveedor' => $nombre_proveedor_val,
                        ':fecha_compra_proveedor' => $fecha_compra_proveedor_db,
                        ':fecha_vencimiento_proveedor' => $fecha_vencimiento_proveedor_db,
                        ':costo_compra' => $costo_compra_val,
                        ':perfiles_total' => $perfiles_total_val,
                        ':perfiles_disponibles' => $perfiles_disponibles_val,
                        ':notas_cuenta' => $notas_cuenta_val,
                        ':estado_cuenta' => $estado_cuenta_val
                    ]);
                    $master_account_id = $pdo->lastInsertId();
                    $rows_inserted_updated_cm++;
                }

                // --- Procesar Perfil Vendido (Opcional, si está en la misma fila) ---
                $client_name = ($col_cliente_nombre !== null) ? safeTrim($rowData[$col_cliente_nombre] ?? null) : null;
                $perfil_nombre = ($col_perfil_nombre !== null) ? safeTrim($rowData[$col_perfil_nombre] ?? null) : null;

                // Solo intentar procesar el perfil vendido si hay datos esenciales del perfil y cliente
                // y la columna 'Fecha Fin Servicio' existe en el Excel
                if (!empty($client_name) && !empty($perfil_nombre) && $col_fecha_fin_servicio !== null) {
                    $fecha_fin_servicio_excel_val = safeTrim($rowData[$col_fecha_fin_servicio] ?? null);

                    if (empty($fecha_fin_servicio_excel_val)) {
                        $errors[] = "Fila " . $row_num . ": 'Fecha Fin Servicio (YYYY-MM-DD)' es esencial para un perfil de cliente y está vacía. Saltando perfil de cliente.";
                        continue; // Saltar a la siguiente fila del Excel para el perfil, pero la CM ya fue procesada
                    }

                    // --- Extracción y procesamiento de datos del cliente ---
                    $client_phone = ($col_cliente_telefono !== null) ? safeTrim($rowData[$col_cliente_telefono] ?? null) : null;
                    $client_email = ($col_cliente_email !== null) ? safeTrim($rowData[$col_cliente_email] ?? null) : null;
                    $client_notas = ($col_cliente_notas !== null) ? safeTrim($rowData[$col_cliente_notas] ?? null) : null;
                    
                    // Obtener o crear Cliente (usando 'nombre_completo' y adaptando la búsqueda)
                    // FIX: Incluir id_usuario en la búsqueda de clientes
                    $stmt_client = $pdo->prepare("SELECT id_cliente FROM clientes WHERE id_usuario = :id_usuario AND nombre_completo = :nombre_completo AND (telefono = :telefono OR :telefono IS NULL) AND (email = :email OR :email IS NULL)");
                    $stmt_client->execute([
                        ':id_usuario' => $id_usuario_sesion, // Importante para multiusuario
                        ':nombre_completo' => $client_name, 
                        ':telefono' => $client_phone,
                        ':email' => $client_email
                    ]);
                    $client_id = $stmt_client->fetchColumn();

                    if (!$client_id) {
                        // FIX: Columnas reales de la tabla 'clientes' son 'id_usuario', 'nombre_completo', 'telefono', 'email', 'notas_cliente'
                        $stmt_insert_client = $pdo->prepare("INSERT INTO clientes (id_usuario, nombre_completo, telefono, email, notas_cliente) VALUES (:id_usuario, :nombre_completo, :telefono, :email, :notas_cliente)");
                        $stmt_insert_client->execute([
                            ':id_usuario' => $id_usuario_sesion, // Valor para la nueva columna id_usuario
                            ':nombre_completo' => $client_name,
                            ':telefono' => $client_phone,
                            ':email' => $client_email,
                            ':notas_cliente' => $client_notas
                        ]);
                        $client_id = $pdo->lastInsertId();
                    }

                    // --- Extracción y procesamiento de datos del perfil vendido ---
                    $perfil_pin_val = ($col_perfil_pin !== null) ? safeTrim($rowData[$col_perfil_pin] ?? null) : null;
                    
                    // Convertir fecha de inicio del servicio (opcional, con manejo de errores y default a hoy si es NOT NULL)
                    $fecha_inicio_servicio_db = date('Y-m-d'); // Default seguro
                    if ($col_fecha_inicio_servicio !== null) {
                        $fecha_inicio_servicio_excel_val = safeTrim($rowData[$col_fecha_inicio_servicio] ?? null);
                        if ($fecha_inicio_servicio_excel_val !== null) {
                           $fecha_inicio_servicio_db = convertExcelDate($fecha_inicio_servicio_excel_val);
                        }
                    }
                    
                    // Convertir fecha de fin del servicio (esencial para perfil, NOT NULL en BD)
                    $fecha_fin_servicio_db = convertExcelDate($fecha_fin_servicio_excel_val); // Ya validada como no vacía
                    
                    $precio_venta_val = 0.00;
                    if ($col_precio_venta !== null) {
                        $raw_precio_venta = safeTrim($rowData[$col_precio_venta]);
                        if ($raw_precio_venta !== null) {
                            $precio_venta_val = floatval(str_replace([$moneda_simbolo, ',', ' '], '', $raw_precio_venta));
                        }
                    }
                    
                    $estado_suscripcion_val = 1; // Default
                    if ($col_estado_suscripcion !== null) {
                        $raw_estado_suscripcion = safeTrim($rowData[$col_estado_suscripcion]);
                        if ($raw_estado_suscripcion !== null && in_array($raw_estado_suscripcion, ['0', '1', '2'], true)) {
                            $estado_suscripcion_val = intval($raw_estado_suscripcion);
                        }
                    }

                    $notas_suscripcion_val = ($col_notas_suscripcion !== null) ? safeTrim($rowData[$col_notas_suscripcion]) : NULL;

                    // Buscar si el Perfil Vendido ya existe para esta Cuenta Maestra, Cliente y Nombre de Perfil
                    // FIX: Incluir id_usuario en la búsqueda de perfiles vendidos
                    $stmt_check_pv = $pdo->prepare("SELECT id_perfil_vendido FROM perfiles_vendidos WHERE id_cuenta_maestra = :id_cuenta_maestra AND id_cliente = :id_cliente AND nombre_perfil_cliente = :nombre_perfil_cliente AND id_usuario = :id_usuario");
                    $stmt_check_pv->execute([
                        ':id_cuenta_maestra' => $master_account_id,
                        ':id_cliente' => $client_id,
                        ':nombre_perfil_cliente' => $perfil_nombre, // Corregido para usar el nombre de columna de la BD
                        ':id_usuario' => $id_usuario_sesion // ¡Importante para multiusuario!
                    ]);
                    $existing_sold_profile_id = $stmt_check_pv->fetchColumn();

                    if ($existing_sold_profile_id) {
                        // Actualizar Perfil Vendido existente
                        // FIX: Nombres de columnas y parámetros a los de la BD
                        $stmt_update_pv = $pdo->prepare("
                            UPDATE perfiles_vendidos SET 
                                nombre_perfil_cliente = :nombre_perfil_cliente, 
                                pin_perfil_cliente = :pin_perfil_cliente, 
                                fecha_inicio_servicio = :fecha_inicio_servicio, 
                                fecha_fin_servicio = :fecha_fin_servicio, 
                                precio_venta = :precio_venta, 
                                estado_suscripcion = :estado_suscripcion, 
                                notas_suscripcion = :notas_suscripcion 
                            WHERE id_perfil_vendido = :id_perfil_vendido AND id_usuario = :id_usuario
                        ");
                        // FIX: Nombres de parámetros para execute() deben coincidir
                        $stmt_update_pv->execute([
                            ':nombre_perfil_cliente' => $perfil_nombre,
                            ':pin_perfil_cliente' => $perfil_pin_val,
                            ':fecha_inicio_servicio' => $fecha_inicio_servicio_db,
                            ':fecha_fin_servicio' => $fecha_fin_servicio_db,
                            ':precio_venta' => $precio_venta_val,
                            ':estado_suscripcion' => $estado_suscripcion_val,
                            ':notas_suscripcion' => $notas_suscripcion_val,
                            ':id_perfil_vendido' => $existing_sold_profile_id,
                            ':id_usuario' => $id_usuario_sesion // ¡Importante para multiusuario!
                        ]);
                        $rows_inserted_updated_pv++;
                    } else {
                        // Insertar nuevo Perfil Vendido
                        // FIX: Nombres de columnas y parámetros a los de la BD, añadir id_usuario
                        $stmt_insert_pv = $pdo->prepare("
                            INSERT INTO perfiles_vendidos (
                                id_usuario, id_cliente, id_cuenta_maestra, 
                                nombre_perfil_cliente, pin_perfil_cliente, 
                                fecha_inicio_servicio, fecha_fin_servicio, 
                                precio_venta, estado_suscripcion, notas_suscripcion
                            ) VALUES (
                                :id_usuario, :id_cliente, :id_cuenta_maestra, 
                                :nombre_perfil_cliente, :pin_perfil_cliente, 
                                :fecha_inicio_servicio, :fecha_fin_servicio, 
                                :precio_venta, :estado_suscripcion, :notas_suscripcion
                            )
                        ");
                        // FIX: Nombres de parámetros para execute() deben coincidir
                        $stmt_insert_pv->execute([
                            ':id_usuario' => $id_usuario_sesion, // Valor para la nueva columna id_usuario
                            ':id_cliente' => $client_id,
                            ':id_cuenta_maestra' => $master_account_id,
                            ':nombre_perfil_cliente' => $perfil_nombre,
                            ':pin_perfil_cliente' => $perfil_pin_val,
                            ':fecha_inicio_servicio' => $fecha_inicio_servicio_db,
                            ':fecha_fin_servicio' => $fecha_fin_servicio_db,
                            ':precio_venta' => $precio_venta_val,
                            ':estado_suscripcion' => $estado_suscripcion_val,
                            ':notas_suscripcion' => $notas_suscripcion_val
                        ]);
                        $rows_inserted_updated_pv++;
                    }
                }
            } // Fin del bucle for ($row_num)

            // Si todo fue bien, confirmar la transacción
            $pdo->commit();
            $message = "Importación completada con éxito. Total filas procesadas: {$total_rows_processed}. Cuentas Maestras insertadas/actualizadas: {$rows_inserted_updated_cm}. Perfiles vendidos insertados/actualizados: {$rows_inserted_updated_pv}. Filas saltadas: {$rows_skipped}.";
            $message_type = "success";
            if (!empty($errors)) {
                $message .= "<br><b>Advertencias / Errores de fila:</b><br>" . implode("<br>", $errors);
                $message_type = "warning"; // Cambiar a warning si hay errores de fila
            }

        } catch (Exception $e) {
            // Si algo falla, revertir todos los cambios en la BD
            $pdo->rollBack();
            $message = "Error crítico durante la importación: " . $e->getMessage() . ". No se realizó ningún cambio en la base de datos.";
            $message_type = "danger";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH" crossorigin="anonymous">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" integrity="sha512-SnH5WK+bZxgPHs44uWIX+LLJAJ9/2PkPKZ5QiAj6Ta86w+fsb2TkcmfRyVX3pBnMFcV7oQPJkl9QevSCWr3W6A==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-color: #5d5edd; 
            --secondary-color: #7f8a96; 
            --accent-success: #5cb85c; 
            --accent-info: #5bc0de;    
            --accent-warning: #f0ad4e; 
            --accent-danger: #d9534f;  
            --dark-text: #343a40;   
            --light-bg: #f5f8fa;    
            --card-bg: #ffffff; 
            --border-color: #dee2e6; 
            --shadow-light: rgba(0, 0, 0, 0.05);
            --shadow-medium: rgba(0, 0, 0, 0.1);
            --shadow-hover: rgba(0, 0, 0, 0.15);
            --card-border-radius: 0.75rem; 
            --transition-speed: 0.2s; 
        }

        body {
            background-color: var(--light-bg);
            font-family: 'Inter', sans-serif;
            color: var(--dark-text);
        }

        .container {
            max-width: 900px; /* Increased max-width for better layout on larger screens */
            margin-top: 50px;
            margin-bottom: 50px;
        }
        .card {
            border-radius: var(--card-border-radius);
            box-shadow: 0 0.8rem 1.6rem var(--shadow-medium);
        }
        .card-header {
            background-color: var(--primary-color) !important; /* Use primary color for header */
            color: white;
            font-weight: 700;
            font-size: 1.5rem;
            padding: 1.5rem;
            border-top-left-radius: var(--card-border-radius) !important;
            border-top-right-radius: var(--card-border-radius) !important;
        }
        .card-body {
            padding: 2.5rem;
        }
        .form-label {
            font-weight: 600;
            color: var(--dark-text);
        }
        .form-control {
            border-radius: 0.5rem;
            border-color: var(--border-color);
        }
        .btn-lg {
            padding: 0.8rem 1.8rem;
            font-size: 1.1rem;
            border-radius: 0.6rem;
            font-weight: 600;
        }
        .btn-outline-success {
            border-color: var(--accent-success);
            color: var(--accent-success);
            transition: all var(--transition-speed) ease-in-out;
        }
        .btn-outline-success:hover {
            background-color: var(--accent-success);
            color: white;
            transform: translateY(-2px);
            box-shadow: 0 0.25rem 0.5rem rgba(0,0,0,0.1);
        }
        .lead {
            color: var(--secondary-color);
        }
        hr {
            border-top: 1px dashed var(--border-color);
        }

        /* Responsive adjustments for import page */
        @media (max-width: 767.98px) {
            .container {
                padding: 0 1rem;
            }
            .card-header {
                font-size: 1.2rem;
                padding: 1rem;
            }
            .card-body {
                padding: 1.5rem;
            }
            .btn-lg {
                font-size: 1rem;
                padding: 0.6rem 1.2rem;
            }
        }
    </style>
</head>
<body>

<div class="container mt-4">
    <div class="card shadow-lg border-0 rounded-3">
        <div class="card-header bg-primary text-white d-flex align-items-center rounded-top-3">
            <i class="fas fa-file-excel fa-2x me-3"></i>
            <h4 class="mb-0">Importar Datos desde Excel</h4>
        </div>
        <div class="card-body p-4">
            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
                    <?php echo $message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <p class="lead mb-4">Sube un archivo Excel (.xlsx o .xls) para importar o actualizar cuentas maestras y perfiles vendidos.</p>
            <p class="text-muted small">
                <i class="fas fa-info-circle me-1"></i>Asegúrate de que la **primera fila** de tu archivo Excel contenga los encabezados exactos como se describe en la plantilla.
            </p>

            <h5 class="mt-4 mb-3 text-primary"><i class="fas fa-table me-2"></i>Descargar Plantilla de Ejemplo</h5>
            <div class="d-grid gap-2 col-md-8 mx-auto">
                <a href="/digitalpremium/assets/templates/plantilla_importacion.xlsx" download="plantilla_importacion_digitalpremium.xlsx" class="btn btn-outline-success btn-lg">
                    <i class="fas fa-download me-2"></i>Descargar Plantilla Excel (.xlsx)
                </a>
            </div>
            
            <hr class="my-5" style="border-top: 1px dashed var(--border-color);">

            <h5 class="mb-3 text-primary"><i class="fas fa-upload me-2"></i>Subir Archivo Excel</h5>
            <form action="importar_excel.php" method="POST" enctype="multipart/form-data" class="needs-validation" novalidate>
                <div class="mb-4">
                    <label for="excel_file" class="form-label fw-bold">Seleccionar archivo Excel:</label>
                    <input type="file" class="form-control" id="excel_file" name="excel_file" accept=".xlsx, .xls" required>
                    <div class="invalid-feedback">
                        Por favor, selecciona un archivo Excel (.xlsx o .xls).
                    </div>
                </div>
                <div class="d-grid gap-2">
                    <button type="submit" class="btn btn-primary btn-lg">
                        <i class="fas fa-file-import me-2"></i>Importar Datos
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php 
// Asume que 'includes/footer.php' contiene la etiqueta de cierre </body> y </html>
// y que incluye el Bootstrap JS Bundle. Si no, descomenta la línea de script de Bootstrap aquí.
include_once '../includes/footer.php'; 
?>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Bootstrap form validation
    var forms = document.querySelectorAll('.needs-validation');
    Array.prototype.slice.call(forms).forEach(function(form) {
        form.addEventListener('submit', function(event) {
            if (!form.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
            }
            form.classList.add('was-validated');
        }, false);
    });

    // Auto-hide messages after 5 seconds
    const alertMessage = document.querySelector('.alert');
    if (alertMessage) {
        setTimeout(() => {
            const bootstrapAlert = bootstrap.Alert.getInstance(alertMessage);
            if (bootstrapAlert) { // Check if an instance exists before trying to close
                bootstrapAlert.close();
            } else { // Fallback if instance not found (e.g., alert not fully initialized)
                alertMessage.remove(); 
            }
        }, 5000);
    }
});
</script>
</body>
</html>