<?php
// FILE: /digitalpremium/pages/ingresos.php
// VERSION FINAL (V7 - ULTIMATE PREMIUM DESIGN & UX REVAMP) POR GEMINI

// MODO DE DEPURACIÓN: Cámbialo a 'true' para ver el estado detallado de todas tus cuentas en la parte superior.
$debug_mode = false;

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['id_usuario'])) {
    header("Location: /digitalpremium/login.php");
    exit();
}

require_once '../config/db.php';
require_once '../includes/message_helper.php';

// <<< INICIO DE LÓGICA PHP OPTIMIZADA >>>

// 1. OBTENER DATOS DE SESIÓN Y CONFIGURACIÓN
$id_usuario_sesion = $_SESSION['id_usuario'];
$rol_usuario_sesion = $_SESSION['rol'];

// Función getConfigValue se asume que está en message_helper.php o se ha incluido previamente
if (!function_exists('getConfigValue')) {
    function getConfigValue($key, $id_usuario, $default = null) {
        global $pdo;
        try {
            $stmt = $pdo->prepare("SELECT valor_config FROM configuraciones WHERE clave_config = :key AND id_usuario = :id_usuario");
            $stmt->execute([':key' => $key, ':id_usuario' => $id_usuario]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$result) {
                $stmt = $pdo->prepare("SELECT valor_config FROM configuraciones WHERE clave_config = :key AND id_usuario IS NULL");
                $stmt->execute([':key' => $key]);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
            }
            
            return $result ? $result['valor_config'] : $default;
        } catch (PDOException $e) { error_log("Error al obtener config value {$key}: " . $e->getMessage()); return $default; }
    }
}

// Obtener el símbolo de la moneda desde la configuración del usuario
$moneda_simbolo = getConfigValue('moneda_simbolo', $id_usuario_sesion, '$');


// 2. LÓGICA DE FILTROS Y CONSULTA DE INGRESOS
$where_clauses = [];
$params = [];
$filtros_aplicados = false;

// Regla de Oro: Aplicar filtro de usuario si no es Superadmin
if ($rol_usuario_sesion !== 'superadmin') {
    $where_clauses[] = "pv.id_usuario = :id_usuario_sesion";
    $params[':id_usuario_sesion'] = $id_usuario_sesion;
}

// Obtener valores de los filtros desde GET (para persistencia en la URL)
$fecha_desde_in = $_GET['filtro_fecha_desde_in'] ?? '';
$fecha_hasta_in = $_GET['filtro_fecha_hasta_in'] ?? '';
$filtro_plataforma_in = $_GET['filtro_plataforma_in'] ?? '';
$filtro_cliente_in = $_GET['filtro_cliente_in'] ?? ''; // Filtro para cliente
$filtro_tipo_venta = $_GET['filtro_tipo_venta'] ?? ''; // Filtro para tipo de venta (Nueva/Renovación)
$filtro_vendedor = $_GET['filtro_vendedor'] ?? ''; // Filtro para vendedor
$filtro_estado = $_GET['filtro_estado'] ?? ''; // Filtro para estado del perfil

if (!empty($fecha_desde_in)) {
    $where_clauses[] = "pv.fecha_inicio_servicio >= :fecha_desde_in";
    $params[':fecha_desde_in'] = $fecha_desde_in;
    $filtros_aplicados = true;
}
if (!empty($fecha_hasta_in)) {
    $where_clauses[] = "pv.fecha_inicio_servicio <= :fecha_hasta_in";
    $params[':fecha_hasta_in'] = $fecha_hasta_in;
    $filtros_aplicados = true;
}
if (!empty($filtro_plataforma_in)) {
    $where_clauses[] = "cm.id_plataforma = :id_plataforma_in"; 
    $params[':id_plataforma_in'] = $filtro_plataforma_in;
    $filtros_aplicados = true;
}
if (!empty($filtro_cliente_in)) {
    $where_clauses[] = "c.nombre_completo LIKE :nombre_cliente_in";
    $params[':nombre_cliente_in'] = '%' . trim($filtro_cliente_in) . '%';
    $filtros_aplicados = true;
}
if (!empty($filtro_tipo_venta)) {
    if ($filtro_tipo_venta === 'renovacion') {
        $where_clauses[] = "EXISTS (
            SELECT 1 FROM perfiles_vendidos pv2 
            WHERE pv2.id_cliente = pv.id_cliente 
            AND pv2.id_cuenta_maestra = pv.id_cuenta_maestra 
            AND pv2.id_perfil_vendido < pv.id_perfil_vendido
        )";
    } else {
        $where_clauses[] = "NOT EXISTS (
            SELECT 1 FROM perfiles_vendidos pv2 
            WHERE pv2.id_cliente = pv.id_cliente 
            AND pv2.id_cuenta_maestra = pv.id_cuenta_maestra 
            AND pv2.id_perfil_vendido < pv.id_perfil_vendido
        )";
    }
    $filtros_aplicados = true;
}
if (!empty($filtro_vendedor)) {
    $where_clauses[] = "u.nombre_completo LIKE :nombre_vendedor";
    $params[':nombre_vendedor'] = '%' . trim($filtro_vendedor) . '%';
    $filtros_aplicados = true;
}
if (!empty($filtro_estado)) {
    $where_clauses[] = "pv.estado_suscripcion = :estado_suscripcion";
    $params[':estado_suscripcion'] = $filtro_estado;
    $filtros_aplicados = true;
}

// Asegurar que solo se listen ingresos válidos (precio_venta > 0 y fecha_inicio_servicio no nula)
$where_clauses[] = "pv.precio_venta > 0";
$where_clauses[] = "pv.fecha_inicio_servicio IS NOT NULL";

$sql_where = " WHERE " . implode(" AND ", $where_clauses);

// Consulta principal de Ingresos mejorada
$sql_ingresos = "SELECT 
                    pv.id_perfil_vendido, 
                    pv.fecha_inicio_servicio, 
                    pv.precio_venta, 
                    pv.nombre_perfil_cliente,
                    pv.fecha_fin_servicio,
                    pv.estado_suscripcion,
                    c.nombre_completo AS nombre_cliente, 
                    c.telefono AS telefono_cliente,
                    c.email AS email_cliente,
                    p.nombre_plataforma, 
                    p.id_plataforma, 
                    cm.email_cuenta,
                    u.nombre_completo AS nombre_vendedor,
                    u.username AS username_vendedor,
                    u.rol AS rol_vendedor,
                    CASE 
                        WHEN EXISTS (
                            SELECT 1 FROM perfiles_vendidos pv2 
                            WHERE pv2.id_cliente = pv.id_cliente 
                            AND pv2.id_cuenta_maestra = pv.id_cuenta_maestra 
                            AND pv2.id_perfil_vendido < pv.id_perfil_vendido
                        ) THEN 'Renovación'
                        ELSE 'Nueva Venta'
                    END AS tipo_venta,
                    CASE 
                        WHEN pv.estado_suscripcion = 1 THEN 'Activo'
                        WHEN pv.estado_suscripcion = 0 THEN 'Vencido'
                        WHEN pv.estado_suscripcion = 2 THEN 'Cancelado'
                        ELSE 'Desconocido'
                    END AS estado_actual
                FROM perfiles_vendidos pv
                JOIN clientes c ON pv.id_cliente = c.id_cliente
                JOIN cuentas_maestras cm ON pv.id_cuenta_maestra = cm.id_cuenta_maestra
                JOIN plataformas p ON cm.id_plataforma = p.id_plataforma
                LEFT JOIN usuarios u ON pv.id_usuario = u.id_usuario
                $sql_where
                ORDER BY pv.fecha_inicio_servicio DESC"; // Ordenar por fecha_inicio_servicio de forma descendente para que las más recientes aparezcan primero

$stmt_ingresos = $pdo->prepare($sql_ingresos);
$stmt_ingresos->execute($params);
$ingresos = $stmt_ingresos->fetchAll(PDO::FETCH_ASSOC);

// --- Procesamiento de datos para los Gráficos y KPIs ---
$total_ingresos = 0;
$ingresos_por_dia_raw = [];
$ingresos_por_plataforma = [];

foreach ($ingresos as $ingreso) {
    $total_ingresos += $ingreso['precio_venta'];
    
    $fecha = date('Y-m-d', strtotime($ingreso['fecha_inicio_servicio'])); 
    if (!isset($ingresos_por_dia_raw[$fecha])) {
        $ingresos_por_dia_raw[$fecha] = 0;
    }
    $ingresos_por_dia_raw[$fecha] += $ingreso['precio_venta'];
    
    $plataforma = $ingreso['nombre_plataforma'];
    if (!isset($ingresos_por_plataforma[$plataforma])) {
        $ingresos_por_plataforma[$plataforma] = 0;
    }
    $ingresos_por_plataforma[$plataforma] += $ingreso['precio_venta'];
}

$numero_de_ventas = count($ingresos);
$promedio_por_venta = ($numero_de_ventas > 0) ? $total_ingresos / $numero_de_ventas : 0;

// Asegurar que el rango de fechas para el gráfico sea continuo, incluso si no hay datos todos los días
$ingresos_por_dia_completos = [];
if ($numero_de_ventas > 0) {
    $all_dates = array_keys($ingresos_por_dia_raw);
    $fecha_inicio_grafico = !empty($fecha_desde_in) ? $fecha_desde_in : (empty($all_dates) ? date('Y-m-d') : min($all_dates));
    $fecha_fin_grafico = !empty($fecha_hasta_in) ? $fecha_hasta_in : (empty($all_dates) ? date('Y-m-d') : max($all_dates));
    
    $period = new DatePeriod(
        new DateTime($fecha_inicio_grafico),
        new DateInterval('P1D'),
        (new DateTime($fecha_fin_grafico))->modify('+1 day')
    );

    foreach ($period as $date) {
        $ingresos_por_dia_completos[$date->format('Y-m-d')] = 0;
    }

    foreach ($ingresos_por_dia_raw as $fecha_venta_actual => $monto) {
        if (isset($ingresos_por_dia_completos[$fecha_venta_actual])) {
            $ingresos_por_dia_completos[$fecha_venta_actual] = $monto;
        }
    }
}
ksort($ingresos_por_dia_completos); // Ordenar por fecha para el gráfico de línea
arsort($ingresos_por_plataforma); // Ordenar plataformas por ingresos (descendente)


// --- 3. CONSULTAS PARA DROPDOWNS DE FILTROS ---
$plataformas_filtro = $pdo->query("SELECT id_plataforma, nombre_plataforma FROM plataformas WHERE estado = 1 ORDER BY nombre_plataforma ASC")->fetchAll(PDO::FETCH_ASSOC);

// Obtener nombres de clientes para el filtro
$sql_clientes_filtro = "SELECT DISTINCT nombre_completo FROM clientes WHERE nombre_completo IS NOT NULL AND nombre_completo != ''";
$params_clientes_filtro = [];

if ($rol_usuario_sesion !== 'superadmin') {
    $sql_clientes_filtro = "SELECT DISTINCT c.nombre_completo FROM clientes c JOIN perfiles_vendidos pv ON c.id_cliente = pv.id_cliente WHERE pv.id_usuario = :id_usuario_sesion AND c.nombre_completo IS NOT NULL AND c.nombre_completo != '' ORDER BY c.nombre_completo ASC";
    $params_clientes_filtro[':id_usuario_sesion'] = $id_usuario_sesion;
} else {
    $sql_clientes_filtro .= " ORDER BY nombre_completo ASC";
}

$stmt_cli_filtro = $pdo->prepare($sql_clientes_filtro);
$stmt_cli_filtro->execute($params_clientes_filtro);
$clientes_filtro = $stmt_cli_filtro->fetchAll(PDO::FETCH_COLUMN);

// Obtener vendedores para el filtro
$sql_vendedores_filtro = "SELECT DISTINCT u.nombre_completo FROM usuarios u 
                          JOIN perfiles_vendidos pv ON u.id_usuario = pv.id_usuario 
                          WHERE u.nombre_completo IS NOT NULL AND u.nombre_completo != ''";
$params_vendedores_filtro = [];

if ($rol_usuario_sesion !== 'superadmin') {
    $sql_vendedores_filtro .= " AND pv.id_usuario = :id_usuario_sesion";
    $params_vendedores_filtro[':id_usuario_sesion'] = $id_usuario_sesion;
}

$sql_vendedores_filtro .= " ORDER BY u.nombre_completo ASC";

$stmt_vend_filtro = $pdo->prepare($sql_vendedores_filtro);
$stmt_vend_filtro->execute($params_vendedores_filtro);
$vendedores_filtro = $stmt_vend_filtro->fetchAll(PDO::FETCH_COLUMN);

$page_title = "Dashboard de Ingresos - DigitalPremium";
include_once '../includes/header.php';

// Función para obtener el color del badge de plataforma en la tabla (se mantiene similar, pero los colores se ajustan en CSS)
function getPlatformBadgeColor(string $platform_name): string {
    $platform_name = strtolower($platform_name);
    // Puedes definir colores específicos para cada plataforma si lo deseas, o usar una paleta predefinida.
    // Aquí mantenemos la lógica base y ajustamos los colores en CSS general.
    if (str_contains($platform_name, 'netflix')) return '#E50914';
    if (str_contains($platform_name, 'disney')) return '#1A2A4B';
    if (str_contains($platform_name, 'prime')) return '#00A8E1';
    if (str_contains($platform_name, 'hbo') || str_contains($platform_name, 'max')) return '#6C2B80';
    if (str_contains($platform_name, 'spotify')) return '#1DB954';
    if (str_contains($platform_name, 'youtube')) return '#FF0000';
    if (str_contains($platform_name, 'paramount')) return '#0064FF';
    if (str_contains($platform_name, 'crunchyroll')) return '#F47521';
    if (str_contains($platform_name, 'canva')) return '#00C4CC';
    return '#607d8b'; // Color por defecto
}

?>

<style>
    /* VARIABLES DE COLOR Y DISEÑO (REFINADAS) */
    :root {
        --font-primary: 'Inter', sans-serif;
        --body-bg: #f5f8ff; /* Un azul muy claro, casi blanco */
        --card-bg: #ffffff;
        --card-border: #e0e5f0;
        --text-dark: #212529; /* Casi negro */
        --text-secondary: #607d8b; /* Gris azulado */
        
        --primary-main: #673AB7; /* Morado profundo (primary) */
        --primary-light: #9C27B0; /* Morado más claro para degradados */
        --primary-accent: #7E57C2; /* Morado intermedio para acentos */

        --danger-main: #F44336; /* Rojo principal */
        --danger-light: #FFCDD2;
        --danger-accent: #D32F2F;

        --success-main: #4CAF50; /* Verde */
        --success-light: #C8E6C9;
        --success-accent: #388E3C;

        --warning-main: #FFC107; /* Amarillo */
        --warning-light: #FFECB3;
        --warning-accent: #FFA000;

        --info-main: #03A9F4; /* Azul info */
        --info-light: #B3E5FC;

        /* COLORES ESPECÍFICOS PARA INGRESOS */
        --income-main: var(--success-main); /* Verde para ingresos */
        --income-accent: var(--success-accent);
        --income-light: var(--success-light);


        --shadow-sm: 0 1px 3px rgba(0,0,0,0.06);
        --shadow-md: 0 5px 15px rgba(0,0,0,0.08);
        --shadow-lg: 0 10px 30px rgba(0,0,0,0.12);
        
        --border-radius-sm: 0.5rem;
        --border-radius-md: 0.85rem;
        --border-radius-lg: 1.25rem;
    }

    /* GENERALES */
    body {  
        background-color: var(--body-bg);  
        font-family: var(--font-primary);  
        color: var(--text-dark);  
        line-height: 1.6;
    }

    /* CONTENEDOR PRINCIPAL Y PADDING */
    .container-fluid {
        padding-left: 1rem; /* Más reducido */
        padding-right: 1rem; /* Más reducido */
    }
    @media (max-width: 768px) {
        .container-fluid {
            padding-left: 0.5rem; /* Máxima reducción en móvil */
            padding-right: 0.5rem;
        }
    }

    /* ENCABEZADO DE PÁGINA */
    .page-header {
        margin-top: 1rem; /* Aún más reducido */
        margin-bottom: 1rem; /* Aún más reducido */
    }
    .page-header h1 {  
        font-weight: 800;  
        font-size: 1.8rem;  /* Aún más reducido */
        color: var(--text-dark);
        display: flex;
        align-items: center;
    }
    .page-header h1 .fas {
        font-size: 1.6rem; /* Aún más reducido */
        margin-right: 0.5rem; /* Aún más reducido */
        color: var(--income-main);
    }
    .page-header .btn {
        font-weight: 600;
        border-radius: 50rem;
        padding: 0.5rem 1rem; /* Más reducido */
        font-size: 0.85rem; /* Más reducido */
        transition: all 0.2s ease-in-out;
    }
    .page-header .btn-primary { 
        background: linear-gradient(45deg, var(--income-main), var(--income-accent));
        border: none;
        box-shadow: var(--shadow-md);
    }
    .page-header .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: var(--shadow-lg);
        filter: brightness(1.1);
    }
    .page-header .btn-outline-secondary {
        border-color: var(--card-border);
        color: var(--text-secondary);
        box-shadow: var(--shadow-sm);
    }
    .page-header .btn-outline-secondary:hover {
        background-color: var(--body-bg);
        color: var(--primary-accent);
        border-color: var(--primary-accent);
        box-shadow: var(--shadow-md);
    }
    .page-header .dropdown-menu {
        border-radius: var(--border-radius-sm);
        box-shadow: var(--shadow-md);
        border: none;
    }
    .page-header .dropdown-item {
        padding: 0.5rem 0.8rem; /* Más reducido */
        font-size: 0.85rem; /* Más reducido */
    }
    .page-header .dropdown-item .fas {
        margin-right: 0.5rem; /* Más reducido */
    }

    /* TARJETAS DE ESTADÍSTICAS (KPIs) - AÚN MÁS PEQUEÑAS */
    .stat-card-kpi {
        background: var(--card-bg);
        border: none;
        border-radius: var(--border-radius-md);
        box-shadow: var(--shadow-md);
        padding: 0.8rem; /* Más reducido */
        display: flex;
        flex-direction: column; 
        align-items: center;
        text-align: center;
        transition: all 0.2s ease-in-out;
        position: relative;
        overflow: hidden;
        margin-bottom: 0.8rem; /* Más reducido */
    }
    .stat-card-kpi::before {
        content: '';
        position: absolute;
        top: 0; left: 0; right: 0; bottom: 0;
        background: linear-gradient(135deg, rgba(76, 175, 80, 0.03), transparent 60%); /* Degradado más sutil */
        border-radius: var(--border-radius-md);
        z-index: -1;
    }
    .stat-card-kpi:hover {  
        transform: translateY(-2px); /* Menos movimiento */
        box-shadow: var(--shadow-lg);  
    }
    .stat-card-kpi .stat-icon {  
        font-size: 1.3rem; /* Más reducido */
        width: 45px; height: 45px; /* Más reducido */
        border-radius: 50%;  
        display: inline-flex;  
        align-items: center;  
        justify-content: center;  
        margin: 0 auto 0.4rem auto; /* Más reducido */
        flex-shrink: 0;  
        box-shadow: 0 2px 8px rgba(0,0,0,0.1); /* Sombra más sutil */
        color: white;  
    }
    /* Colores específicos para los iconos de KPI - ajustados para ingresos */
    .stat-card-kpi.total .stat-icon { background: linear-gradient(45deg, var(--income-main), var(--income-accent)); } 
    .stat-card-kpi.ventas .stat-icon { background: linear-gradient(45deg, var(--info-main), var(--info-light)); } 
    .stat-card-kpi.promedio .stat-icon { background: linear-gradient(45deg, var(--primary-main), var(--primary-accent)); } 

    .stat-card-kpi .stat-title {
        font-size: 0.8rem; /* Más reducido */
        font-weight: 500;
        color: var(--text-secondary);
        margin-bottom: 0.1rem; /* Más reducido */
        text-transform: uppercase;
        letter-spacing: 0.4px;
    }
    .stat-card-kpi .stat-value {
        font-size: 1.6rem;  /* Más reducido */
        font-weight: 800;  
        color: var(--text-dark);
        line-height: 1;  
    }

    /* TARJETAS DE GRÁFICOS Y TABLAS */
    .card.shadow-lg {
        border: none;
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow-lg);
        margin-bottom: 1rem; /* Más reducido */
        overflow: hidden;
    }
    .card-header {
        background-color: var(--card-bg);
        border-bottom: 1px solid var(--card-border);
        padding: 0.8rem 1rem; /* Más reducido */
        font-weight: 700;
        font-size: 0.95rem; /* Más reducido */
        color: var(--text-dark);
        display: flex;
        align-items: center;
    }
    .card-header .fas {
        margin-right: 0.5rem; /* Más reducido */
        color: var(--income-main); 
    }

    /* GRÁFICOS (CHART.JS) */
    .chart-area, .chart-pie {
        position: relative;
        height: 250px;  /* Altura aún más reducida para gráficos */
        width: 100%;
        margin: auto;
    }
    .card-body canvas {
        width: 100% !important;
        height: 100% !important;
    }

    /* TABLA DE INGRESOS - ESTILOS MUCHO MÁS COMPACTOS Y EXTENDIDOS */
    .table-responsive {
        border-radius: var(--border-radius-md);  
        overflow-x: auto; /* Mantener scroll horizontal como fallback */
        border: 1px solid var(--card-border); 
    }
    .table {
        margin-bottom: 0;  
        border-collapse: separate; 
        border-spacing: 0; 
        width: 100%; 
        /* Eliminamos min-width para permitir que se comprima al máximo */
    }
    .table-hover tbody tr:hover {
        background-color: rgba(76, 175, 80, 0.08); 
    }
    .table thead {
        background-color: black; 
        color: #FFFFFF; 
        text-transform: uppercase;
        font-size: 0.7em; /* Fuente muy, muy pequeña para el encabezado */
        letter-spacing: 0.1px; 
    }
    .table thead th {
        border-bottom: none;
        padding: 0.5rem 0.6rem; /* Padding mínimo en encabezados */
        font-weight: 700; 
        color: #FFFFFF;
        border-right: 1px solid rgba(255,255,255,0.1); 
    }
    .table thead th:last-child {
        border-right: none; 
    }
         /* Ajuste de anchos para las columnas específicas para que quepan */
     .table thead th:nth-child(1) { width: 9%; } /* Fecha Venta */
     .table thead th:nth-child(2) { width: 8%; } /* Tipo */
     .table thead th:nth-child(3) { width: 16%; } /* Cliente */
     .table thead th:nth-child(4) { width: 13%; } /* Vendedor */
     .table thead th:nth-child(5) { width: 11%; } /* Plataforma */
     .table thead th:nth-child(6) { width: 18%; } /* Cuenta */
     .table thead th:nth-child(7) { width: 13%; } /* Perfil */
     .table thead th:nth-child(8) { width: 9%; } /* Estado */
     .table thead th:nth-child(9) { width: 13%; } /* Monto */


    .table tbody td {
        padding: 0.5rem 0.6rem; /* Menos padding en celdas de datos */
        vertical-align: middle;
        font-size: 0.78rem; /* Fuente más pequeña para los datos */
        color: var(--text-dark);
        border-bottom: 1px solid var(--card-border); 
        border-right: 1px solid var(--card-border); 
        white-space: nowrap; 
        overflow: hidden; 
        text-overflow: ellipsis; 
    }
    .table tbody td:last-child {
        border-right: none; 
    }
    .table tbody tr:last-child td {
        border-bottom: none; 
    }
    .table tbody tr:nth-of-type(even) {
        background-color: rgba(0, 0, 0, 0.02); /* Aún más sutil */
    }
    /* Badge de plataforma en la tabla */
    .table tbody .badge {
        font-size: 0.7em; /* Aún más pequeño */
        padding: 0.25em 0.5em; /* Padding mínimo del badge */
        border-radius: var(--border-radius-sm);
        font-weight: 700; 
        color: white;
        text-shadow: 0 1px 1px rgba(0,0,0,0.3); 
        box-shadow: var(--shadow-sm); 
    }
    .table tbody small.text-muted {
        font-size: 0.65em; /* Tamaño aún más pequeño para el teléfono */
        margin-top: 1px;
        opacity: 0.5; /* Más sutil */
    }
    
    /* Mejoras para los badges de tipo de venta */
    .badge.bg-success {
        background: linear-gradient(135deg, #28a745 0%, #20c997 100%) !important;
        color: white !important;
        font-weight: 600;
        text-shadow: 0 1px 2px rgba(0,0,0,0.3);
    }
    
    .badge.bg-warning {
        background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%) !important;
        color: #212529 !important;
        font-weight: 600;
        text-shadow: 0 1px 2px rgba(255,255,255,0.5);
    }
    
    .badge.bg-danger {
        background: linear-gradient(135deg, #dc3545 0%, #c82333 100%) !important;
        color: white !important;
        font-weight: 600;
        text-shadow: 0 1px 2px rgba(0,0,0,0.3);
    }
    
    .badge.bg-secondary {
        background: linear-gradient(135deg, #6c757d 0%, #495057 100%) !important;
        color: white !important;
        font-weight: 600;
        text-shadow: 0 1px 2px rgba(0,0,0,0.3);
    }
    
    .badge-sm {
        font-size: 0.6em;
        padding: 0.2em 0.4em;
    }
    
    /* Mejoras para la información del cliente y vendedor */
    .table tbody td .fw-bold {
        color: var(--text-dark);
        font-weight: 700;
    }
    
    .table tbody td small i {
        width: 12px;
        text-align: center;
        margin-right: 4px;
    }
    
    /* Hover mejorado para las filas */
    .table-hover tbody tr:hover {
        background: linear-gradient(135deg, rgba(76, 175, 80, 0.08) 0%, rgba(76, 175, 80, 0.12) 100%);
        transform: translateY(-1px);
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        transition: all 0.3s ease;
    }
    
    /* Indicador de filtros activos */
    .filtros-activos {
        display: flex;
        align-items: center;
        flex-wrap: wrap;
        gap: 0.5rem;
    }
    
    .badge.bg-info {
        background: linear-gradient(135deg, #17a2b8 0%, #138496 100%) !important;
        color: white !important;
        font-weight: 600;
        text-shadow: 0 1px 2px rgba(0,0,0,0.3);
    }
    .text-end {
        text-align: end !important; 
    }

    /* MODAL DE FILTROS */
    .modal-content {
        border-radius: var(--border-radius-lg);
        box-shadow: var(--shadow-lg);
        border: none;
    }
    .modal-header {
        background: linear-gradient(45deg, var(--income-main), var(--income-accent)); 
        color: white;
        border-bottom: none;
        padding: 1rem 1.2rem; /* Más reducido */
        border-top-left-radius: var(--border-radius-lg);
        border-top-right-radius: var(--border-radius-lg);
    }
    .modal-header .modal-title {
        color: white;
        font-size: 1.2rem; /* Más reducido */
        font-weight: 700;
    }
    .modal-header .btn-close {
        filter: invert(1) brightness(2); 
    }
    .modal-body {
        padding: 1.2rem; /* Más reducido */
    }
    .modal-body .form-label {
        font-weight: 600;
        color: var(--text-dark);
        margin-bottom: 0.3rem; /* Más reducido */
        font-size: 0.85rem; /* Más reducido */
    }
    .modal-body .form-control,
    .modal-body .form-select {
        border-radius: var(--border-radius-sm);
        border-color: var(--card-border);
        box-shadow: inset 0 1px 3px rgba(0,0,0,0.05);
        padding: 0.5rem 0.7rem; /* Más reducido */
        font-size: 0.85rem; /* Más reducido */
    }
    .modal-body .input-group-text {
        background-color: var(--body-bg);
        border-color: var(--card-border);
        color: var(--text-secondary);
        border-radius: var(--border-radius-sm);
    }
    .modal-footer {
        border-top: none;
        padding: 1rem 1.2rem; /* Más reducido */
        background-color: var(--body-bg);
        border-bottom-left-radius: var(--border-radius-lg);
        border-bottom-right-radius: var(--border-radius-lg);
    }
    .modal-footer .btn {
        font-weight: 600;
        padding: 0.5rem 1rem; /* Más reducido */
        border-radius: 50rem;
        box-shadow: var(--shadow-sm);
        transition: all 0.2s ease-in-out;
    }
    .modal-footer .btn-outline-secondary {
        color: var(--text-secondary);
        border-color: var(--text-secondary);
    }
    .modal-footer .btn-outline-secondary:hover {
        background-color: var(--text-secondary);
        color: white;
    }
    .modal-footer .btn-primary { 
        background: linear-gradient(45deg, var(--income-main), var(--income-accent));
        border: none;
        color: white;
    }
         .modal-footer .btn-primary:hover {
         filter: brightness(1.1);
         transform: translateY(-2px);
     }

     /* ===== NUEVAS CARDS DE VENTAS ===== */
     .venta-card {
         background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
         border: 1px solid #e2e8f0;
         border-radius: 16px;
         box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
         transition: all 0.3s ease;
         overflow: hidden;
         height: 100%;
     }
     
     .venta-card:hover {
         transform: translateY(-4px);
         box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
         border-color: #cbd5e0;
     }
     
     .venta-card-header {
         background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
         color: white;
         padding: 1.5rem;
         display: flex;
         justify-content: space-between;
         align-items: flex-start;
     }
     
     .venta-fecha {
         flex: 1;
     }
     
     .fecha-principal {
         font-size: 1.1rem;
         font-weight: 700;
         margin-bottom: 0.25rem;
     }
     
     .fecha-hora {
         font-size: 0.9rem;
         opacity: 0.9;
     }
     
     .venta-tipo {
         text-align: right;
     }
     
     .badge.tipo-nueva {
         background: linear-gradient(135deg, #10b981 0%, #059669 100%);
         color: white;
         font-weight: 600;
         padding: 0.5rem 1rem;
         border-radius: 20px;
         font-size: 0.8rem;
         text-shadow: 0 1px 2px rgba(0,0,0,0.3);
     }
     
     .badge.tipo-renovacion {
         background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
         color: white;
         font-weight: 600;
         padding: 0.5rem 1rem;
         border-radius: 20px;
         font-size: 0.8rem;
         text-shadow: 0 1px 2px rgba(0,0,0,0.3);
     }
     
     .venta-card-body {
         padding: 1.5rem;
     }
     
     .venta-monto {
         text-align: center;
         margin-bottom: 1.5rem;
         padding: 1rem;
         background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
         border-radius: 12px;
         border: 2px solid #0ea5e9;
     }
     
     .monto-label {
         font-size: 0.9rem;
         color: #0369a1;
         font-weight: 600;
         margin-bottom: 0.5rem;
         text-transform: uppercase;
         letter-spacing: 0.5px;
     }
     
     .monto-valor {
         font-size: 1.8rem;
         font-weight: 800;
         color: #0c4a6e;
         text-shadow: 0 1px 2px rgba(0,0,0,0.1);
     }
     
     .venta-info {
         display: flex;
         flex-direction: column;
         gap: 1rem;
     }
     
     .info-section {
         border-bottom: 1px solid #e2e8f0;
         padding-bottom: 1rem;
     }
     
     .info-section:last-child {
         border-bottom: none;
         padding-bottom: 0;
     }
     
     .info-title {
         font-size: 0.85rem;
         font-weight: 700;
         color: #475569;
         margin-bottom: 0.5rem;
         text-transform: uppercase;
         letter-spacing: 0.5px;
     }
     
     .info-content {
         display: flex;
         flex-direction: column;
         gap: 0.25rem;
     }
     
     .cliente-nombre,
     .vendedor-nombre,
     .perfil-nombre {
         font-size: 1rem;
         font-weight: 700;
         color: #1e293b;
         margin-bottom: 0.25rem;
     }
     
     .cliente-detalle,
     .vendedor-detalle,
     .perfil-detalle {
         font-size: 0.8rem;
         color: #64748b;
         display: flex;
         align-items: center;
     }
     
     .cuenta-email {
         font-size: 0.9rem;
         color: #1e293b;
         font-weight: 600;
         word-break: break-all;
     }
     
     .badge.rol-badge {
         background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
         color: white;
         font-size: 0.7rem;
         padding: 0.25rem 0.5rem;
         border-radius: 12px;
         font-weight: 600;
         align-self: flex-start;
         margin-top: 0.25rem;
     }
     
     .badge.plataforma-badge {
         font-size: 0.8rem;
         padding: 0.5rem 1rem;
         border-radius: 20px;
         font-weight: 600;
         color: white;
         text-shadow: 0 1px 2px rgba(0,0,0,0.3);
         box-shadow: 0 2px 4px rgba(0,0,0,0.1);
     }
     
     .badge.estado-activo {
         background: linear-gradient(135deg, #10b981 0%, #059669 100%);
         color: white;
         font-size: 0.8rem;
         padding: 0.5rem 1rem;
         border-radius: 20px;
         font-weight: 600;
         text-shadow: 0 1px 2px rgba(0,0,0,0.3);
     }
     
     .badge.estado-vencido {
         background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
         color: white;
         font-size: 0.8rem;
         padding: 0.5rem 1rem;
         border-radius: 20px;
         font-weight: 600;
         text-shadow: 0 1px 2px rgba(0,0,0,0.3);
     }
     
     .badge.estado-cancelado {
         background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
         color: white;
         font-size: 0.8rem;
         padding: 0.5rem 1rem;
         border-radius: 20px;
         font-weight: 600;
         text-shadow: 0 1px 2px rgba(0,0,0,0.3);
     }
     
     .badge.estado-desconocido {
         background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
         color: white;
         font-size: 0.8rem;
         padding: 0.5rem 1rem;
         border-radius: 20px;
         font-weight: 600;
         text-shadow: 0 1px 2px rgba(0,0,0,0.3);
     }
     
     .badge.bg-primary {
         background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%) !important;
         color: white !important;
         font-weight: 600;
         text-shadow: 0 1px 2px rgba(0,0,0,0.3);
         font-size: 0.75rem;
         padding: 0.4rem 0.8rem;
         border-radius: 12px;
     }
     
     /* Responsive para las cards */
     @media (max-width: 1200px) {
         .col-xl-4 {
             flex: 0 0 50%;
             max-width: 50%;
         }
     }
     
     @media (max-width: 768px) {
         .col-lg-6,
         .col-xl-4 {
             flex: 0 0 100%;
             max-width: 100%;
         }
         
         .venta-card-header {
             flex-direction: column;
             gap: 1rem;
             align-items: flex-start;
         }
         
         .venta-tipo {
             text-align: left;
         }
         
         .monto-valor {
             font-size: 1.5rem;
         }
     }

</style>

<div class="container-fluid px-lg-5 px-md-4 py-4">
    <header class="page-header d-flex justify-content-between align-items-center">
        <h1><i class="fas fa-chart-line"></i>Dashboard de Ingresos</h1>
        <div class="d-flex gap-3">
            <button class="btn btn-primary" type="button" data-bs-toggle="modal" data-bs-target="#modalFiltros">
                <i class="fas fa-filter me-2"></i> Filtrar Reporte
            </button>
            <div class="btn-group">
                <button type="button" class="btn btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">
                    <i class="fas fa-download me-2"></i> Exportar
                </button>
                <ul class="dropdown-menu dropdown-menu-end">
                    <li><a class="dropdown-item" href="#" id="exportExcel"><i class="fas fa-file-excel me-2 text-success"></i>Exportar a Excel</a></li>
                    <li><a class="dropdown-item" href="#" id="exportPdf"><i class="fas fa-file-pdf me-2 text-danger"></i>Exportar a PDF</a></li>
                    <li><hr class="dropdown-divider"></li>
                    <li><a class="dropdown-item" href="#" id="printTable"><i class="fas fa-print me-2 text-primary"></i>Imprimir</a></li>
                </ul>
            </div>
        </div>
    </header>

    <?php if (isset($_GET['mensaje'])): ?>
    <div class="alert alert-<?php echo htmlspecialchars($_GET['tipo'] ?? 'info'); ?> alert-dismissible fade show"><button type="button" class="btn-close" data-bs-dismiss="alert"></button><?php echo htmlspecialchars(urldecode($_GET['mensaje'])); ?></div>
    <?php endif; ?>

    <div class="row g-4 mb-4">
        <div class="col-lg-4 col-md-6">
            <div class="stat-card-kpi total">
                <div class="stat-icon"><i class="fas fa-hand-holding-usd"></i></div><div>
                    <div class="stat-title">Ingresos Totales</div>
                    <div class="stat-value"><?php echo htmlspecialchars($moneda_simbolo); ?> <?php echo number_format($total_ingresos, 2); ?></div>
                </div>
            </div>
        </div>
        <div class="col-lg-4 col-md-6">
            <div class="stat-card-kpi ventas">
                <div class="stat-icon"><i class="fas fa-tags"></i></div><div>
                    <div class="stat-title">Número de Ventas</div>
                    <div class="stat-value"><?php echo number_format($numero_de_ventas); ?></div>
                </div>
            </div>
        </div>
        <div class="col-lg-4 col-md-12">
            <div class="stat-card-kpi promedio">
                <div class="stat-icon"><i class="fas fa-chart-bar"></i></div><div>
                    <div class="stat-title">Ingreso Promedio / Venta</div>
                    <div class="stat-value"><?php echo htmlspecialchars($moneda_simbolo); ?> <?php echo number_format($promedio_por_venta, 2); ?></div>
                </div>
            </div>
        </div>
    </div>

    <div class="row g-4">
        <div class="col-lg-9"> <div class="card shadow-lg mb-4">
                <div class="card-header">
                    <h6 class="m-0 fw-bold"><i class="fas fa-chart-area me-2"></i>Evolución de Ingresos por Día</h6>
                </div>
                <div class="card-body">
                    <div class="chart-area" style="height: 250px;"> <canvas id="graficoIngresosLinea"></canvas>
                    </div>
                </div>
            </div>
            <div class="card shadow-lg">
                               <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                                                 <h6 class="m-0 fw-bold">
                             <i class="fas fa-cards-blank me-2"></i>Detalle de Ventas - Vista de Cards
                             <span class="badge bg-primary ms-2"><?php echo count($ingresos); ?> ventas</span>
                         </h6>
                        <?php if ($filtros_aplicados): ?>
                            <div class="filtros-activos">
                                <span class="badge bg-info me-2">
                                    <i class="fas fa-filter me-1"></i>Filtros Activos
                                </span>
                                <small class="text-muted">
                                    <?php 
                                    $filtros_texto = [];
                                    if (!empty($fecha_desde_in)) $filtros_texto[] = "Desde: " . date('d/m/Y', strtotime($fecha_desde_in));
                                    if (!empty($fecha_hasta_in)) $filtros_texto[] = "Hasta: " . date('d/m/Y', strtotime($fecha_hasta_in));
                                    if (!empty($filtro_plataforma_in)) $filtros_texto[] = "Plataforma seleccionada";
                                    if (!empty($filtro_cliente_in)) $filtros_texto[] = "Cliente: " . $filtro_cliente_in;
                                    if (!empty($filtro_tipo_venta)) $filtros_texto[] = "Tipo: " . ucfirst($filtro_tipo_venta);
                                    if (!empty($filtro_vendedor)) $filtros_texto[] = "Vendedor: " . $filtro_vendedor;
                                    if (!empty($filtro_estado)) $filtros_texto[] = "Estado: " . ucfirst($filtro_estado);
                                    echo implode(', ', $filtros_texto);
                                    ?>
                                </small>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="card-body">
                                         <div class="row g-4">
                         <?php if (empty($ingresos)): ?>
                             <div class="col-12">
                                 <div class="text-center py-5">
                                     <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                                     <h5 class="text-muted">No se encontraron ventas</h5>
                                     <p class="text-muted">No hay ventas que coincidan con los filtros aplicados.</p>
                                 </div>
                             </div>
                         <?php else: ?>
                         <?php foreach ($ingresos as $ingreso): ?>
                             <div class="col-lg-6 col-xl-4">
                                 <div class="venta-card">
                                     <div class="venta-card-header">
                                         <div class="venta-fecha">
                                             <div class="fecha-principal">
                                                 <i class="fas fa-calendar-alt me-2"></i>
                                                 <?php echo htmlspecialchars(date("d/m/Y", strtotime($ingreso['fecha_inicio_servicio']))); ?>
                                             </div>
                                             <div class="fecha-hora">
                                                 <i class="fas fa-clock me-1"></i>
                                                 <?php echo htmlspecialchars(date("H:i", strtotime($ingreso['fecha_inicio_servicio']))); ?>
                                             </div>
                                         </div>
                                         <div class="venta-tipo">
                                             <?php if ($ingreso['tipo_venta'] === 'Renovación'): ?>
                                                 <span class="badge tipo-renovacion">
                                                     <i class="fas fa-arrows-rotate me-1"></i>Renovación
                                                 </span>
                                             <?php else: ?>
                                                 <span class="badge tipo-nueva">
                                                     <i class="fas fa-plus-circle me-1"></i>Nueva Venta
                                                 </span>
                                             <?php endif; ?>
                                         </div>
                                     </div>
                                     
                                     <div class="venta-card-body">
                                         <div class="venta-monto">
                                             <div class="monto-label">Monto de Venta</div>
                                             <div class="monto-valor">
                                                 <?php echo htmlspecialchars($moneda_simbolo); ?> <?php echo number_format($ingreso['precio_venta'], 2); ?>
                                             </div>
                                         </div>
                                         
                                         <div class="venta-info">
                                             <div class="info-section">
                                                 <div class="info-title">
                                                     <i class="fas fa-user me-2"></i>Cliente
                                                 </div>
                                                 <div class="info-content">
                                                     <div class="cliente-nombre"><?php echo htmlspecialchars($ingreso['nombre_cliente'] ?? 'N/A'); ?></div>
                                                     <?php if (!empty($ingreso['telefono_cliente'])): ?>
                                                         <div class="cliente-detalle">
                                                             <i class="fas fa-phone me-1"></i><?php echo htmlspecialchars($ingreso['telefono_cliente']); ?>
                                                         </div>
                                                     <?php endif; ?>
                                                     <?php if (!empty($ingreso['email_cliente'])): ?>
                                                         <div class="cliente-detalle">
                                                             <i class="fas fa-envelope me-1"></i><?php echo htmlspecialchars($ingreso['email_cliente']); ?>
                                                         </div>
                                                     <?php endif; ?>
                                                 </div>
                                             </div>
                                             
                                             <div class="info-section">
                                                 <div class="info-title">
                                                     <i class="fas fa-user-tie me-2"></i>Vendedor
                                                 </div>
                                                 <div class="info-content">
                                                     <div class="vendedor-nombre"><?php echo htmlspecialchars($ingreso['nombre_vendedor'] ?? 'N/A'); ?></div>
                                                     <?php if (!empty($ingreso['username_vendedor'])): ?>
                                                         <div class="vendedor-detalle">@<?php echo htmlspecialchars($ingreso['username_vendedor']); ?></div>
                                                     <?php endif; ?>
                                                     <?php if (!empty($ingreso['rol_vendedor'])): ?>
                                                         <span class="badge rol-badge"><?php echo htmlspecialchars($ingreso['rol_vendedor']); ?></span>
                                                     <?php endif; ?>
                                                 </div>
                                             </div>
                                             
                                             <div class="info-section">
                                                 <div class="info-title">
                                                     <i class="fas fa-tv me-2"></i>Plataforma
                                                 </div>
                                                 <div class="info-content">
                                                     <span class="badge plataforma-badge" style="background-color: <?php echo getPlatformBadgeColor($ingreso['nombre_plataforma']); ?>;">
                                                         <?php echo htmlspecialchars($ingreso['nombre_plataforma']); ?>
                                                     </span>
                                                 </div>
                                             </div>
                                             
                                             <div class="info-section">
                                                 <div class="info-title">
                                                     <i class="fas fa-envelope me-2"></i>Cuenta
                                                 </div>
                                                 <div class="info-content">
                                                     <div class="cuenta-email"><?php echo htmlspecialchars($ingreso['email_cuenta']); ?></div>
                                                 </div>
                                             </div>
                                             
                                             <div class="info-section">
                                                 <div class="info-title">
                                                     <i class="fas fa-user-circle me-2"></i>Perfil
                                                 </div>
                                                 <div class="info-content">
                                                     <div class="perfil-nombre"><?php echo htmlspecialchars($ingreso['nombre_perfil_cliente']); ?></div>
                                                     <?php if (!empty($ingreso['fecha_fin_servicio'])): ?>
                                                         <div class="perfil-detalle">
                                                             <i class="fas fa-calendar me-1"></i>Vence: <?php echo htmlspecialchars(date("d/m/Y", strtotime($ingreso['fecha_fin_servicio']))); ?>
                                                         </div>
                                                     <?php endif; ?>
                                                 </div>
                                             </div>
                                             
                                             <div class="info-section">
                                                 <div class="info-title">
                                                     <i class="fas fa-info-circle me-2"></i>Estado
                                                 </div>
                                                 <div class="info-content">
                                                     <?php 
                                                     $estado_class = 'estado-desconocido';
                                                     $estado_icon = 'fas fa-question';
                                                     
                                                     switch($ingreso['estado_actual']) {
                                                         case 'Activo':
                                                             $estado_class = 'estado-activo';
                                                             $estado_icon = 'fas fa-check-circle';
                                                             break;
                                                         case 'Vencido':
                                                             $estado_class = 'estado-vencido';
                                                             $estado_icon = 'fas fa-clock';
                                                             break;
                                                         case 'Cancelado':
                                                             $estado_class = 'estado-cancelado';
                                                             $estado_icon = 'fas fa-times-circle';
                                                             break;
                                                     }
                                                     ?>
                                                     <span class="badge <?php echo $estado_class; ?>">
                                                         <i class="<?php echo $estado_icon; ?> me-1"></i><?php echo htmlspecialchars($ingreso['estado_actual']); ?>
                                                     </span>
                                                 </div>
                                             </div>
                                         </div>
                                     </div>
                                 </div>
                             </div>
                         <?php endforeach; ?>
                         <?php endif; ?>
                     </div>
                </div>
            </div>
        </div>

        <div class="col-lg-3"> <div class="card shadow-lg">
                <div class="card-header">
                    <h6 class="m-0 fw-bold"><i class="fas fa-pie-chart me-2"></i>Distribución por Plataforma (Ingresos)</h6>
                </div>
                <div class="card-body">
                    <div class="chart-pie pt-4" style="height: 250px;"> <canvas id="graficoPlataformasDona"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="modalFiltros" tabindex="-1" aria-labelledby="modalFiltrosLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="modalFiltrosLabel"><i class="fas fa-filter me-2"></i>Filtrar Reporte de Ingresos</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="" method="GET">
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-12">
                            <label class="form-label">Rango de Fechas de Venta:</label>
                            <div class="input-group">
                                <input type="date" name="filtro_fecha_desde_in" class="form-control" value="<?php echo htmlspecialchars($fecha_desde_in); ?>">
                                <span class="input-group-text">a</span>
                                <input type="date" name="filtro_fecha_hasta_in" class="form-control" value="<?php echo htmlspecialchars($fecha_hasta_in); ?>">
                            </div>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Plataforma:</label>
                            <select name="filtro_plataforma_in" class="form-select">
                                <option value="">Todas las plataformas</option>
                                <?php foreach ($plataformas_filtro as $plataforma_f): ?>
                                    <option value="<?php echo $plataforma_f['id_plataforma']; ?>" <?php echo ($filtro_plataforma_in == $plataforma_f['id_plataforma']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($plataforma_f['nombre_plataforma']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Cliente:</label>
                            <input list="clientes_datalist" name="filtro_cliente_in" class="form-control" value="<?php echo htmlspecialchars($filtro_cliente_in); ?>" placeholder="Escriba para buscar cliente...">
                            <datalist id="clientes_datalist">
                                <?php foreach ($clientes_filtro as $cli_nombre): ?>
                                    <option value="<?php echo htmlspecialchars($cli_nombre); ?>">
                                <?php endforeach; ?>
                            </datalist>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Tipo de Venta:</label>
                            <select name="filtro_tipo_venta" class="form-select">
                                <option value="">Todos los tipos</option>
                                <option value="nueva" <?php echo ($filtro_tipo_venta === 'nueva') ? 'selected' : ''; ?>>Nueva Venta</option>
                                <option value="renovacion" <?php echo ($filtro_tipo_venta === 'renovacion') ? 'selected' : ''; ?>>Renovación</option>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Vendedor:</label>
                            <input list="vendedores_datalist" name="filtro_vendedor" class="form-control" value="<?php echo htmlspecialchars($filtro_vendedor); ?>" placeholder="Escriba para buscar vendedor...">
                            <datalist id="vendedores_datalist">
                                <?php foreach ($vendedores_filtro as $vend_nombre): ?>
                                    <option value="<?php echo htmlspecialchars($vend_nombre); ?>">
                                <?php endforeach; ?>
                            </datalist>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Estado del Perfil:</label>
                            <select name="filtro_estado" class="form-select">
                                <option value="">Todos los estados</option>
                                <option value="1" <?php echo ($filtro_estado === '1') ? 'selected' : ''; ?>>Activo</option>
                                <option value="0" <?php echo ($filtro_estado === '0') ? 'selected' : ''; ?>>Vencido</option>
                                <option value="2" <?php echo ($filtro_estado === '2') ? 'selected' : ''; ?>>Cancelado</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <a href="/digitalpremium/pages/ingresos.php" class="btn btn-outline-secondary">
                        <i class="fas fa-eraser me-1"></i>Limpiar Todo
                    </a>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-check me-1"></i> Aplicar Filtros
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php include_once '../includes/footer.php'; ?>

<script src="https://code.jquery.com/jquery-3.7.0.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap5.min.css">
<script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/dataTables.buttons.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.1.53/pdfmake.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.1.53/vfs_fonts.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.html5.min.js"></script>
 <script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.print.min.js"></script>
 <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const monedaSimbolo = '<?php echo htmlspecialchars($moneda_simbolo ?? ''); ?>';

    // Gráfico de Línea: Evolución de Ingresos
    const ctxLinea = document.getElementById('graficoIngresosLinea')?.getContext('2d');
    if (ctxLinea) {
        new Chart(ctxLinea, {
            type: 'line',
            data: {
                labels: <?php echo json_encode(array_keys($ingresos_por_dia_completos)); ?>,
                datasets: [{
                    label: 'Ingresos',
                    data: <?php echo json_encode(array_values($ingresos_por_dia_completos)); ?>,
                    borderColor: '#4CAF50', // Usar success-main (verde) para ingresos
                    backgroundColor: 'rgba(76, 175, 80, 0.1)', // Fondo más suave para el área
                    fill: true,
                    tension: 0.3,
                    pointRadius: 3, 
                    pointHoverRadius: 5,
                    pointBackgroundColor: '#4CAF50',
                    pointBorderColor: '#fff',
                }]
            },
            options: {
                responsive: true, 
                maintainAspectRatio: false,
                scales: { 
                    x: {
                        grid: {
                            display: false 
                        },
                        ticks: {
                            color: '#607d8b' 
                        }
                    },
                    y: { 
                        beginAtZero: true, 
                        grid: {
                            color: '#e0e5f0' 
                        },
                        ticks: { 
                            callback: (v) => `${monedaSimbolo} ${v.toLocaleString()}`,
                            color: '#607d8b' 
                        } 
                    } 
                },
                plugins: {
                    legend: { display: false },
                    tooltip: { 
                        callbacks: { 
                            label: (c) => ` ${c.dataset.label}: ${monedaSimbolo} ${c.parsed.y.toLocaleString(undefined, {minimumFractionDigits: 2})}` 
                        },
                        backgroundColor: 'rgba(0,0,0,0.8)', 
                        titleColor: '#fff',
                        bodyColor: '#fff',
                        borderColor: '#4CAF50', // Borde del tooltip en verde
                        borderWidth: 1,
                        cornerRadius: 4,
                        displayColors: false
                    }
                }
            }
        });
    }

    // Gráfico de Dona: Distribución por Plataforma (Ingresos)
    const ctxDona = document.getElementById('graficoPlataformasDona')?.getContext('2d');
    if (ctxDona) {
        new Chart(ctxDona, {
            type: 'doughnut',
            data: {
                labels: <?php echo json_encode(array_keys($ingresos_por_plataforma)); ?>,
                datasets: [{
                    data: <?php echo json_encode(array_values($ingresos_por_plataforma)); ?>,
                    backgroundColor: [
                        '#4CAF50', // Success-main (verde)
                        '#03A9F4', // Info-main (azul)
                        '#673AB7', // Primary-main (morado)
                        '#FFC107', // Warning-main (amarillo)
                        '#FF5722', // Deeper Orange
                        '#00BCD4', // Cyan
                        '#E91E63', // Pink
                        '#F44336', // Danger-main (rojo, si aplica alguna plataforma que lo justifique)
                        '#9E9E9E'  // Grey
                    ],
                    hoverOffset: 8, 
                    borderWidth: 2, 
                    borderColor: '#fff' 
                }]
            },
            options: {
                responsive: true, 
                maintainAspectRatio: false,
                plugins: {
                    legend: { 
                        position: 'bottom',
                        labels: {
                            boxWidth: 12, 
                            padding: 15, 
                            color: '#607d8b' 
                        }
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const label = context.label || '';
                                const value = context.parsed || 0;
                                const total = context.chart.data.datasets[0].data.reduce((a, b) => a + b, 0);
                                const percentage = total > 0 ? ((value / total) * 100).toFixed(2) : 0;
                                return ` ${label}: ${monedaSimbolo} ${value.toLocaleString(undefined, {minimumFractionDigits: 2})} (${percentage}%)`;
                            }
                        },
                        backgroundColor: 'rgba(0,0,0,0.8)',
                        titleColor: '#fff',
                        bodyColor: '#fff',
                        borderColor: '#4CAF50', // Borde del tooltip en verde
                        borderWidth: 1,
                        cornerRadius: 4,
                        displayColors: true
                    }
                }
            }
        });
    }

         // Función para exportar datos de las cards
     function exportarDatos(formato) {
         const datos = <?php echo json_encode($ingresos); ?>;
         let contenido = '';
         
         if (formato === 'excel' || formato === 'csv') {
             // Encabezados
             contenido = 'Fecha,Fecha Hora,Tipo,Cliente,Telefono Cliente,Email Cliente,Vendedor,Username Vendedor,Rol Vendedor,Plataforma,Cuenta,Perfil,Fecha Vencimiento,Estado,Monto\n';
             
             // Datos
             datos.forEach(ingreso => {
                 const fecha = new Date(ingreso.fecha_inicio_servicio).toLocaleDateString('es-ES');
                 const hora = new Date(ingreso.fecha_inicio_servicio).toLocaleTimeString('es-ES', {hour: '2-digit', minute: '2-digit'});
                 const fechaVenc = ingreso.fecha_fin_servicio ? new Date(ingreso.fecha_fin_servicio).toLocaleDateString('es-ES') : '';
                 
                 contenido += `"${fecha}","${hora}","${ingreso.tipo_venta}","${ingreso.nombre_cliente || 'N/A'}","${ingreso.telefono_cliente || ''}","${ingreso.email_cliente || ''}","${ingreso.nombre_vendedor || 'N/A'}","${ingreso.username_vendedor || ''}","${ingreso.rol_vendedor || ''}","${ingreso.nombre_plataforma}","${ingreso.email_cuenta}","${ingreso.nombre_perfil_cliente}","${fechaVenc}","${ingreso.estado_actual}","${ingreso.precio_venta}"\n`;
             });
             
             if (formato === 'excel') {
                 // Crear archivo Excel usando SheetJS
                 const ws = XLSX.utils.aoa_to_sheet(contenido.split('\n').map(row => row.split(',')));
                 const wb = XLSX.utils.book_new();
                 XLSX.utils.book_append_sheet(wb, ws, "Ventas");
                 XLSX.writeFile(wb, "reporte_ventas.xlsx");
             } else {
                 // CSV
                 const blob = new Blob([contenido], { type: 'text/csv;charset=utf-8;' });
                 const link = document.createElement("a");
                 link.href = URL.createObjectURL(blob);
                 link.download = "reporte_ventas.csv";
                 link.click();
             }
         } else if (formato === 'print') {
             window.print();
         }
     }
     
     // Asignar funciones a los botones de exportación/impresión
     $('#exportExcel').on('click', (e) => { e.preventDefault(); exportarDatos('excel'); });
     $('#exportPdf').on('click', (e) => { e.preventDefault(); exportarDatos('csv'); }); // Cambiado a CSV por simplicidad
     $('#printTable').on('click', (e) => { e.preventDefault(); exportarDatos('print'); });

});

</script>