<?php
// FILE: /digitalpremium/pages/perfiles_vendidos.php

// Enable error display for debugging (COMMENT OUT OR REMOVE IN PRODUCTION!)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['id_usuario'])) {
    header("Location: /digitalpremium/login.php");
    exit();
}

require_once '../config/db.php';
require_once '../includes/message_helper.php';
$OWNER_ID = (int)($_SESSION['data_owner_id'] ?? $_SESSION['id_usuario']);

// --- 1. DATOS DE SESIÓN Y FUNCIÓN DE CONFIGURACIÓN ---
$id_usuario_sesion = $_SESSION['id_usuario'];
$rol_usuario_sesion = $_SESSION['rol'];

$page_title = "Suscripciones de Clientes - DigitalPremium";
include_once '../includes/header.php';

// --- Helper Functions ---
function getPlatformHeaderClass(string $platform_name): string {
    $map = [
        'netflix' => 'header-netflix',
        'disney' => 'header-disney',
        'prime' => 'header-prime',
        'hbo' => 'header-hbo',
        'max' => 'header-hbo',
        'spotify' => 'header-spotify',
        'crunchyroll' => 'header-crunchyroll',
        'canva' => 'header-canva',
        'paramount' => 'header-paramount',
        'iptv' => 'header-iptv',
        'capcut' => 'header-capcut',
        'directv' => 'header-directv',
        'movistar' => 'header-movistar',
        'flujo' => 'header-flujo',
        'viki' => 'header-viki',
        'youtube' => 'header-youtube',
    ];
    $platform_name = strtolower($platform_name);
    foreach ($map as $key => $class) {
        if (strpos($platform_name, $key) !== false) {
            return $class;
        }
    }
    return 'header-default';
}

function getPlatformIconClass(string $platform_name): string {
    $platform_name = strtolower($platform_name);
    if (strpos($platform_name, 'netflix') !== false) return 'fab fa-netflix';
    if (strpos($platform_name, 'disney') !== false) return 'fas fa-mouse-pointer';
    if (strpos($platform_name, 'prime video') !== false) return 'fab fa-amazon';
    if (strpos($platform_name, 'hbo') !== false || strpos($platform_name, 'max') !== false) return 'fas fa-tv';
    if (strpos($platform_name, 'spotify') !== false) return 'fab fa-spotify';
    if (strpos($platform_name, 'youtube') !== false) return 'fab fa-youtube';
    if (strpos($platform_name, 'crunchyroll') !== false) return 'fas fa-star';
    if (strpos($platform_name, 'canva') !== false) return 'fas fa-paint-brush';
    if (strpos($platform_name, 'paramount') !== false) return 'fas fa-mountain';
    if (strpos($platform_name, 'iptv') !== false) return 'fas fa-satellite-dish';
    if (strpos($platform_name, 'capcut') !== false) return 'fas fa-video';
    if (strpos($platform_name, 'directv') !== false) return 'fas fa-satellite';
    if (strpos($platform_name, 'movistar') !== false) return 'fas fa-wifi';
    if (strpos($platform_name, 'flujo tv') !== false) return 'fas fa-stream';
    if (strpos($platform_name, 'viki') !== false) return 'fas fa-heart';
    return 'fas fa-globe';
}

function getMasterAccountStatusBadge(int $estado, ?string $fecha_vencimiento): string {
    if ($estado == 2) {
        return '<span class="badge bg-danger-soft text-danger-emphasis rounded-pill px-3 py-2" title="Esta cuenta tiene problemas reportados en las notas."><i class="fas fa-triangle-exclamation me-1"></i>Con Problemas</span>';
    }
    if ($estado == 0) {
        return '<span class="badge bg-dark-soft text-dark-emphasis rounded-pill px-3 py-2">Inactiva</span>';
    }
    if ($fecha_vencimiento) {
        $is_past = (new DateTime()) > (new DateTime($fecha_vencimiento));
        if ($is_past) {
            return '<span class="badge bg-danger-soft text-danger-emphasis rounded-pill px-3 py-2">Vencida (Prov.)</span>';
        }
    }
    return '<span class="badge bg-success-soft text-success-emphasis rounded-pill px-3 py-2">Activa</span>';
}

// --- Fin funciones auxiliares ---

// --- Función helper para verificar permisos ---
function puedeEditarPerfil($rol_usuario, $id_usuario_perfil, $id_usuario_sesion): bool {
    if ($rol_usuario === 'superadmin' || $rol_usuario === 'admin') {
        return true;
    }
    if ($rol_usuario === 'ventas') {
        return true;
    }
    if ($rol_usuario === 'cortes') {
        return false;
    }
    return false;
}

function puedeEliminarPerfil($rol_usuario): bool {
    if ($rol_usuario === 'superadmin' || $rol_usuario === 'admin') {
        return true;
    }
    if ($rol_usuario === 'cortes') {
        return true;
    }
    if ($rol_usuario === 'ventas') {
        return false;
    }
    return false;
}

function puedeRenovarPerfil($rol_usuario): bool {
    if ($rol_usuario === 'superadmin' || $rol_usuario === 'admin') {
        return true;
    }
    if ($rol_usuario === 'ventas') {
        return true;
    }
    if ($rol_usuario === 'cortes') {
        return false;
    }
    return false;
}

function puedeVerAlertas($rol_usuario): bool {
    if ($rol_usuario === 'superadmin' || $rol_usuario === 'admin') {
        return true;
    }
    if ($rol_usuario === 'ventas') {
        return true;
    }
    if ($rol_usuario === 'cortes') {
        return true;
    }
    return false;
}

function puedeCrearPerfil($rol_usuario): bool {
    if ($rol_usuario === 'superadmin' || $rol_usuario === 'admin') {
        return true;
    }
    if ($rol_usuario === 'ventas') {
        return true;
    }
    if ($rol_usuario === 'cortes') {
        return false;
    }
    return false;
}

function puedeCobrarPerfil($rol_usuario): bool {
    if ($rol_usuario === 'superadmin' || $rol_usuario === 'admin') {
        return true;
    }
    if ($rol_usuario === 'ventas') {
        return true;
    }
    if ($rol_usuario === 'cortes') {
        return false;
    }
    return false;
}

// --- Fin funciones helper de permisos ---


// --- 2. LÓGICA DE FILTROS ---
// IMPORTANTE: Se muestran TODOS los perfiles vendidos por defecto para poder identificarlos
// incluyendo los que tienen cuentas maestras inactivas o con problemas
$where_clauses_pv = [];
$params_pv = [];
$filtros_aplicados_pv = false;

// Agregar filtro de usuario solo para roles limitados
// superadmin, admin, ventas y cortes pueden ver todas las ventas
// otros roles solo ven sus propias ventas
if (!in_array($rol_usuario_sesion, ['superadmin', 'admin', 'ventas', 'cortes'])) {
    $where_clauses_pv[] = "pv.id_usuario = :id_usuario_sesion";
    $params_pv[':id_usuario_sesion'] = $id_usuario_sesion;
}


// Filtros directos
if (isset($_GET['filtro_cliente']) && trim($_GET['filtro_cliente']) !== '') {
    $where_clauses_pv[] = "pv.id_cliente = :id_cliente";
    $params_pv[':id_cliente'] = $_GET['filtro_cliente'];
    $filtros_aplicados_pv = true;
}
if (isset($_GET['filtro_plataforma']) && trim($_GET['filtro_plataforma']) !== '') {
    $where_clauses_pv[] = "p.id_plataforma = :id_plataforma";
    $params_pv[':id_plataforma'] = $_GET['filtro_plataforma'];
    $filtros_aplicados_pv = true;
}
if (isset($_GET['filtro_estado']) && trim($_GET['filtro_estado']) !== '') {
    $estado_suscripcion = trim($_GET['filtro_estado']);
    
    if ($estado_suscripcion === '0') {
        // Filtro para perfiles vencidos (estado = 0 o fecha_fin_servicio < CURDATE())
        $where_clauses_pv[] = "(pv.estado_suscripcion = 0 OR (pv.estado_suscripcion = 1 AND pv.fecha_fin_servicio < CURDATE()))";
    } elseif ($estado_suscripcion === 'por_vencer') {
        // Filtro para perfiles por vencer (estado = 1 y fecha_fin_servicio <= 7 días)
        $where_clauses_pv[] = "(pv.estado_suscripcion = 1 AND DATEDIFF(pv.fecha_fin_servicio, CURDATE()) >= 0 AND DATEDIFF(pv.fecha_fin_servicio, CURDATE()) <= 7)";
    } else {
        // Otros estados (activa = 1, cancelada = 2)
        $where_clauses_pv[] = "pv.estado_suscripcion = :estado_suscripcion";
        $params_pv[':estado_suscripcion'] = $estado_suscripcion;
    }
    
    $filtros_aplicados_pv = true;
}
// NOTA: Se eliminó el filtro por defecto que ocultaba perfiles cancelados
// Ahora se muestran TODOS los perfiles vendidos para poder identificarlos
if (isset($_GET['filtro_estado_cm']) && $_GET['filtro_estado_cm'] !== '') {
    $estado_cm = $_GET['filtro_estado_cm'];
    if ($estado_cm === 'vencida') {
        $where_clauses_pv[] = "cm.estado_cuenta = 1 AND cm.fecha_vencimiento_proveedor < CURDATE()";
    } else {
        $where_clauses_pv[] = "cm.estado_cuenta = :estado_cuenta_maestra";
        $params_pv[':estado_cuenta_maestra'] = $estado_cm;
    }
    $filtros_aplicados_pv = true;
}
if (isset($_GET['filtro_tipo_cuenta']) && trim($_GET['filtro_tipo_cuenta']) !== '') {
    $tipo = strtoupper(trim($_GET['filtro_tipo_cuenta']));
    if ($tipo === 'PERFIL') {
        $where_clauses_pv[] = "p.nombre_plataforma LIKE '%PERFIL%'";
    } elseif ($tipo === 'COMPLETA') {
        $where_clauses_pv[] = "p.nombre_plataforma LIKE '%COMPLETA%'";
    }
    $filtros_aplicados_pv = true;
}
if (isset($_GET['filtro_vendedor']) && trim($_GET['filtro_vendedor']) !== '') {
    $where_clauses_pv[] = "pv.id_usuario = :id_vendedor";
    $params_pv[':id_vendedor'] = $_GET['filtro_vendedor'];
    $filtros_aplicados_pv = true;
}

// LÓGICA DE FILTRADO MEJORADA para búsqueda flexible de teléfonos
if (isset($_GET['buscar']) && trim($_GET['buscar']) !== '') {
    $buscar_texto = trim($_GET['buscar']);
    $buscar_param = '%' . $buscar_texto . '%';
    
    // Función helper para normalizar números de teléfono
    function normalizarTelefono($telefono) {
        // Eliminar espacios, guiones, paréntesis, + y otros símbolos
        $telefono_limpio = preg_replace('/[\s\-\(\)\+\.]/', '', $telefono);
        // Eliminar código de país si empieza con 51 (Perú)
        if (strpos($telefono_limpio, '51') === 0 && strlen($telefono_limpio) > 9) {
            $telefono_limpio = substr($telefono_limpio, 2);
        }
        return $telefono_limpio;
    }
    
    // Normalizar el término de búsqueda si parece ser un número
    $buscar_normalizado = $buscar_texto;
    if (preg_match('/[\d\s\-\(\)\+\.]/', $buscar_texto)) {
        $buscar_normalizado = normalizarTelefono($buscar_texto);
    }
    
    // Búsqueda flexible: texto normal + teléfono normalizado
    $where_clauses_pv[] = "(c.nombre_completo LIKE :buscar_cliente 
                           OR p.nombre_plataforma LIKE :buscar_plataforma 
                           OR cm.email_cuenta LIKE :buscar_email 
                           OR pv.nombre_perfil_cliente LIKE :buscar_perfil 
                           OR c.telefono LIKE :buscar_telefono
                           OR REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(c.telefono, ' ', ''), '-', ''), '(', ''), ')', ''), '+', '') LIKE :buscar_telefono_normalizado)";
    
    $params_pv[':buscar_cliente'] = $buscar_param;
    $params_pv[':buscar_plataforma'] = $buscar_param;
    $params_pv[':buscar_perfil'] = $buscar_param;
    $params_pv[':buscar_email'] = $buscar_param;
    $params_pv[':buscar_telefono'] = $buscar_param;
    $params_pv[':buscar_telefono_normalizado'] = '%' . $buscar_normalizado . '%';
    
    // NO activar filtros_aplicados_pv solo por búsqueda básica
    // Solo se activará si hay otros filtros avanzados
}

// Filtros de fecha
if (isset($_GET['filtro_fecha_inicio']) && trim($_GET['filtro_fecha_inicio']) !== '') {
    $where_clauses_pv[] = "pv.fecha_inicio_servicio >= :fecha_inicio";
    $params_pv[':fecha_inicio'] = $_GET['filtro_fecha_inicio'];
    $filtros_aplicados_pv = true;
}

if (isset($_GET['filtro_fecha_fin']) && trim($_GET['filtro_fecha_fin']) !== '') {
    $where_clauses_pv[] = "pv.fecha_fin_servicio <= :fecha_fin";
    $params_pv[':fecha_fin'] = $_GET['filtro_fecha_fin'];
    $filtros_aplicados_pv = true;
}

// Filtros de precio
if (isset($_GET['filtro_precio_min']) && trim($_GET['filtro_precio_min']) !== '') {
    $where_clauses_pv[] = "pv.precio_venta >= :precio_min";
    $params_pv[':precio_min'] = floatval($_GET['filtro_precio_min']);
    $filtros_aplicados_pv = true;
}

if (isset($_GET['filtro_precio_max']) && trim($_GET['filtro_precio_max']) !== '') {
    $where_clauses_pv[] = "pv.precio_venta <= :precio_max";
    $params_pv[':precio_max'] = floatval($_GET['filtro_precio_max']);
    $filtros_aplicados_pv = true;
}

// Lógica de ordenamiento
$orden_por = $_GET['ordenar_por'] ?? 'fecha_inicio_servicio';
$orden = $_GET['orden'] ?? 'DESC';

// Validar campos de ordenamiento permitidos
$campos_ordenamiento_permitidos = [
    'fecha_inicio_servicio', 'fecha_fin_servicio', 'precio_venta', 
    'nombre_completo', 'nombre_plataforma', 'estado_suscripcion'
];

if (!in_array($orden_por, $campos_ordenamiento_permitidos)) {
    $orden_por = 'fecha_inicio_servicio';
}

if (!in_array(strtoupper($orden), ['ASC', 'DESC'])) {
    $orden = 'DESC';
}

// Mapear campos de ordenamiento a nombres de columna SQL
$mapeo_campos_ordenamiento = [
    'fecha_inicio_servicio' => 'pv.fecha_inicio_servicio',
    'fecha_fin_servicio' => 'pv.fecha_fin_servicio',
    'precio_venta' => 'pv.precio_venta',
    'nombre_completo' => 'c.nombre_completo',
    'nombre_plataforma' => 'p.nombre_plataforma',
    'estado_suscripcion' => 'pv.estado_suscripcion'
];

$campo_ordenamiento_sql = $mapeo_campos_ordenamiento[$orden_por] ?? 'pv.fecha_inicio_servicio';


$sql_where_pv = $where_clauses_pv ? ' WHERE ' . implode(' AND ', $where_clauses_pv) : '';

// DEBUG DE FILTROS - MOVIDO DESPUÉS DE LAS CONSULTAS PARA EVITAR VARIABLES NO DEFINIDAS

// --- Conteo total para estadísticas ---
$sql_count = "SELECT COUNT(*) AS total
             FROM perfiles_vendidos pv
             JOIN clientes c ON pv.id_cliente = c.id_cliente
             JOIN cuentas_maestras cm ON pv.id_cuenta_maestra = cm.id_cuenta_maestra
             JOIN plataformas p ON cm.id_plataforma = p.id_plataforma
             LEFT JOIN usuarios u ON pv.id_usuario = u.id_usuario
             $sql_where_pv";
$stmt_count = $pdo->prepare($sql_count);
$stmt_count->execute($params_pv);
$total_perfiles_count = (int)($stmt_count->fetchColumn() ?: 0);

// --- Carga diferida: solo primeros 50 registros inicialmente ---
$registros_por_carga = 50;
if ($filtros_aplicados_pv) {
    $registros_cargados = $total_perfiles_count;
} else {
    $registros_cargados = isset($_GET['cargar_mas']) ? (int)$_GET['cargar_mas'] : $registros_por_carga;
}

// --- Acción masiva: activar todos los vencidos de la vista actual ---
if (isset($_GET['bulk']) && $_GET['bulk'] === 'activar') {
    $sql_bulk = "UPDATE perfiles_vendidos pv
                 JOIN clientes c ON pv.id_cliente = c.id_cliente
                 JOIN cuentas_maestras cm ON pv.id_cuenta_maestra = cm.id_cuenta_maestra
                 JOIN plataformas p ON cm.id_plataforma = p.id_plataforma
                 LEFT JOIN usuarios u ON pv.id_usuario = u.id_usuario
                 SET pv.estado_suscripcion = 1, pv.fecha_actualizacion = CURRENT_TIMESTAMP
                 $sql_where_pv AND pv.estado_suscripcion = 0";
    $stmt_bulk = $pdo->prepare($sql_bulk);
    $stmt_bulk->execute($params_pv);
}

// --- 3. CONSULTA PRINCIPAL ---
$sql_perfiles = "
    SELECT
        pv.id_perfil_vendido, pv.id_cliente, pv.id_cuenta_maestra, pv.nombre_perfil_cliente,
        pv.pin_perfil_cliente, pv.fecha_inicio_servicio, pv.fecha_fin_servicio, pv.precio_venta,
        pv.estado_suscripcion, pv.notas_suscripcion, pv.id_usuario,
        c.nombre_completo AS nombre_cliente,
        c.notas_cliente,
        c.telefono AS cliente_telefono,
        cm.email_cuenta AS email_cuenta_maestra,
        cm.password_cuenta AS password_cuenta_maestra,
        cm.notas_cuenta AS notas_cuenta_maestra,
        cm.estado_cuenta,
        cm.fecha_vencimiento_proveedor,
        cm.perfiles_disponibles,
        p.nombre_plataforma,
        p.max_perfiles_cuenta,
        DATEDIFF(pv.fecha_fin_servicio, CURDATE()) AS dias_restantes_sql,
        u.nombre_completo AS nombre_vendedor
    FROM perfiles_vendidos pv
    JOIN clientes c ON pv.id_cliente = c.id_cliente
    JOIN cuentas_maestras cm ON pv.id_cuenta_maestra = cm.id_cuenta_maestra
    JOIN plataformas p ON cm.id_plataforma = p.id_plataforma
    LEFT JOIN usuarios u ON pv.id_usuario = u.id_usuario
    $sql_where_pv
    ORDER BY
        -- Prioridad 1: Cancelados (últimos), vencidos primero, por vencer, activos
        CASE
            WHEN pv.estado_suscripcion = 2 THEN 4
            WHEN DATEDIFF(pv.fecha_fin_servicio, CURDATE()) <= 0 THEN 1
            WHEN DATEDIFF(pv.fecha_fin_servicio, CURDATE()) <= 7 THEN 2
            ELSE 3
        END ASC,
        -- Prioridad 2: Días restantes (vencidos primero, más días vencidos primero)
        DATEDIFF(pv.fecha_fin_servicio, CURDATE()) ASC,
        -- Prioridad 3: Campo de ordenamiento seleccionado por el usuario
        $campo_ordenamiento_sql $orden
    LIMIT $registros_cargados
";
$stmt_perfiles = $pdo->prepare($sql_perfiles);

// Debug de la ejecución de la consulta
if (isset($_GET['debug'])) {
    try {
        $stmt_perfiles->execute($params_pv);
        $perfiles = $stmt_perfiles->fetchAll(PDO::FETCH_ASSOC);
        echo "<div class='alert alert-info border-0 shadow-sm mb-3'>";
        echo "<small><strong>✅ Consulta ejecutada exitosamente</strong></small><br>";
        echo "<small><strong>Filas encontradas:</strong> " . count($perfiles) . "</small>";
        echo "</div>";
    } catch (PDOException $e) {
        echo "<div class='alert alert-danger border-0 shadow-sm mb-3'>";
        echo "<small><strong>❌ Error en la consulta:</strong></small><br>";
        echo "<small>" . htmlspecialchars($e->getMessage()) . "</small>";
        echo "</div>";
        $perfiles = [];
    }
} else {
    $stmt_perfiles->execute($params_pv);
    $perfiles = $stmt_perfiles->fetchAll(PDO::FETCH_ASSOC);
}


// --- 4. CONSULTAS PARA DROPDOWNS DE FILTROS ---
$sql_clientes_filtro = "SELECT id_cliente, nombre_completo
                        FROM clientes
                        WHERE estado_cliente = 1
                        ORDER BY nombre_completo ASC";
$params_clientes_filtro = [];
$stmt_clientes_filtro = $pdo->prepare($sql_clientes_filtro);
$stmt_clientes_filtro->execute($params_clientes_filtro);
$clientes_filtro = $stmt_clientes_filtro->fetchAll(PDO::FETCH_ASSOC);

$plataformas_filtro = $pdo->query("SELECT id_plataforma, nombre_plataforma FROM plataformas WHERE estado = 1 ORDER BY nombre_plataforma ASC")->fetchAll(PDO::FETCH_ASSOC);

$vendedores_filtro = [];
$sql_vendedores = "SELECT DISTINCT pv.id_usuario, u.nombre_completo
                   FROM perfiles_vendidos pv
                   JOIN usuarios u ON pv.id_usuario = u.id_usuario
                   ORDER BY u.nombre_completo ASC";
$vendedores_filtro = $pdo->query($sql_vendedores)->fetchAll(PDO::FETCH_ASSOC);

$estados_suscripcion_filtro = [ "1" => "Activa", "0" => "Vencida", "2" => "Cancelada", "por_vencer" => "Por Vencer" ];
$estados_cm_filtro = [ "1" => "Activa", "0" => "Inactiva", "2" => "Con Problemas", "vencida" => "Vencida (Proveedor)" ];
$tipos_cuenta_filtro = ['Perfil' => 'Perfil', 'Completa' => 'Completa'];

// --- Carga de configuraciones ---
$dias_alerta_clientes_global  = (int)getConfigValue('dias_alerta_clientes',  $OWNER_ID, 2);
$dias_alerta_proveedor_global = (int)getConfigValue('dias_alerta_maestras', $OWNER_ID, 2);
$moneda_simbolo               = getConfigValue('moneda_simbolo',            $OWNER_ID, 'S/');


// Manejo del mensaje post-compra (si viene de perfiles_vendidos_crear)
$mensaje_para_copiar_al_cliente = null;
if (isset($_SESSION['mensaje_post_compra_para_copiar'])) {
    $mensaje_para_copiar_al_cliente = $_SESSION['mensaje_post_compra_para_copiar'];
    unset($_SESSION['mensaje_post_compra_para_copiar']);
}

$telefono_cliente_para_whatsapp = null;
if (isset($_SESSION['telefono_cliente_para_whatsapp'])) {
    $telefono_cliente_para_whatsapp = $_SESSION['telefono_cliente_para_whatsapp'];
    unset($_SESSION['telefono_cliente_para_whatsapp']);
}

// --- Lógica de Alertas MEJORADA - CONSULTAS INDEPENDIENTES ---
$alerts_data = [
    'clientes_vencidos' => [],
    'clientes_proximos' => [],
    'cuentas_maestras_proximas' => []
];

// CONSULTA INDEPENDIENTE para Cuentas Maestras por vencer
$sql_alertas_cm = "SELECT id_cuenta_maestra, email_cuenta, fecha_vencimiento_proveedor,
                    DATEDIFF(fecha_vencimiento_proveedor, CURDATE()) as dias_restantes_cm
                    FROM cuentas_maestras
                    WHERE estado_cuenta = 1
                      AND fecha_vencimiento_proveedor BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL :dias_alerta DAY)
                    ORDER BY dias_restantes_cm ASC";

$params_alertas_cm = [
    ':dias_alerta'=> $dias_alerta_proveedor_global
];
$stmt_alertas_cm = $pdo->prepare($sql_alertas_cm);
$stmt_alertas_cm->execute($params_alertas_cm);
$alerts_data['cuentas_maestras_proximas'] = $stmt_alertas_cm->fetchAll(PDO::FETCH_ASSOC);

// CONSULTA INDEPENDIENTE para TODOS los perfiles vencidos (sin límite de paginación)
$sql_perfiles_vencidos = "SELECT pv.id_perfil_vendido, pv.nombre_perfil_cliente, pv.fecha_fin_servicio, 
                          pv.precio_venta, pv.estado_suscripcion,
                          c.nombre_completo AS nombre_cliente,
                          c.telefono AS cliente_telefono,
                          p.nombre_plataforma,
                          DATEDIFF(pv.fecha_fin_servicio, CURDATE()) as dias_restantes_sql
                          FROM perfiles_vendidos pv
                          JOIN clientes c ON pv.id_cliente = c.id_cliente
                          JOIN cuentas_maestras cm ON pv.id_cuenta_maestra = cm.id_cuenta_maestra
                          JOIN plataformas p ON cm.id_plataforma = p.id_plataforma
                          WHERE pv.estado_suscripcion IN (0, 1)
                            AND DATEDIFF(pv.fecha_fin_servicio, CURDATE()) <= 0
                          ORDER BY ABS(DATEDIFF(pv.fecha_fin_servicio, CURDATE())) ASC";

$stmt_perfiles_vencidos = $pdo->prepare($sql_perfiles_vencidos);
$stmt_perfiles_vencidos->execute();
$alerts_data['clientes_vencidos'] = $stmt_perfiles_vencidos->fetchAll(PDO::FETCH_ASSOC);

// CONSULTA INDEPENDIENTE para TODOS los perfiles por vencer (sin límite de paginación)
$sql_perfiles_proximos = "SELECT pv.id_perfil_vendido, pv.nombre_perfil_cliente, pv.fecha_fin_servicio, 
                          pv.precio_venta, pv.estado_suscripcion,
                          c.nombre_completo AS nombre_cliente,
                          c.telefono AS cliente_telefono,
                          p.nombre_plataforma,
                          DATEDIFF(pv.fecha_fin_servicio, CURDATE()) as dias_restantes_sql
                          FROM perfiles_vendidos pv
                          JOIN clientes c ON pv.id_cliente = c.id_cliente
                          JOIN cuentas_maestras cm ON pv.id_cuenta_maestra = cm.id_cuenta_maestra
                          JOIN plataformas p ON cm.id_plataforma = p.id_plataforma
                          WHERE pv.estado_suscripcion = 1
                            AND DATEDIFF(pv.fecha_fin_servicio, CURDATE()) > 0
                            AND DATEDIFF(pv.fecha_fin_servicio, CURDATE()) <= :dias_alerta_clientes
                          ORDER BY DATEDIFF(pv.fecha_fin_servicio, CURDATE()) ASC";

$params_perfiles_proximos = [':dias_alerta_clientes' => $dias_alerta_clientes_global];
$stmt_perfiles_proximos = $pdo->prepare($sql_perfiles_proximos);
$stmt_perfiles_proximos->execute($params_perfiles_proximos);
$alerts_data['clientes_proximos'] = $stmt_perfiles_proximos->fetchAll(PDO::FETCH_ASSOC);

$has_alerts = !empty($alerts_data['clientes_vencidos']) || !empty($alerts_data['clientes_proximos']) || !empty($alerts_data['cuentas_maestras_proximas']);

// DEBUG DE FILTROS - AHORA DESPUÉS DE QUE TODAS LAS VARIABLES ESTÉN DEFINIDAS
if (!empty($_GET) && (isset($_GET['debug']) || count($_GET) > 1 || !isset($_GET['mensaje']))) {
    $parametros_relevantes = array_diff_key($_GET, ['mensaje' => '', 'debug' => '']);
    if (!empty($parametros_relevantes)) {
        echo "<div class='alert alert-success border-0 shadow-sm mb-3'>";
        echo "<div class='d-flex align-items-center'>";
        echo "<i class='fas fa-filter text-success me-2'></i>";
        // Mensaje de filtros preservados eliminado - Sistema funcionando correctamente
        echo "</div>";
        
        if (isset($_GET['debug'])) {
            echo "<hr class='my-2'>";
            // Parámetros preservados - Información eliminada del debug
            echo "<br><small><strong>Filtros aplicados:</strong> " . ($filtros_aplicados_pv ? 'SÍ' : 'NO') . "</small>";
            echo "<br><small><strong>Total resultados:</strong> " . $total_perfiles_count . "</small>";
            echo "<br><small><strong>Cláusulas WHERE:</strong> " . count($where_clauses_pv) . "</small>";
            echo "<br><small><strong>Parámetros SQL:</strong> " . count($params_pv) . "</small>";
            echo "<br><small><strong>SQL WHERE:</strong> " . htmlspecialchars($sql_where_pv) . "</small>";
            
            // Debug detallado de cada filtro
            echo "<hr class='my-2'>";
            echo "<small><strong>Debug Detallado de Filtros:</strong></small><br>";
            echo "<small>buscar: '" . ($_GET['buscar'] ?? 'NO') . "' → " . (!empty($_GET['buscar']) ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            echo "<small>filtro_estado: '" . ($_GET['filtro_estado'] ?? 'NO') . "' → " . (isset($_GET['filtro_estado']) && $_GET['filtro_estado'] !== '' ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            echo "<small>filtro_estado_cm: '" . ($_GET['filtro_estado_cm'] ?? 'NO') . "' → " . (isset($_GET['filtro_estado_cm']) && $_GET['filtro_estado_cm'] !== '' ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            echo "<small>filtro_tipo_cuenta: '" . ($_GET['filtro_tipo_cuenta'] ?? 'NO') . "' → " . (isset($_GET['filtro_tipo_cuenta']) && $_GET['filtro_tipo_cuenta'] !== '' ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            echo "<small>filtro_vendedor: '" . ($_GET['filtro_vendedor'] ?? 'NO') . "' → " . (isset($_GET['filtro_vendedor']) && $_GET['filtro_vendedor'] !== '' ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            echo "<small>filtro_fecha_inicio: '" . ($_GET['filtro_fecha_inicio'] ?? 'NO') . "' → " . (isset($_GET['filtro_fecha_inicio']) && $_GET['filtro_fecha_inicio'] !== '' ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            echo "<small>filtro_fecha_fin: '" . ($_GET['filtro_fecha_fin'] ?? 'NO') . "' → " . (isset($_GET['filtro_fecha_fin']) && $_GET['filtro_fecha_fin'] !== '' ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            echo "<small>filtro_precio_min: '" . ($_GET['filtro_precio_min'] ?? 'NO') . "' → " . (isset($_GET['filtro_precio_min']) && $_GET['filtro_precio_min'] !== '' ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            echo "<small>filtro_precio_max: '" . ($_GET['filtro_precio_max'] ?? 'NO') . "' → " . (isset($_GET['filtro_precio_max']) && $_GET['filtro_precio_max'] !== '' ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            echo "<small>ordenar_por: '" . ($_GET['ordenar_por'] ?? 'NO') . "' → " . (isset($_GET['ordenar_por']) && $_GET['ordenar_por'] !== '' ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            echo "<small>orden: '" . ($_GET['orden'] ?? 'NO') . "' → " . (isset($_GET['orden']) && $_GET['orden'] !== '' ? 'APLICADO' : 'NO APLICADO') . "</small><br>";
            
            // Debug de la consulta SQL completa
            echo "<hr class='my-2'>";
            echo "<small><strong>Consulta SQL Completa:</strong></small><br>";
            echo "<small><code>" . htmlspecialchars($sql_perfiles) . "</code></small><br><br>";
            echo "<small><strong>Parámetros de la consulta:</strong></small><br>";
            echo "<small><pre>" . print_r($params_pv, true) . "</pre></small>";
            
            // Debug del resultado de la consulta
            echo "<hr class='my-2'>";
            echo "<small><strong>Resultado de la consulta:</strong></small><br>";
            echo "<small><strong>Total de perfiles encontrados:</strong> " . count($perfiles) . "</small><br>";
            if (!empty($perfiles)) {
                echo "<small><strong>Primer perfil:</strong> " . htmlspecialchars($perfiles[0]['nombre_perfil_cliente'] ?? 'N/A') . "</small><br>";
                echo "<small><strong>Cliente:</strong> " . htmlspecialchars($perfiles[0]['nombre_cliente'] ?? 'N/A') . "</small><br>";
                echo "<small><strong>Plataforma:</strong> " . htmlspecialchars($perfiles[0]['nombre_plataforma'] ?? 'N/A') . "</small><br>";
            } else {
                echo "<small><strong>No se encontraron perfiles</strong></small><br>";
            }
        }
        echo "</div>";
    }
}

?>

<style>
    /* ==========================================================================
        DigitalPremium - Estilos V9 (Diseño de Alto Impacto y Refinado)
        CSS auto-contenido en perfiles_vendidos.php
        ========================================================================== */
    :root {
        --primary-hue: 235;
        --primary-color: hsl(var(--primary-hue), 65%, 60%);
        --primary-color-dark: hsl(var(--primary-hue), 60%, 50%);
        --primary-bg-subtle: hsl(var(--primary-hue), 65%, 97%);
        --primary-text-emphasis: hsl(var(--primary-hue), 50%, 35%);

        --secondary-color: #6c757d;
        --success-hue: 150;
        --success-color: hsl(var(--success-hue), 55%, 45%);
        --success-bg-subtle: hsl(var(--success-hue), 55%, 96%);
        --success-text-emphasis: hsl(var(--success-hue), 50%, 30%);

        --info-hue: 195;
        --info-color: hsl(var(--info-hue), 75%, 50%);
        --info-bg-subtle: hsl(var(--info-hue), 75%, 96%);
        --info-text-emphasis: hsl(var(--info-hue), 65%, 35%);
        
        --warning-hue: 40;
        --warning-color: hsl(var(--warning-hue), 95%, 55%);
        --warning-bg-subtle: hsl(var(--warning-hue), 95%, 96%);
        --warning-text-emphasis: hsl(var(--warning-hue), 80%, 35%);

        --danger-hue: 355;
        --danger-color: hsl(var(--danger-hue), 70%, 55%);
        --danger-bg-subtle: hsl(var(--danger-hue), 70%, 96%);
        --danger-text-emphasis: hsl(var(--danger-hue), 60%, 35%);
        
        --content-bg: #ffffff;
        --body-bg: #eef2f6;

        --text-dark: #212529;
        --text-muted: #6c757d;
        --text-muted-light: #aab2bd;

        --dark-hue: 210;
        --dark-bg-subtle: hsl(var(--dark-hue), 10%, 95%);
        --dark-text-emphasis: hsl(var(--dark-hue), 10%, 10%);

        --border-radius-sm: 0.375rem;
        --border-radius-md: 0.5rem;
        --border-radius-lg: 1rem;
        --border-color: #e5e7eb;

        --box-shadow-sm: 0 2px 8px rgba(0,0,0,0.06);
        --box-shadow-md: 0 5px 18px rgba(0,0,0,0.08);
        --box-shadow-lg: 0 10px 30px -5px rgba(0,0,0,0.15);

        --transition-base: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
    }
    .bg-success-soft { background-color: var(--success-bg-subtle) !important; } .text-success-emphasis { color: var(--success-text-emphasis) !important; }
    .bg-danger-soft { background-color: var(--danger-bg-subtle) !important; } .text-danger-emphasis { color: var(--danger-text-emphasis) !important; }
    .bg-warning-soft { background-color: var(--warning-bg-subtle) !important; } .text-warning-emphasis { color: var(--warning-text-emphasis) !important; }
    .bg-info-soft { background-color: var(--info-bg-subtle) !important; } .text-info-emphasis { color: var(--info-text-emphasis) !important; }
    .bg-dark-soft { background-color: var(--dark-bg-subtle) !important; } .text-dark-emphasis { color: var(--dark-text-emphasis) !important; }

    /* -------------------------------------------------------------------------- */
    /* Estilos de Tarjeta de Suscripción (Rediseño)
    /* -------------------------------------------------------------------------- */
    .subscription-card {
        border: none;
        border-radius: var(--border-radius-lg);
        background-color: var(--content-bg);
        box-shadow: var(--box-shadow-md);
        transition: var(--transition-base);
        overflow: hidden;
        height: 100%;
        display: flex;
        flex-direction: column;
    }
    .subscription-card:hover {
        transform: translateY(-8px);
        box-shadow: var(--box-shadow-md);
    }
    .card-glow-warning { box-shadow: 0 0 0 4px hsla(var(--warning-hue), 95%, 55%, 0.5), var(--box-shadow-md); }
    .card-glow-danger { box-shadow: 0 0 0 4px hsla(var(--danger-hue), 70%, 55%, 0.5), var(--box-shadow-md); }

    .subscription-card .card-header {
        padding: 1rem 1.5rem;
        border-bottom: 1px solid rgba(255,255,255,0.2);
        font-weight: 600;
        color: white;
    }
    .subscription-card .card-header h5 { font-weight: 700; display: flex; align-items: center; margin-bottom: 0; flex-grow: 1; color: inherit; }
    .subscription-card .platform-icon { font-size: 1.6rem; margin-right: 0.85rem; text-shadow: 0 1px 3px rgba(0,0,0,0.25); }

    /* Colores de Encabezado de Plataforma con Degradado */
    .header-netflix { background: linear-gradient(135deg, #E50914, #B20710); }
    .header-disney { background: linear-gradient(135deg, #1A2A4B, #00123D); }
    .header-prime { background: linear-gradient(135deg, #00A8E1, #0077B5); }
    .header-hbo { background: linear-gradient(135deg, #6C2B80, #4D1F5C); }
    .header-spotify { background: linear-gradient(135deg, #1DB954, #1ED760); }
    .header-crunchyroll { background: linear-gradient(135deg, #F47521, #E8630C); }
    .header-canva { background: linear-gradient(135deg, #00C4CC, #008D94); }
    .header-paramount { background: linear-gradient(135deg, #0064FF, #003C99); }
    .header-iptv { background: linear-gradient(135deg, #6a1b9a, #4a148c); }
    .header-capcut { background: linear-gradient(135deg, #16C79A, #0FA981); }
    .header-directv { background: linear-gradient(135deg, #0199d6, #006b94); }
    .header-movistar { background: linear-gradient(135deg, #147ee5, #0056b3); }
    .header-flujo { background: linear-gradient(135deg, #FF5722, #E64A19); }
    .header-viki { background: linear-gradient(135deg, #FF005D, #D3004D); }
    .header-youtube { background: linear-gradient(135deg, #FF0000, #CC0000); }
    .header-default { background: linear-gradient(135deg, #6c757d, #495057); }

    .subscription-card .btn-renew {
        font-size: 0.8rem; padding: 0.3rem 0.7rem; font-weight: 600;
        background-color: rgba(255,255,255,0.95) !important; color: #333 !important;
        border: none !important; transition: var(--transition-base); box-shadow: var(--box-shadow-sm);
    }
    .subscription-card .btn-renew:hover { transform: scale(1.08); background-color: white !important; }

    .subscription-card .card-body { padding: 1.5rem; flex-grow: 1; }

    .card-section-title {
        font-size: 0.8rem; font-weight: 700; color: var(--text-muted);
        text-transform: uppercase; letter-spacing: 0.05em; margin-bottom: 1rem;
        padding-bottom: 0.5rem; border-bottom: 1px solid var(--border-color);
    }
    .detail-item { display: flex; justify-content: space-between; align-items: center; padding: 0.6rem 0; font-size: 0.9rem; }
    .detail-label { color: var(--text-muted); display: flex; align-items: center; gap: 0.75rem; }
    .detail-label i { font-size: 1em; width: 20px; text-align: center; color: var(--text-muted-light); }
    .detail-value { font-weight: 500; text-align: right; }
    .detail-actions { display: flex; align-items: center; gap: 0.75rem; }

    .copy-btn, .toggle-password-btn {
        cursor: pointer; transition: color 0.2s, transform 0.1s;
        color: var(--text-muted-light); font-size: 1rem;
    }
    .copy-btn:hover, .toggle-password-btn:hover { color: var(--primary-color); transform: scale(1.1); }
    .copy-btn-header {
        cursor: pointer;
        transition: color 0.2s, transform 0.1s;
        color: rgba(255,255,255,0.7);
        font-size: 0.9em;
        vertical-align: middle;
    }
    .copy-btn-header:hover { color: white; transform: scale(1.15); }

    .detail-actions .form-control-plaintext {
        background: transparent !important; border: none !important; padding: 0;
        font-size: inherit; color: inherit; width: 130px; text-align: right;
        font-family: 'Roboto Mono', monospace; font-weight: 500;
    }

    .notes-toggle {
        cursor: pointer; color: var(--text-muted); font-size: 0.8rem;
        transition: color 0.2s ease; margin-top: 0.5rem; display: inline-block;
    }
    .notes-toggle:hover { color: var(--primary-color); }
    .notes-collapse-content {
        margin-top: 0.75rem; padding: 1rem; background-color: var(--body-bg);
        border-radius: var(--border-radius-sm); font-size: 0.85rem;
        border: 1px solid var(--border-color); word-break: break-word;
    }

    /* --- NUEVOS ESTILOS PARA RE-DISEÑO DEL CUERPO DE LA TARJETA --- */
    .subscription-card .card-body {
        padding: 0;
        display: flex;
        flex-direction: column;
        flex-grow: 1;
    }
    .card-section {
        padding: 1.25rem 1.5rem;
    }
    .customer-details {
        border-bottom: 1px solid var(--border-color);
    }
    .master-account-section {
        margin: 1rem 1.5rem;
        border-radius: var(--border-radius-md);
        padding: 1rem 1.25rem;
        transition: background-color 0.3s ease, border-color 0.3s ease;
        border: 1px solid transparent;
    }
    .master-account-section.bg-danger-soft { border-color: hsla(var(--danger-hue), 70%, 55%, 0.4); }
    .master-account-section.bg-warning-soft { border-color: hsla(var(--warning-hue), 95%, 55%, 0.4); }
    .master-account-section.bg-dark-soft { border-color: hsla(var(--dark-hue), 10%, 23%, 0.2); }

    .master-account-section .card-section-title { margin-top: 0; border-bottom-color: rgba(0,0,0,0.08); }
    .detail-item { border-bottom: 1px dashed var(--border-color); }
    .detail-item:last-child { border-bottom: none; padding-bottom: 0; }
    .card-section .detail-item:first-of-type { padding-top: 0; }
    .master-account-section .detail-item { border-color: rgba(0,0,0,0.05); }

    .progress-section {
        margin-top: auto;
        padding: 0 1.5rem 1.25rem;
    }
    .progress { height: 8px; border-radius: 8px; background-color: var(--border-color); margin-top: 8px; }

    /* -------------------------------------------------------------------------- */
    /* Centro de Notificaciones (Modal de Alertas Rediseñado)
    /* -------------------------------------------------------------------------- */
    #modalAlertas .modal-header {
        background: var(--warning-color);
        color: var(--dark-text-emphasis);
        border-bottom: none;
    }
    #modalAlertas .modal-header .btn-close { filter: none; }
    #modalAlertas .modal-body { background-color: var(--body-bg); padding: 1.5rem; }
    #modalAlertas .modal-footer { background-color: var(--content-bg); }

    /* Resumen de Alertas */
    .alert-summary {
        background: var(--content-bg);
        border-radius: var(--border-radius-lg);
        padding: 1.5rem;
        border: 1px solid var(--border-color);
        box-shadow: var(--box-shadow-sm);
    }
    
    .summary-card {
        display: flex;
        align-items: center;
        gap: 1rem;
        padding: 1rem;
        border-radius: var(--border-radius-md);
        border: 2px solid;
        transition: var(--transition-base);
    }
    
    .summary-card:hover {
        transform: translateY(-2px);
        box-shadow: var(--box-shadow-md);
    }
    
    .summary-icon {
        flex-shrink: 0;
    }
    
    .summary-content h6 {
        margin: 0;
        font-weight: 600;
        color: var(--text-dark);
    }
    
    .summary-count {
        font-size: 2rem;
        font-weight: 700;
        color: var(--text-dark);
        line-height: 1;
    }

    /* Acordeón de Alertas */
    .accordion-item {
        border-radius: var(--border-radius-md) !important;
        margin-bottom: 1rem;
        overflow: hidden;
        box-shadow: var(--box-shadow-sm);
    }
    
    .accordion-button {
        border: none !important;
        box-shadow: none !important;
        font-weight: 600;
        padding: 1rem 1.5rem;
    }
    
    .accordion-button:not(.collapsed) {
        background-color: inherit !important;
        color: inherit !important;
    }
    
    .accordion-button:focus {
        box-shadow: none !important;
        border-color: transparent !important;
    }
    
    .accordion-button::after {
        background-size: 1.2rem;
        transition: var(--transition-base);
    }
    
    .accordion-button.collapsed::after {
        transform: rotate(-90deg);
    }

    .alert-list {
        list-style: none;
        padding: 1rem;
        margin: 0;
        display: flex;
        flex-direction: column;
        gap: 0.75rem;
        background-color: var(--body-bg);
    }

    .alert-item-card {
        background-color: var(--content-bg);
        border-radius: var(--border-radius-md);
        box-shadow: var(--box-shadow-sm);
        border-left: 5px solid;
        padding: 1rem 1.25rem;
        display: flex;
        align-items: center;
        gap: 1rem;
        transition: var(--transition-base);
        border: 1px solid var(--border-color);
    }
    
    .alert-item-card:hover { 
        transform: translateX(5px); 
        box-shadow: var(--box-shadow-md); 
    }

    .alert-item-card.border-danger { border-left-color: var(--danger-color); }
    .alert-item-card.border-warning { border-left-color: var(--warning-color); }
    .alert-item-card.border-info { border-left-color: var(--info-color); }

    .alert-item-card .alert-icon { 
        font-size: 1.5rem; 
        width: 40px; 
        text-align: center; 
        flex-shrink: 0;
    }

    .alert-item-card .alert-info-text { 
        flex-grow: 1; 
        min-width: 0;
    }
    
    .alert-item-card .alert-info-text strong { 
        font-weight: 600; 
        color: var(--text-dark); 
        font-size: 1rem; 
        display: block;
        margin-bottom: 0.25rem;
    }
    
    .alert-item-card .alert-info-text small { 
        font-size: 0.85rem; 
        color: var(--text-muted); 
        display: block; 
        line-height: 1.4;
    }

    .alert-item-card .alert-actions { 
        display: flex; 
        gap: 0.5rem; 
        flex-shrink: 0; 
        align-items: center;
    }
    
    .alert-item-card .alert-actions .btn { 
        font-size: 0.8rem; 
        padding: 0.4rem 0.8rem; 
        border-radius: 50rem; 
        font-weight: 500; 
        white-space: nowrap;
    }
    
    /* Badges de urgencia */
    .badge {
        font-size: 0.75rem;
        padding: 0.25rem 0.5rem;
        border-radius: 50rem;
        font-weight: 500;
    }
    
    /* Responsive */
    @media (max-width: 768px) {
        .summary-card {
            flex-direction: column;
            text-align: center;
            padding: 1.5rem 1rem;
        }
        
        .alert-item-card {
            flex-direction: column;
            align-items: flex-start;
            gap: 0.75rem;
        }
        
        .alert-item-card .alert-actions {
            align-self: stretch;
            justify-content: center;
        }
    }

    /* Estilos para el modal de renovación */
    #modalRenovarSuscripcion .modal-header { background-color: var(--primary-color); color: white; }
    #modalRenovarSuscripcion .modal-header .btn-close { filter: invert(1) brightness(2); }
    #modalRenovarSuscripcion .info-cliente-display {
        background-color: var(--primary-bg-subtle); padding: 1rem 1.5rem; border-radius: var(--border-radius-md); margin-bottom: 1.5rem; font-weight: 500; color: var(--primary-text-emphasis); border: 1px solid var(--primary-color);
    }
    
    /* Estilos para filtros */
    .filter-card {
        background-color: var(--content-bg); border-radius: var(--border-radius-lg);
        padding: 1.5rem; box-shadow: var(--box-shadow-sm);
        border: 1px solid var(--border-color);
    }
    /* Estilos para etiquetas de filtros */
    .filter-card .form-label { font-size: 0.8rem; font-weight: 600; color: var(--text-muted) !important; text-transform: uppercase; }
    .filter-card .form-select-sm, .filter-card .form-control-sm { font-size: 0.9rem; border-radius: var(--border-radius-md); }
    .filter-card .btn { border-radius: var(--border-radius-md); }

    /* Búsqueda rápida */
    .quick-search-wrapper { position: relative; }
    .quick-search-icon {
        position: absolute; left: 1.5rem; top: 50%;
        transform: translateY(-50%); color: var(--text-muted-light);
        font-size: 1.2rem; pointer-events: none;
    }
    #quickSearchInput { padding-left: 3.5rem !important; border-radius: 50rem; font-size: 1rem; height: auto; }

    /* Accordion for notes inside the card */
    .notes-section-container {
        padding: 0 1.5rem;
    }
    .notes-section-container .accordion-flush .accordion-item {
        border: 1px solid var(--border-color);
        border-radius: var(--border-radius-md) !important; /* Override flush styles */
        margin-bottom: 0.5rem;
        background-color: transparent;
        overflow: hidden; /* To contain the rounded corners */
    }
    .notes-section-container .accordion-flush .accordion-item:last-child {
        margin-bottom: 0;
    }
    .notes-section-container .accordion-header .accordion-button {
        background-color: var(--primary-bg-subtle);
        color: var(--text-muted);
        font-weight: 600;
        font-size: 0.9rem;
        padding: 0.75rem 1.25rem;
        border-radius: 0; /* No radius on button itself */
    }
    .notes-section-container .accordion-header .accordion-button:not(.collapsed) {
        background-color: var(--primary-color);
        color: white;
        box-shadow: none;
    }
    .notes-section-container .accordion-header .accordion-button:not(.collapsed) i { color: white !important; }
    .notes-section-container .accordion-header .accordion-button:focus { box-shadow: 0 0 0 0.2rem hsla(var(--primary-hue), 65%, 60%, 0.25); z-index: 5; }
    .notes-section-container .accordion-header .accordion-button::after { filter: brightness(0.5); }
    .notes-section-container .accordion-header .accordion-button:not(.collapsed)::after { filter: invert(1) brightness(2); }
    .notes-section-container .accordion-body {
        padding: 1rem 1.25rem;
        font-size: 0.85rem;
        background-color: var(--content-bg);
        word-break: break-word;
    }

    /* -------------------------------------------------------------------------- */
    /* Responsive Adjustments
    /* -------------------------------------------------------------------------- */
    @media (max-width: 991.98px) {
        .overdue-client-actions {
            width: 100%;
            justify-content: flex-end;
        }
    }
    @media (max-width: 767.98px) {
        .overdue-client-item {
            flex-direction: column;
            align-items: flex-start;
        }
        .overdue-client-actions {
            margin-top: 0.75rem;
            justify-content: flex-start;
            width: auto;
        }
        .profitability-metrics {
            flex-direction: column;
            align-items: flex-start;
            gap: 0.5rem;
        }
        .profitability-metrics .metric-item strong {
            display: inline-block;
            margin-left: 0.5rem;
        }
        
        .platform-header {
            flex-direction: column;
            text-align: center;
            gap: 1rem;
        }
        
        .platform-header .d-flex {
            flex-direction: column;
            gap: 1rem;
        }
        
        .account-card {
            min-height: 350px;
        }
        
        .account-card .card-header {
            flex-direction: column;
            gap: 0.5rem;
            text-align: center;
        }
    }
    @media (max-width: 575.98px) {
        .kpi-card .card-body { padding: 1rem; flex-direction: column; text-align: center; }
        .kpi-card .kpi-icon-wrapper { margin-right: 0; margin-bottom: 0.8rem; width: 50px; height: 50px; }
        .kpi-card .kpi-icon-wrapper i { font-size: 1.6rem; }
        .kpi-card .kpi-value { font-size: 1.8rem; }
        .kpi-card .kpi-title { font-size: 0.75rem; }
    }

</style>
<div class="container-fluid px-4">
    <header class="page-header d-flex justify-content-between align-items-center mt-5 mb-4">
        <h1 class="h2 page-title fw-bold text-dark"><i class="fas fa-user-check me-2 text-primary"></i>Gestión de Suscripciones</h1>
        <div class="d-flex align-items-center gap-3">
            <?php if (puedeVerAlertas($rol_usuario_sesion)): ?>
            <button type="button" class="btn btn-warning shadow-sm px-4 py-2 rounded-pill alerts-button" data-bs-toggle="modal" data-bs-target="#modalAlertas">
                <i class="fas fa-bell me-2"></i> Alertas
                <?php if ($has_alerts): $total_alerts = count($alerts_data['clientes_vencidos']) + count($alerts_data['clientes_proximos']) + count($alerts_data['cuentas_maestras_proximas']); ?>
                    <span class="badge bg-danger ms-1 alerts-count-badge"><?php echo $total_alerts; ?></span>
                <?php endif; ?>
            </button>
            <?php endif; ?>
            <?php if (puedeCrearPerfil($rol_usuario_sesion)): ?>
            <a href="/digitalpremium/pages/perfiles_vendidos_crear.php" class="btn btn-primary shadow-sm px-4 py-2 rounded-pill">
                <i class="fas fa-plus-circle me-2"></i> Nueva Suscripción
            </a>
            <?php endif; ?>
        </div>
    </header>
    
    <?php if (isset($_GET['mensaje'])): ?>
        <div class="alert alert-<?php echo htmlspecialchars($_GET['tipo'] ?? 'info'); ?> alert-dismissible fade show rounded-3 shadow-sm" role="alert">
            <i class="fas fa-info-circle me-2"></i> <?php echo htmlspecialchars(urldecode($_GET['mensaje'])); ?>
            
            <?php 
            // Mostrar información sobre filtros preservados
            $filtros_activos = [];
            $parametros_filtro = ['buscar', 'filtro_estado', 'filtro_estado_cm', 'filtro_tipo_cuenta', 'filtro_vendedor', 'filtro_fecha_inicio', 'filtro_fecha_fin', 'filtro_precio_min', 'filtro_precio_max', 'ordenar_por', 'orden'];
            foreach ($parametros_filtro as $param) {
                if (isset($_GET[$param]) && $_GET[$param] !== '') {
                    $filtros_activos[] = $param;
                }
            }
            
            if (!empty($filtros_activos)): ?>
                <div class="mt-2">
                    <small class="text-muted">
                        <i class="fas fa-filter me-1"></i>Los filtros de búsqueda se han preservado automáticamente para tu comodidad.
                    </small>
                </div>
            <?php endif; ?>
            
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>

    <div class="card mb-4 shadow-sm rounded-3">
        <div class="card-header bg-light py-3">
            <a class="text-decoration-none text-body fw-semibold d-block" data-bs-toggle="collapse" href="#filtrosPerfilesCollapse" role="button" aria-expanded="<?php echo $filtros_aplicados_pv ? 'true' : 'false'; ?>" aria-controls="filtrosPerfilesCollapse">
                <i class="fas fa-filter me-2 text-secondary"></i> 
                Filtros de Búsqueda
                <?php if ($filtros_aplicados_pv): ?>
                    <span class="badge bg-success text-white ms-2"><?php echo $total_perfiles_count; ?> resultados</span>
                <?php endif; ?>
                <i class="fas fa-chevron-<?php echo $filtros_aplicados_pv ? 'up' : 'down'; ?> float-end ms-2" style="font-size: 0.8rem;"></i>
            </a>
        </div>
        <div class="collapse <?php echo $filtros_aplicados_pv ? 'show' : ''; ?>" id="filtrosPerfilesCollapse">
            <div class="card-body p-3 filter-card">
                <form action="" method="GET" class="row g-3 align-items-end">
                    <!-- Primera fila -->
                    <div class="col-lg-4 col-md-6">
                        <label for="buscar" class="form-label form-label-sm">Buscar (Nombre, Plat., Email, Teléfono):</label>
                        <input type="text" name="buscar" id="buscar" class="form-control form-control-sm rounded-3" value="<?php echo htmlspecialchars($_GET['buscar'] ?? ''); ?>" placeholder="Ej. Juan Pérez, Netflix, +51 927 198 311, 927198311...">
                        <small class="form-text text-muted">
                            <i class="fas fa-info-circle me-1"></i>
                            Búsqueda inteligente: Los números de teléfono se normalizan automáticamente
                        </small>
                    </div>
                    <div class="col-lg-2 col-md-6">
                        <label for="filtro_plataforma" class="form-label form-label-sm">Plataforma:</label>
                        <select name="filtro_plataforma" id="filtro_plataforma" class="form-select form-select-sm rounded-3">
                            <option value="">Todas</option>
                            <?php foreach ($plataformas_filtro as $p):?>
                                <option value="<?php echo $p['id_plataforma']; ?>" <?php echo (($_GET['filtro_plataforma'] ?? '') == $p['id_plataforma']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($p['nombre_plataforma']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-lg-2 col-md-6">
                        <label for="filtro_estado" class="form-label form-label-sm">Estado Suscripción:</label>
                        <select name="filtro_estado" id="filtro_estado" class="form-select form-select-sm rounded-3">
                            <option value="">Todos</option>
                            <?php foreach ($estados_suscripcion_filtro as $v => $t):?>
                                <option value="<?php echo $v; ?>" <?php echo (isset($_GET['filtro_estado']) && $_GET['filtro_estado'] !== '' && $_GET['filtro_estado'] == $v) ? 'selected' : ''; ?>>
                                    <?php echo $t; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-lg-2 col-md-6">
                        <label for="filtro_estado_cm" class="form-label form-label-sm">Estado Cta. Maestra:</label>
                        <select name="filtro_estado_cm" id="filtro_estado_cm" class="form-select form-select-sm rounded-3">
                            <option value="">Todos</option>
                            <?php foreach ($estados_cm_filtro as $v => $t):?>
                                <option value="<?php echo $v; ?>" <?php echo (isset($_GET['filtro_estado_cm']) && $_GET['filtro_estado_cm'] !== '' && $_GET['filtro_estado_cm'] == $v) ? 'selected' : ''; ?>>
                                    <?php echo $t; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- Segunda fila -->
                    <div class="col-lg-2 col-md-6">
                        <label for="filtro_tipo_cuenta" class="form-label form-label-sm">Tipo de Cuenta:</label>
                        <select name="filtro_tipo_cuenta" id="filtro_tipo_cuenta" class="form-select form-select-sm rounded-3">
                            <option value="">Todos</option>
                            <option value="PERFIL" <?php echo (isset($_GET['filtro_tipo_cuenta']) && $_GET['filtro_tipo_cuenta'] === 'PERFIL') ? 'selected' : ''; ?>>Perfil</option>
                            <option value="COMPLETA" <?php echo (isset($_GET['filtro_tipo_cuenta']) && $_GET['filtro_tipo_cuenta'] === 'COMPLETA') ? 'selected' : ''; ?>>Completa</option>
                        </select>
                    </div>
                    <div class="col-lg-2 col-md-6">
                        <label for="filtro_vendedor" class="form-label form-label-sm">Vendedor:</label>
                        <select name="filtro_vendedor" id="filtro_vendedor" class="form-select form-select-sm rounded-3">
                            <option value="">Todos</option>
                            <?php foreach ($vendedores_filtro as $v):?>
                                <option value="<?php echo $v['id_usuario']; ?>" <?php echo (isset($_GET['filtro_vendedor']) && $_GET['filtro_vendedor'] == $v['id_usuario']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($v['nombre_completo']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-lg-2 col-md-6">
                        <label for="filtro_fecha_inicio" class="form-label form-label-sm">Fecha Inicio:</label>
                        <input type="date" name="filtro_fecha_inicio" id="filtro_fecha_inicio" class="form-control form-control-sm rounded-3" value="<?php echo htmlspecialchars($_GET['filtro_fecha_inicio'] ?? ''); ?>">
                    </div>
                    <div class="col-lg-2 col-md-6">
                        <label for="filtro_fecha_fin" class="form-label form-label-sm">Fecha Fin:</label>
                        <input type="date" name="filtro_fecha_fin" id="filtro_fecha_fin" class="form-control form-control-sm rounded-3" value="<?php echo htmlspecialchars($_GET['filtro_fecha_fin'] ?? ''); ?>">
                    </div>
                    
                    <!-- Tercera fila -->
                    <div class="col-lg-2 col-md-6">
                        <label for="filtro_precio_min" class="form-label form-label-sm">Precio Mín:</label>
                        <input type="number" step="0.01" min="0" name="filtro_precio_min" id="filtro_precio_min" class="form-control form-control-sm rounded-3" value="<?php echo htmlspecialchars($_GET['filtro_precio_min'] ?? ''); ?>" placeholder="0.00">
                    </div>
                    <div class="col-lg-2 col-md-6">
                        <label for="filtro_precio_max" class="form-label form-label-sm">Precio Máx:</label>
                        <input type="number" step="0.01" min="0" name="filtro_precio_max" id="filtro_precio_max" class="form-control form-control-sm rounded-3" value="<?php echo htmlspecialchars($_GET['filtro_precio_max'] ?? ''); ?>" placeholder="999.99">
                    </div>
                    <div class="col-lg-2 col-md-6">
                        <label for="ordenar_por" class="form-label form-label-sm">Ordenar por:</label>
                        <select name="ordenar_por" id="ordenar_por" class="form-select form-select-sm rounded-3">
                            <option value="fecha_inicio_servicio" <?php echo (isset($_GET['ordenar_por']) && $_GET['ordenar_por'] === 'fecha_inicio_servicio') ? 'selected' : ''; ?>>Fecha Inicio</option>
                            <option value="fecha_fin_servicio" <?php echo (isset($_GET['ordenar_por']) && $_GET['ordenar_por'] === 'fecha_fin_servicio') ? 'selected' : ''; ?>>Fecha Fin</option>
                            <option value="precio_venta" <?php echo (isset($_GET['ordenar_por']) && $_GET['ordenar_por'] === 'precio_venta') ? 'selected' : ''; ?>>Precio</option>
                            <option value="nombre_completo" <?php echo (isset($_GET['ordenar_por']) && $_GET['ordenar_por'] === 'nombre_completo') ? 'selected' : ''; ?>>Cliente</option>
                            <option value="nombre_plataforma" <?php echo (isset($_GET['ordenar_por']) && $_GET['ordenar_por'] === 'nombre_plataforma') ? 'selected' : ''; ?>>Plataforma</option>
                            <option value="estado_suscripcion" <?php echo (isset($_GET['ordenar_por']) && $_GET['ordenar_por'] === 'estado_suscripcion') ? 'selected' : ''; ?>>Estado</option>
                        </select>
                    </div>
                    <div class="col-lg-2 col-md-6">
                        <label for="orden" class="form-label form-label-sm">Orden:</label>
                        <select name="orden" id="orden" class="form-select form-select-sm rounded-3">
                            <option value="DESC" <?php echo (isset($_GET['orden']) && $_GET['orden'] === 'DESC') ? 'selected' : ''; ?>>Descendente</option>
                            <option value="ASC" <?php echo (isset($_GET['orden']) && $_GET['orden'] === 'ASC') ? 'selected' : ''; ?>>Ascendente</option>
                        </select>
                    </div>
                    <div class="col-lg-2 col-md-6 d-flex align-items-end">
                        <button type="submit" class="btn btn-info btn-sm w-100 me-2">
                            <i class="fas fa-search me-1"></i> Aplicar
                        </button>
                        <button type="button" class="btn btn-outline-secondary btn-sm w-100" id="btnLimpiarFiltros" title="Limpiar todos los filtros">
                            <i class="fas fa-eraser me-1"></i> Limpiar
                        </button>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- Script para preservar filtros en localStorage -->
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Guardar filtros en localStorage cuando se apliquen
            const formFiltros = document.querySelector('form[method="GET"]');
            if (formFiltros) {
                formFiltros.addEventListener('submit', function() {
                    const filtros = {};
                    const filtrosInputs = formFiltros.querySelectorAll('input[name^="filtro_"], input[name="buscar"], select[name="ordenar_por"], select[name="orden"]');
                    
                    filtrosInputs.forEach(input => {
                        if (input.value && input.value !== '') {
                            filtros[input.name] = input.value;
                        }
                    });
                    
                    // Guardar en localStorage para persistencia
                    localStorage.setItem('filtros_perfiles_vendidos', JSON.stringify(filtros));
                    console.log('Filtros guardados:', filtros);
                });
            }
            
            // Restaurar filtros desde localStorage al cargar la página
            const filtrosGuardados = localStorage.getItem('filtros_perfiles_vendidos');
            if (filtrosGuardados) {
                try {
                    const filtros = JSON.parse(filtrosGuardados);
                    Object.keys(filtros).forEach(nombre => {
                        const input = formFiltros.querySelector(`[name="${nombre}"]`);
                        if (input) {
                            input.value = filtros[nombre];
                        }
                    });
                    console.log('Filtros restaurados:', filtros);
                } catch (e) {
                    console.error('Error al restaurar filtros:', e);
                }
            }
            
            // Botón Limpiar Filtros
            const btnLimpiarFiltros = document.getElementById('btnLimpiarFiltros');
            if (btnLimpiarFiltros) {
                btnLimpiarFiltros.addEventListener('click', function() {
                    // Limpiar todos los campos del formulario
                    const formFiltros = document.querySelector('form[method="GET"]');
                    if (formFiltros) {
                        // Limpiar inputs
                        formFiltros.querySelectorAll('input[type="text"], input[type="number"], input[type="date"]').forEach(input => {
                            input.value = '';
                        });
                        
                        // Resetear selects a la primera opción
                        formFiltros.querySelectorAll('select').forEach(select => {
                            if (select.options.length > 0) {
                                select.selectedIndex = 0;
                            }
                        });
                        
                        // Limpiar localStorage
                        localStorage.removeItem('filtros_perfiles_vendidos');
                        
                        // Enviar el formulario limpio
                        formFiltros.submit();
                    }
                });
            }
        });
        </script>
    </div>
    
    <div class="d-flex flex-wrap align-items-center justify-content-between mb-3 gap-2">
        <div class="text-muted small">
            <?php if ($filtros_aplicados_pv): ?>
                <span class="text-success">
                    <i class="fas fa-filter me-1"></i>Filtros aplicados: Mostrando <strong><?php echo $total_perfiles_count; ?></strong> perfiles
                    <span class="badge bg-success-soft text-success-emphasis ms-2">Filtros Activos</span>
                </span>
            <?php else: ?>
                Mostrando <strong><?php echo min($registros_cargados, $total_perfiles_count); ?></strong> de <strong><?php echo $total_perfiles_count; ?></strong> perfiles
                <?php if ($registros_cargados < $total_perfiles_count): ?>
                    <span class="text-info">(Carga diferida activada)</span>
                <?php endif; ?>
            <?php endif; ?>

        </div>
        <?php if (!$filtros_aplicados_pv && $registros_cargados < $total_perfiles_count): ?>
            <div class="d-flex gap-2">
                <button type="button" class="btn btn-primary btn-sm" id="btnCargarMas" data-registros-actuales="<?php echo $registros_cargados; ?>" data-total="<?php echo $total_perfiles_count; ?>">
                    <i class="fas fa-plus-circle me-2"></i>Cargar más perfiles
                </button>
            </div>
        <?php endif; ?>
    </div>

    <div class="row" id="subscriptions-container">
        <?php if (count($perfiles) > 0): ?>
            <?php foreach ($perfiles as $perfil): ?>
                <?php
                $dias_restantes = intval($perfil['dias_restantes_sql']);
                $vencimiento_text = '';
                $vencimiento_class = '';
                $card_glow_class = '';
                $progress_bar_class = 'bg-success';
                $progress_bar_value = 100;
                $status_badge = '<span class="badge bg-success-soft text-success-emphasis rounded-pill px-3 py-2">Activa</span>';

                if ($perfil['estado_suscripcion'] == 1) {
                    if ($dias_restantes <= 0) {
                        $vencimiento_text = 'VENCIDO' . ($dias_restantes < 0 ? ' hace ' . abs($dias_restantes) . ' día' . (abs($dias_restantes) != 1 ? 's' : '') : '');
                        $vencimiento_class = 'text-danger fw-bold';
                        $card_glow_class = 'card-glow-danger';
                        $progress_bar_class = 'bg-danger';
                        $progress_bar_value = 100;
                        $status_badge = '<span class="badge bg-danger-soft text-danger-emphasis rounded-pill px-3 py-2">Vencido</span>';
                    } elseif ($dias_restantes <= $dias_alerta_clientes_global) {
                        $vencimiento_text = "Restan " . $dias_restantes . ' día' . ($dias_restantes != 1 ? 's' : '');
                        $vencimiento_class = 'text-warning-emphasis fw-bold';
                        $card_glow_class = 'card-glow-warning';
                        $progress_bar_class = 'bg-warning';
                        $progress_bar_value = max(0, ($dias_restantes / 30) * 100);
                        $status_badge = '<span class="badge bg-warning-soft text-warning-emphasis rounded-pill px-3 py-2">Por Vencer</span>';
                    } else {
                        $vencimiento_text = "Restan " . $dias_restantes . ' día' . ($dias_restantes != 1 ? 's' : '');
                        $vencimiento_class = 'text-success-emphasis fw-bold';
                        $progress_bar_value = max(0, ($dias_restantes / 30) * 100);
                    }
                } elseif ($perfil['estado_suscripcion'] == 0) {
                    $vencimiento_text = "VENCIDA";
                    $vencimiento_class = 'text-dark-emphasis fw-bold';
                    $progress_bar_class = 'bg-dark-subtle';
                    $progress_bar_value = 100;
                    $status_badge = '<span class="badge bg-dark-soft text-dark-emphasis rounded-pill px-3 py-2">Inactiva</span>';
                } elseif ($perfil['estado_suscripcion'] == 2) {
                    $vencimiento_text = "CANCELADA";
                    $vencimiento_class = 'text-secondary-emphasis fw-bold';
                    $progress_bar_class = 'bg-secondary';
                    $progress_bar_value = 100;
                    $status_badge = '<span class="badge bg-secondary-soft text-secondary-emphasis rounded-pill px-3 py-2">Cancelada</span>';
                    $card_glow_class = 'card-cancelada';
                }

                $header_platform_class = getPlatformHeaderClass($perfil['nombre_plataforma']);
                $platform_icon = getPlatformIconClass($perfil['nombre_plataforma']);

                $master_account_section_class = '';
                $is_master_account_problem = false;

                if ($perfil['estado_cuenta'] == 2) {
                    $master_account_section_class = 'bg-danger-soft';
                    $is_master_account_problem = true;
                } elseif ($perfil['estado_cuenta'] == 0) {
                    $master_account_section_class = 'bg-dark-soft';
                    $is_master_account_problem = true;
                } elseif ($perfil['fecha_vencimiento_proveedor']) {
                    $is_past = (new DateTime()) > (new DateTime($perfil['fecha_vencimiento_proveedor']));
                    if ($is_past) {
                        $master_account_section_class = 'bg-danger-soft';
                        $is_master_account_problem = true;
                    }
                }

                $searchable_text = strtolower($perfil['nombre_cliente'] . ' ' . $perfil['nombre_plataforma'] . ' ' . $perfil['email_cuenta_maestra'] . ' ' . ($perfil['cliente_telefono'] ?? '') . ' ' . $perfil['nombre_perfil_cliente']);
                ?>

                <div class="col-xl-4 col-lg-6 col-md-6 mb-4 subscription-card-wrapper">
                    <div class="card subscription-card h-100 <?php echo $card_glow_class; ?>" data-search-text="<?php echo htmlspecialchars($searchable_text); ?>">
                        <div class="card-header <?php echo $header_platform_class; ?> d-flex justify-content-between align-items-center">
                            <div class="d-flex align-items-center flex-grow-1 text-truncate me-2">
                                <i class="<?php echo $platform_icon; ?> platform-icon"></i>
                                <div class="text-truncate">
                                    <h5 class="mb-0 fw-bold text-truncate" title="<?php echo htmlspecialchars($perfil['nombre_cliente']); ?>">
                                        <?php echo htmlspecialchars($perfil['nombre_cliente']); ?>
                                    </h5>
                                    <?php if(!empty($perfil['cliente_telefono'])): ?>
                                    <small class="d-block text-white-50">
                                        <i class="fas fa-phone-alt fa-xs me-1"></i>
                                        <span id="phone_<?php echo $perfil['id_perfil_vendido']; ?>"><?php echo htmlspecialchars($perfil['cliente_telefono']); ?></span>
                                        <i class="fas fa-copy copy-btn-header ms-2" title="Copiar Teléfono" data-bs-toggle="tooltip" onclick="copyToClipboard('#phone_<?php echo $perfil['id_perfil_vendido']; ?>', this)"></i>
                                    </small>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <?php if(puedeRenovarPerfil($rol_usuario_sesion)): ?>
                            <button type="button" class="btn btn-sm btn-renew flex-shrink-0"
                                data-bs-toggle="modal" data-bs-target="#modalRenovarSuscripcion"
                                title="Renovación Rápida"
                                data-id_perfil_vendido="<?php echo $perfil['id_perfil_vendido']; ?>"
                                data-info_cliente="<?php echo htmlspecialchars($perfil['nombre_cliente'] . ' - ' . $perfil['nombre_plataforma'] . ' (' . $perfil['nombre_perfil_cliente'] . ')'); ?>"
                                data-precio_venta="<?php echo htmlspecialchars($perfil['precio_venta']); ?>"
                                data-fecha-vencimiento="<?php echo htmlspecialchars($perfil['fecha_fin_servicio']); ?>">
                                <i class="fas fa-sync-alt me-1"></i> Renovar
                            </button>
                            <?php endif; ?>
                        </div>
                        <div class="card-body p-0">
                            <div class="card-section customer-details">
                                <h6 class="card-section-title">Suscripción del Cliente</h6>
                                <div class="detail-item"><span class="detail-label"><i class="fas fa-tv fa-fw"></i>Plataforma</span> <span class="detail-value fw-bold"><?php echo htmlspecialchars($perfil['nombre_plataforma']); ?></span></div>
                                <div class="detail-item"><span class="detail-label"><i class="fas fa-user-tag fa-fw"></i>Perfil</span> <span class="detail-value fw-bold"><?php echo htmlspecialchars($perfil['nombre_perfil_cliente']); ?></span></div>
                                <div class="detail-item"><span class="detail-label"><i class="fas fa-lock fa-fw"></i>PIN</span> <div class="detail-actions"><span class="fw-bold badge bg-light text-dark border py-1 px-2 me-2" id="pin_<?php echo $perfil['id_perfil_vendido']; ?>"><?php echo htmlspecialchars($perfil['pin_perfil_cliente'] ?? '----'); ?></span><i class="fas fa-copy copy-btn" title="Copiar PIN" data-bs-toggle="tooltip" onclick="copyToClipboard('#pin_<?php echo $perfil['id_perfil_vendido']; ?>', this)"></i></div></div>
                                <div class="detail-item"><span class="detail-label"><i class="fas fa-calendar-plus fa-fw"></i>Inicio</span> <span class="detail-value"><?php echo htmlspecialchars(date("d/m/Y", strtotime($perfil['fecha_inicio_servicio'] ?? ''))); ?></span></div>
                                <div class="detail-item"><span class="detail-label"><i class="fas fa-calendar-times fa-fw"></i>Vence</span> <span class="detail-value fw-bold <?php echo $vencimiento_class; ?>"><?php echo htmlspecialchars(date("d/m/Y", strtotime($perfil['fecha_fin_servicio']))); ?></span></div>
                                <div class="detail-item"><span class="detail-label"><i class="fas fa-dollar-sign fa-fw"></i>Precio</span> <span class="detail-value"><?php echo htmlspecialchars($moneda_simbolo . number_format($perfil['precio_venta'], 2)); ?></span></div>
                                <div class="detail-item"><span class="detail-label"><i class="fas fa-user-tie fa-fw"></i>Vendedor</span> <span class="detail-value fw-bold text-primary"><?php echo htmlspecialchars($perfil['nombre_vendedor'] ?? 'N/A'); ?></span></div>
                                <div class="detail-item"><span class="detail-label"><i class="fas fa-info-circle fa-fw"></i>Estado</span> <span class="detail-value"><?php echo $status_badge; ?></span></div>
                            </div>
                            <?php if (!empty($perfil['notas_cliente']) || !empty($perfil['notas_suscripcion'])): ?>
                            <div class="notes-section-container pt-0 pb-3">
                                <div class="accordion accordion-flush" id="accordionNotes_<?php echo $perfil['id_perfil_vendido']; ?>">
                                    <?php if (!empty($perfil['notas_cliente'])): ?>
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingCliente_<?php echo $perfil['id_perfil_vendido']; ?>">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseCliente_<?php echo $perfil['id_perfil_vendido']; ?>" aria-expanded="false" aria-controls="collapseCliente_<?php echo $perfil['id_perfil_vendido']; ?>">
                                                <i class="fas fa-user-pen me-2 text-primary"></i> Notas del Cliente
                                            </button>
                                        </h2>
                                        <div id="collapseCliente_<?php echo $perfil['id_perfil_vendido']; ?>" class="accordion-collapse collapse" aria-labelledby="headingCliente_<?php echo $perfil['id_perfil_vendido']; ?>" data-bs-parent="#accordionNotes_<?php echo $perfil['id_perfil_vendido']; ?>">
                                            <div class="accordion-body">
                                                <?php echo nl2br(htmlspecialchars($perfil['notas_cliente'])); ?>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    <?php if (!empty($perfil['notas_suscripcion'])): ?>
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingSuscripcion_<?php echo $perfil['id_perfil_vendido']; ?>">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseSuscripcion_<?php echo $perfil['id_perfil_vendido']; ?>" aria-expanded="false" aria-controls="collapseSuscripcion_<?php echo $perfil['id_perfil_vendido']; ?>">
                                                <i class="fas fa-sticky-note me-2 text-info"></i> Notas de Suscripción
                                            </button>
                                        </h2>
                                        <div id="collapseSuscripcion_<?php echo $perfil['id_perfil_vendido']; ?>" class="accordion-collapse collapse" aria-labelledby="headingSuscripcion_<?php echo $perfil['id_perfil_vendido']; ?>" data-bs-parent="#accordionNotes_<?php echo $perfil['id_perfil_vendido']; ?>">
                                            <div class="accordion-body">
                                                <?php echo nl2br(htmlspecialchars($perfil['notas_suscripcion'])); ?>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($perfil['notas_cuenta_maestra'])): ?>
                            <div class="notes-section-container pt-0 pb-3">
                                <div class="accordion accordion-flush" id="accordionNotesCM_<?php echo $perfil['id_perfil_vendido']; ?>">
                                    <div class="accordion-item">
                                        <h2 class="accordion-header" id="headingCuentaMaestra_<?php echo $perfil['id_perfil_vendido']; ?>">
                                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseCuentaMaestra_<?php echo $perfil['id_perfil_vendido']; ?>" aria-expanded="false" aria-controls="collapseCuentaMaestra_<?php echo $perfil['id_perfil_vendido']; ?>">
                                                <i class="fas fa-key me-2 text-warning"></i> Notas de Cuenta Maestra
                                            </button>
                                        </h2>
                                        <div id="collapseCuentaMaestra_<?php echo $perfil['id_perfil_vendido']; ?>" class="accordion-collapse collapse" aria-labelledby="headingCuentaMaestra_<?php echo $perfil['id_perfil_vendido']; ?>" data-bs-parent="#accordionNotesCM_<?php echo $perfil['id_perfil_vendido']; ?>">
                                            <div class="accordion-body">
                                                <?php echo nl2br(htmlspecialchars($perfil['notas_cuenta_maestra'])); ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <div class="master-account-section <?php echo $master_account_section_class; ?> rounded-3">
                                <h6 class="card-section-title">Cuenta Maestra</h6>
                                <div class="detail-item">
                                    <span class="detail-label"><i class="fas fa-at fa-fw"></i>Email</span>
                                    <div class="detail-actions">
                                        <span class="fw-bold text-dark text-truncate me-2" id="email_<?php echo $perfil['id_perfil_vendido']; ?>" style="max-width: 150px;"><?php echo htmlspecialchars($perfil['email_cuenta_maestra']); ?></span>
                                        <i class="fas fa-copy copy-btn" title="Copiar Email" data-bs-toggle="tooltip" onclick="copyToClipboard('#email_<?php echo $perfil['id_perfil_vendido']; ?>', this)"></i>
                                    </div>
                                </div>
                                <div class="detail-item">
                                    <span class="detail-label"><i class="fas fa-key fa-fw"></i>Contraseña</span>
                                    <div class="detail-actions">
                                        <input type="password" readonly class="form-control-plaintext p-0 border-0" id="password_<?php echo $perfil['id_perfil_vendido']; ?>" value="<?php echo htmlspecialchars($perfil['password_cuenta_maestra']); ?>">
                                        <i class="fas fa-eye toggle-password-btn" title="Mostrar/Ocultar" data-bs-toggle="tooltip" onclick="togglePasswordVisibility('password_<?php echo $perfil['id_perfil_vendido']; ?>', this)"></i>
                                        <i class="fas fa-copy copy-btn" title="Copiar Contraseña" data-bs-toggle="tooltip" onclick="copyToClipboard('#password_<?php echo $perfil['id_perfil_vendido']; ?>', this)"></i>
                                    </div>
                                </div>
                                <div class="detail-item">
                                    <span class="detail-label"><i class="fas fa-server fa-fw"></i>Estado Cta.</span>
                                    <span class="detail-value"><?php echo getMasterAccountStatusBadge($perfil['estado_cuenta'], $perfil['fecha_vencimiento_proveedor']); ?></span>
                                </div>
                                <?php if ($perfil['fecha_vencimiento_proveedor']): ?>
                                <div class="detail-item">
                                    <span class="detail-label"><i class="fas fa-calendar-check fa-fw"></i>Vence Prov.</span>
                                    <span class="detail-value"><?php echo htmlspecialchars(date("d/m/Y", strtotime($perfil['fecha_vencimiento_proveedor']))); ?></span>
                                </div>
                                <?php endif; ?>
                                <div class="detail-item">
                                    <span class="detail-label"><i class="fas fa-users fa-fw"></i>Perfiles Disp.</span>
                                    <span class="detail-value fw-bold"><?php echo (int)($perfil['perfiles_disponibles'] ?? 0); ?></span>
                                </div>
                            </div>
                            
                            <div class="progress-section">
                                <small class="text-muted d-block text-end fw-semibold <?php echo $vencimiento_class; ?>"><?php echo $vencimiento_text; ?></small>
                                <div class="progress" role="progressbar" title="<?php echo $vencimiento_text; ?>" data-bs-toggle="tooltip">
                                    <div class="progress-bar <?php echo $progress_bar_class; ?>" style="width: <?php echo $progress_bar_value; ?>%"></div>
                                </div>
                            </div>
                        </div>
                        <div class="card-footer">
                            <?php
                            $telefono_cliente_limpio = !empty($perfil['cliente_telefono']) ? limpiarNumeroWhatsApp($perfil['cliente_telefono']) : '';
                            $mensaje_bienvenida = generarMensajeParaCliente($pdo, $perfil['id_perfil_vendido'], $OWNER_ID, 'bienvenida') ?? '';
                            $mensaje_recordatorio_btn = ($perfil['estado_suscripcion'] == 1 && $dias_restantes <= $dias_alerta_clientes_global)
                                ? (generarMensajeParaCliente($pdo, $perfil['id_perfil_vendido'], $OWNER_ID, "recordatorio_vencimiento") ?? '')
                                : '';
                            ?>
                            <?php if (puedeVerAlertas($rol_usuario_sesion)): ?>
                            <button type="button"
                                    class="btn btn-sm btn-outline-success rounded-pill shadow-sm"
                                    title="Ver Mensaje para el Cliente"
                                    data-bs-toggle="modal"
                                    data-bs-target="#modalMensajeCliente"
                                    data-id_perfil_vendido="<?php echo $perfil['id_perfil_vendido']; ?>"
                                    data-telefono="<?php echo htmlspecialchars($telefono_cliente_limpio); ?>"
                                    data-mensaje="<?php echo htmlspecialchars($mensaje_bienvenida, ENT_QUOTES); ?>">
                                <i class="fas fa-info-circle me-1"></i> Info
                            </button>
                            <?php endif; ?>

                            <?php if (!empty($mensaje_recordatorio_btn) && puedeCobrarPerfil($rol_usuario_sesion)): ?>
                                <button type="button"
                                        class="btn btn-sm btn-outline-warning text-dark rounded-pill shadow-sm"
                                        title="Recordatorio de Cobro"
                                        data-bs-toggle="modal"
                                        data-bs-target="#modalMensajeCliente"
                                        data-id_perfil_vendido="<?php echo $perfil['id_perfil_vendido']; ?>"
                                        data-telefono="<?php echo htmlspecialchars($telefono_cliente_limpio); ?>"
                                        data-mensaje="<?php echo htmlspecialchars($mensaje_recordatorio_btn, ENT_QUOTES); ?>">
                                    <i class="fas fa-comment-dollar me-1"></i> Cobro
                                </button>
                            <?php endif; ?>

                            <?php if (puedeEditarPerfil($rol_usuario_sesion, $perfil['id_usuario'], $id_usuario_sesion)): ?>
                            <button type="button" class="btn btn-sm btn-outline-primary rounded-pill shadow-sm" title="Editar Suscripción" onclick="editarPerfilVendido(<?php echo $perfil['id_perfil_vendido']; ?>)"><i class="fas fa-edit me-1"></i> Editar</button>
                            <?php endif; ?>

                            <?php if (puedeEliminarPerfil($rol_usuario_sesion)): ?>
                            <button type="button" class="btn btn-sm btn-outline-danger rounded-pill btn-eliminar shadow-sm"
                                data-bs-toggle="modal" data-bs-target="#modalEliminarSuscripcion"
                                title="Eliminar Suscripción"
                                data-id_perfil_vendido="<?php echo $perfil['id_perfil_vendido']; ?>"
                                data-cliente_nombre="<?php echo htmlspecialchars($perfil['nombre_cliente'] . ' - ' . $perfil['nombre_plataforma'] . ' (' . $perfil['nombre_perfil_cliente'] . ')'); ?>">
                                <i class="fas fa-trash-alt me-1"></i> Eliminar
                            </button>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php else: ?>
            <div class="col-12 mt-4">
                <div class="alert alert-light text-center border-0 py-4 shadow-sm rounded-3" role="alert">
                    <h4 class="alert-heading text-muted mb-3"><i class="fas fa-box-open me-2"></i> ¡No hay suscripciones para mostrar!</h4>
                    <p class="mb-0 text-muted">Parece que aún no has registrado ninguna venta de perfiles con los filtros actuales.</p>
                    <p class="mb-0 text-muted">Haz clic en "Nueva Suscripción" para empezar a gestionar tus ventas.</p>
                </div>
            </div>
        <?php endif; ?>
        <div id="no-results-message" class="col-12 text-center" style="display: none;">
            <div class="alert alert-warning border-0 py-4 shadow-sm rounded-3">
                <h4 class="alert-heading"><i class="fas fa-search-minus me-2"></i> Sin resultados</h4>
                <p class="mb-0">No se encontraron suscripciones que coincidan con tu búsqueda.</p>
            </div>
        </div>
    </div>
</div>

<?php if ($mensaje_para_copiar_al_cliente): ?>
<div class="modal fade" id="modalMensajePostCompra" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content rounded-4 shadow-lg">
            <div class="modal-header bg-success text-white rounded-top-4">
                <h5 class="modal-title"><i class="fas fa-check-circle me-2"></i>¡Venta Exitosa!</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Cerrar"></button>
            </div>
            <div class="modal-body p-4">
                <p class="lead mb-3 text-dark">Copia el siguiente mensaje para tu cliente:</p>
                <textarea class="form-control mb-3 rounded-3" id="textoMensajeCliente" rows="10" readonly><?php echo htmlspecialchars($mensaje_para_copiar_al_cliente); ?></textarea>
                <div class="d-flex justify-content-between align-items-center">
                    <?php if ($telefono_cliente_para_whatsapp): ?>
                    <a href="https://wa.me/<?php echo htmlspecialchars($telefono_cliente_para_whatsapp); ?>?text=<?php echo urlencode($mensaje_para_copiar_al_cliente); ?>" target="_blank" class="btn btn-success">
                        <i class="fab fa-whatsapp me-2"></i> Enviar por WhatsApp
                    </a>
                    <?php endif; ?>
                    <div class="d-flex align-items-center">
                        <button type="button" class="btn btn-primary me-3" id="btnCopiarMensajeCliente">
                            <i class="fas fa-copy me-2"></i> Copiar Mensaje
                        </button>
                        <small id="mensajeCopiadoFeedback" class="text-success d-none fw-bold"><i class="fas fa-check me-1"></i>¡Copiado!</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<div class="modal fade" id="modalAlertas" tabindex="-1" aria-labelledby="modalAlertasLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered modal-xl">
        <div class="modal-content rounded-4 shadow-lg">
            <div class="modal-header bg-warning text-dark border-0 rounded-top-4">
                <h5 class="modal-title" id="modalAlertasLabel">
                    <i class="fas fa-bell me-2"></i>Centro de Notificaciones
                    <?php if ($has_alerts): ?>
                        <span class="badge bg-danger ms-2 fs-6"><?php echo count($alerts_data['clientes_vencidos']) + count($alerts_data['clientes_proximos']) + count($alerts_data['cuentas_maestras_proximas']); ?></span>
                    <?php endif; ?>
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body p-4">
                <?php if ($has_alerts): ?>
                    <!-- RESUMEN RÁPIDO DE ALERTAS -->
                    <div class="alert-summary mb-4">
                        <div class="row g-3">
                            <?php if (!empty($alerts_data['clientes_proximos'])): ?>
                            <div class="col-md-4">
                                <div class="summary-card bg-warning-soft border-warning">
                                    <div class="summary-icon text-warning">
                                        <i class="fas fa-hourglass-half fa-2x"></i>
                                    </div>
                                    <div class="summary-content">
                                        <h6 class="mb-1">Próximas a Vencer</h6>
                                        <span class="summary-count"><?php echo count($alerts_data['clientes_proximos']); ?></span>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($alerts_data['clientes_vencidos'])): ?>
                            <div class="col-md-4">
                                <div class="summary-card bg-danger-soft border-danger">
                                    <div class="summary-icon text-danger">
                                        <i class="fas fa-exclamation-triangle fa-2x"></i>
                                    </div>
                                    <div class="summary-content">
                                        <h6 class="mb-1">Vencidas</h6>
                                        <span class="summary-count"><?php echo count($alerts_data['clientes_vencidos']); ?></span>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($alerts_data['cuentas_maestras_proximas'])): ?>
                            <div class="col-md-4">
                                <div class="summary-card bg-info-soft border-info">
                                    <div class="summary-icon text-info">
                                        <i class="fas fa-server fa-2x"></i>
                                    </div>
                                    <div class="summary-content">
                                        <h6 class="mb-1">Cuentas Maestras</h6>
                                        <span class="summary-count"><?php echo count($alerts_data['cuentas_maestras_proximas']); ?></span>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- ACORDEÓN DE ALERTAS -->
                    <div class="accordion" id="accordionAlertas">
                        <!-- PRIORIDAD 1: PERFILES PRÓXIMOS A VENCER -->
                        <?php if (!empty($alerts_data['clientes_proximos'])): ?>
                        <div class="accordion-item border-warning">
                            <h2 class="accordion-header" id="headingProximos">
                                <button class="accordion-button bg-warning-soft text-warning-emphasis" type="button" data-bs-toggle="collapse" data-bs-target="#collapseProximos" aria-expanded="true" aria-controls="collapseProximos">
                                    <i class="fas fa-hourglass-half me-2"></i>
                                    <strong>Perfiles Próximos a Vencer (PRIORIDAD ALTA)</strong>
                                    <span class="badge bg-warning ms-2"><?php echo count($alerts_data['clientes_proximos']); ?></span>
                                </button>
                            </h2>
                            <div id="collapseProximos" class="accordion-collapse collapse show" aria-labelledby="headingProximos" data-bs-parent="#accordionAlertas">
                                <div class="accordion-body p-0">
                                    <div class="alert-list">
                                        <?php foreach ($alerts_data['clientes_proximos'] as $perfil_alert):
                                            $dias = $perfil_alert['dias_restantes_sql'];
                                            $urgency_class = $dias <= 3 ? 'border-danger' : 'border-warning';
                                            $urgency_icon = $dias <= 3 ? 'fas fa-fire text-danger' : 'fas fa-bell text-warning';
                                        ?>
                                            <div class="alert-item-card <?php echo $urgency_class; ?>">
                                                <i class="<?php echo $urgency_icon; ?> alert-icon"></i>
                                                <div class="alert-info-text">
                                                    <strong><?php echo htmlspecialchars($perfil_alert['nombre_cliente']); ?></strong>
                                                    <small class="text-muted">
                                                        <?php echo htmlspecialchars($perfil_alert['nombre_plataforma'] . ' - ' . $perfil_alert['nombre_perfil_cliente']); ?>
                                                        <span class="badge <?php echo $dias <= 3 ? 'bg-danger' : 'bg-warning'; ?> ms-2">
                                                            Vence en <?php echo $dias; ?> día<?php echo ($dias != 1) ? 's' : ''; ?>
                                                        </span>
                                                    </small>
                                                </div>
                                                <div class="alert-actions ms-auto">
                                                    <?php if (puedeRenovarPerfil($rol_usuario_sesion)): ?>
                                                    <button type="button" class="btn btn-sm btn-success rounded-pill shadow-sm btn-renew-alert"
                                                        data-bs-toggle="modal" data-bs-target="#modalRenovarSuscripcion"
                                                        title="Renovar Suscripción"
                                                        data-id_perfil_vendido="<?php echo $perfil_alert['id_perfil_vendido']; ?>"
                                                        data-precio_venta="<?php echo htmlspecialchars($perfil_alert['precio_venta']); ?>"
                                                        data-info_cliente="<?php echo htmlspecialchars($perfil_alert['nombre_cliente'] . ' - ' . $perfil_alert['nombre_plataforma'] . ' (' . $perfil_alert['nombre_perfil_cliente'] . ')'); ?>"
                                                        data-fecha-vencimiento="<?php echo htmlspecialchars($perfil_alert['fecha_fin_servicio']); ?>">
                                                        <i class="fas fa-sync-alt me-1"></i> Renovar
                                                    </button>
                                                    <?php endif; ?>
                                                    <?php if (puedeCobrarPerfil($rol_usuario_sesion)): ?>
                                                    <button type="button"
                                                        class="btn btn-sm btn-outline-warning text-dark rounded-pill shadow-sm"
                                                        title="Recordatorio de Cobro"
                                                        data-bs-toggle="modal"
                                                        data-bs-target="#modalMensajeCliente"
                                                        data-id_perfil_vendido="<?php echo $perfil_alert['id_perfil_vendido']; ?>"
                                                        data-telefono="<?php echo htmlspecialchars(limpiarNumeroWhatsApp($perfil_alert['cliente_telefono'])); ?>"
                                                        data-mensaje="<?php echo htmlspecialchars(generarMensajeParaCliente($pdo, $perfil_alert['id_perfil_vendido'], $OWNER_ID, "recordatorio_vencimiento") ?? '', ENT_QUOTES); ?>">
                                                        <i class="fas fa-comment-dollar me-1"></i> Cobro
                                                    </button>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>

                        <!-- PRIORIDAD 2: PERFILES VENCIDOS -->
                        <?php if (!empty($alerts_data['clientes_vencidos'])): ?>
                        <div class="accordion-item border-danger">
                            <h2 class="accordion-header" id="headingVencidos">
                                <button class="accordion-button collapsed bg-danger-soft text-danger-emphasis" type="button" data-bs-toggle="collapse" data-bs-target="#collapseVencidos" aria-expanded="false" aria-controls="collapseVencidos">
                                    <i class="fas fa-exclamation-triangle me-2"></i>
                                    <strong>Suscripciones Vencidas</strong>
                                    <span class="badge bg-danger ms-2"><?php echo count($alerts_data['clientes_vencidos']); ?></span>
                                </button>
                            </h2>
                            <div id="collapseVencidos" class="accordion-collapse collapse" aria-labelledby="headingVencidos" data-bs-parent="#accordionAlertas">
                                <div class="accordion-body p-0">
                                    <div class="alert-list">
                                        <?php foreach ($alerts_data['clientes_vencidos'] as $perfil_alert):
                                            $dias = abs($perfil_alert['dias_restantes_sql']);
                                        ?>
                                            <div class="alert-item-card border-danger">
                                                <i class="fas fa-calendar-times alert-icon text-danger"></i>
                                                <div class="alert-info-text">
                                                    <strong><?php echo htmlspecialchars($perfil_alert['nombre_cliente']); ?></strong>
                                                    <small class="text-muted">
                                                        <?php echo htmlspecialchars($perfil_alert['nombre_plataforma'] . ' - ' . $perfil_alert['nombre_perfil_cliente']); ?>
                                                        <span class="badge bg-danger ms-2">
                                                            Vencido hace <?php echo $dias; ?> día<?php echo ($dias != 1) ? 's' : ''; ?>
                                                        </span>
                                                    </small>
                                                </div>
                                                <div class="alert-actions ms-auto">
                                                    <?php if (puedeRenovarPerfil($rol_usuario_sesion)): ?>
                                                    <button type="button" class="btn btn-sm btn-success rounded-pill shadow-sm btn-renew-alert"
                                                        data-bs-toggle="modal" data-bs-target="#modalRenovarSuscripcion"
                                                        title="Renovar Suscripción"
                                                        data-id_perfil_vendido="<?php echo $perfil_alert['id_perfil_vendido']; ?>"
                                                        data-precio_venta="<?php echo htmlspecialchars($perfil_alert['precio_venta']); ?>"
                                                        data-info_cliente="<?php echo htmlspecialchars($perfil_alert['nombre_cliente'] . ' - ' . $perfil_alert['nombre_plataforma'] . ' (' . $perfil_alert['nombre_perfil_cliente'] . ')'); ?>"
                                                        data-fecha-vencimiento="<?php echo htmlspecialchars($perfil_alert['fecha_fin_servicio']); ?>">
                                                        <i class="fas fa-sync-alt me-1"></i> Renovar
                                                    </button>
                                                    <?php endif; ?>
                                                    <?php if (puedeCobrarPerfil($rol_usuario_sesion)): ?>
                                                    <button type="button"
                                                        class="btn btn-sm btn-outline-warning text-dark rounded-pill shadow-sm"
                                                        title="Recordatorio de Cobro"
                                                        data-bs-toggle="modal"
                                                        data-bs-target="#modalMensajeCliente"
                                                        data-id_perfil_vendido="<?php echo $perfil_alert['id_perfil_vendido']; ?>"
                                                        data-telefono="<?php echo htmlspecialchars(limpiarNumeroWhatsApp($perfil_alert['cliente_telefono'])); ?>"
                                                        data-mensaje="<?php echo htmlspecialchars(generarMensajeParaCliente($pdo, $perfil_alert['id_perfil_vendido'], $OWNER_ID, "recordatorio_vencimiento") ?? '', ENT_QUOTES); ?>">
                                                        <i class="fas fa-comment-dollar me-1"></i> Cobro
                                                    </button>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>

                        <!-- PRIORIDAD 3: CUENTAS MAESTRAS -->
                        <?php if (!empty($alerts_data['cuentas_maestras_proximas'])): ?>
                        <div class="accordion-item border-info">
                            <h2 class="accordion-header" id="headingCuentasMaestras">
                                <button class="accordion-button collapsed bg-info-soft text-info-emphasis" type="button" data-bs-toggle="collapse" data-bs-target="#collapseCuentasMaestras" aria-expanded="false" aria-controls="collapseCuentasMaestras">
                                    <i class="fas fa-server me-2"></i>
                                    <strong>Cuentas Maestras por Vencer</strong>
                                    <span class="badge bg-info ms-2"><?php echo count($alerts_data['cuentas_maestras_proximas']); ?></span>
                                </button>
                            </h2>
                            <div id="collapseCuentasMaestras" class="accordion-collapse collapse" aria-labelledby="headingCuentasMaestras" data-bs-parent="#accordionAlertas">
                                <div class="accordion-body p-0">
                                    <div class="alert-list">
                                        <?php foreach ($alerts_data['cuentas_maestras_proximas'] as $cm_alert):
                                            $dias_cm = $cm_alert['dias_restantes_cm'];
                                        ?>
                                            <div class="alert-item-card border-info">
                                                <i class="fas fa-clock alert-icon text-info"></i>
                                                <div class="alert-info-text">
                                                    <strong><?php echo htmlspecialchars($cm_alert['email_cuenta']); ?></strong>
                                                    <small class="text-muted">
                                                        <span class="badge bg-info ms-2">
                                                            Vence en <?php echo $dias_cm; ?> día<?php echo ($dias_cm != 1) ? 's' : ''; ?>
                                                        </span>
                                                    </small>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>

                <?php else: ?>
                    <div class="text-center p-5">
                        <i class="fas fa-check-circle fa-3x text-success mb-3"></i>
                        <h5 class="text-muted">¡Todo en orden! No hay alertas pendientes.</h5>
                        <p class="text-muted">Tu sistema está funcionando perfectamente.</p>
                    </div>
                <?php endif; ?>
            </div>
            <div class="modal-footer d-flex justify-content-between p-3 border-top-0">
                <div class="d-flex gap-2">
                    <button type="button" class="btn btn-outline-warning btn-sm" onclick="expandAllAlerts()">
                        <i class="fas fa-expand-alt me-1"></i>Expandir Todo
                    </button>
                    <button type="button" class="btn btn-outline-warning btn-sm" onclick="collapseAllAlerts()">
                        <i class="fas fa-compress-alt me-1"></i>Colapsar Todo
                    </button>
                </div>
                <button type="button" class="btn btn-secondary rounded-pill" data-bs-dismiss="modal">Cerrar</button>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="modalRenovarSuscripcion" tabindex="-1" aria-labelledby="modalRenovarSuscripcionLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content rounded-4 shadow-lg">
            <div class="modal-header bg-primary text-white border-0 rounded-top-4">
                <h5 class="modal-title" id="modalRenovarSuscripcionLabel"><i class="fas fa-sync-alt me-2"></i>Renovar Suscripción</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="/digitalpremium/actions/perfiles_vendidos_controlador.php" method="POST">
                <div class="modal-body p-4">
                    <input type="hidden" name="accion" value="renovar">
                    <input type="hidden" name="id_perfil_vendido" id="id_perfil_vendido_renew">
                    
                    <!-- PRESERVAR FILTROS ACTUALES -->
                    <?php
                    // Parámetros de filtro a preservar
                    $filtros_preservar = [
                        'buscar', 'filtro_estado', 'filtro_estado_cm', 'filtro_tipo_cuenta', 
                        'filtro_vendedor', 'filtro_fecha_inicio', 'filtro_fecha_fin', 
                        'filtro_precio_min', 'filtro_precio_max', 'ordenar_por', 'orden'
                    ];
                    
                    foreach ($filtros_preservar as $filtro) {
                        if (isset($_GET[$filtro]) && $_GET[$filtro] !== '') {
                            echo '<input type="hidden" name="' . htmlspecialchars($filtro) . '" value="' . htmlspecialchars($_GET[$filtro]) . '">';
                        }
                    }
                    ?>
                    
                    <div class="alert alert-info border-0 rounded-3 mb-4">
                        <div class="d-flex align-items-center">
                            <i class="fas fa-user-circle fa-2x text-primary me-3"></i>
                            <div>
                                <h6 class="mb-1 fw-bold text-dark">Cliente a Renovar</h6>
                                <p id="info_cliente_renew" class="mb-0 text-dark fw-semibold"></p>
                            </div>
                        </div>
                    </div>

                    <div class="mb-4">
                        <label class="form-label fw-bold text-dark mb-3">
                            <i class="fas fa-calendar-alt me-2 text-primary"></i>
                            Nueva Fecha de Vencimiento <span class="text-danger">*</span>
                        </label>
                        
                        <div class="d-flex gap-2 mb-3 flex-wrap">
                            <button type="button" class="btn btn-outline-primary rounded-pill preset-renew" data-days="30">
                                <i class="fas fa-plus me-1"></i>+30 días
                            </button>
                            <button type="button" class="btn btn-outline-primary rounded-pill preset-renew" data-days="60">
                                <i class="fas fa-plus me-1"></i>+60 días
                            </button>
                            <button type="button" class="btn btn-outline-primary rounded-pill preset-renew" data-days="90">
                                <i class="fas fa-plus me-1"></i>+90 días
                            </button>
                        </div>
                        
                        <input type="date" class="form-control form-control-lg rounded-3 border-2"
                                id="fecha_fin_servicio_renew" name="fecha_fin_servicio" required>
                        <small class="text-muted mt-2 d-block">
                            <i class="fas fa-info-circle me-1"></i>
                            Los botones rápidos suman días a la fecha de vencimiento actual. También puedes elegir una fecha manual.
                        </small>
                    </div>

                    <div class="mb-4">
                        <label for="precio_venta_renew" class="form-label fw-bold text-dark">
                            <i class="fas fa-dollar-sign me-2 text-primary"></i>
                            Nuevo Precio (<?php echo htmlspecialchars($moneda_simbolo); ?>)
                        </label>
                        <div class="input-group input-group-lg">
                            <span class="input-group-text bg-light border-2"><?php echo htmlspecialchars($moneda_simbolo); ?></span>
                            <input type="number" step="0.01" class="form-control border-2"
                                id="precio_venta_renew" name="precio_venta" placeholder="0.00">
                        </div>
                        <small class="form-text text-muted mt-2 d-block">
                            <i class="fas fa-info-circle me-1"></i>
                            Se prellena con el precio actual. Déjalo en blanco si no cambia.
                        </small>
                    </div>
                    
                    <div class="alert alert-success border-0 rounded-3 mb-0">
                        <div class="d-flex align-items-center">
                            <i class="fas fa-check-circle fa-2x text-success me-3"></i>
                            <div>
                                <h6 class="mb-1 fw-bold text-success">Resumen de Renovación</h6>
                                <p class="mb-0 text-success">
                                    ✅ Esta renovación se registrará automáticamente como un <strong>nuevo ingreso</strong> en tu sistema de finanzas.<br>
                                    💰 El monto se agregará a tu reporte de ingresos con categoría "Renovaciones".
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer d-flex justify-content-between p-4 border-top-0">
                    <button type="button" class="btn btn-outline-secondary rounded-pill px-4" data-bs-dismiss="modal">
                        <i class="fas fa-times me-2"></i>Cerrar
                    </button>
                    <button type="submit" class="btn btn-primary rounded-pill shadow-sm px-4">
                        <i class="fas fa-check-circle me-2"></i>Confirmar Renovación
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>
<div class="modal fade" id="modalMensajeCliente" tabindex="-1" aria-labelledby="modalMensajeClienteLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered modal-lg">
        <div class="modal-content rounded-4 shadow-lg">
            <div class="modal-header border-0 rounded-top-4">
                <h5 class="modal-title" id="modalMensajeClienteLabel"><i class="fas fa-comment-dots me-2"></i>Mensaje para el Cliente</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body p-4">
                <div class="bg-light p-3 rounded-3 border mb-3">
                    <textarea class="form-control border-0 bg-light" id="mensajeClienteModalTextarea" rows="12" readonly></textarea>
                </div>
                <div class="d-flex flex-column flex-md-row justify-content-between align-items-stretch gap-3">
                    <button type="button" class="btn btn-secondary rounded-pill flex-grow-1 shadow-sm" id="btnCopiarMsgModal"><i class="fas fa-copy me-2"></i> Copiar Mensaje</button>
                    <a href="#" id="btnWhatsAppMsgModal" class="btn btn-success rounded-pill flex-grow-1 shadow-sm" target="_blank" rel="noopener noreferrer">
                        <i class="fab fa-whatsapp me-2"></i> Enviar por WhatsApp
                    </a>
                </div>
                <small class="form-text text-muted mt-2 d-block" id="hintNoPhone" style="display:none;">
                    <i class="fas fa-info-circle me-1"></i> No hay teléfono registrado; solo podrás copiar el mensaje.
                </small>
            </div>
        </div>
    </div>
</div>


<div class="modal fade" id="modalEliminarSuscripcion" tabindex="-1" aria-labelledby="modalEliminarSuscripcionLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content rounded-4 shadow-lg">
            <div class="modal-header bg-danger-soft border-0 rounded-top-4">
                <h5 class="modal-title text-danger-emphasis" id="modalEliminarSuscripcionLabel"><i class="fas fa-exclamation-triangle me-2"></i>Gestionar Suscripción</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="/digitalpremium/actions/perfiles_vendidos_controlador.php" method="POST">
                <div class="modal-body p-4">
                    <input type="hidden" name="accion" value="gestionar_eliminacion">
                    <input type="hidden" name="id_perfil_vendido" id="id_perfil_vendido_eliminar">
                    
                    <!-- PRESERVAR FILTROS ACTUALES -->
                    <?php
                    // Parámetros de filtro a preservar
                    $filtros_preservar = [
                        'buscar', 'filtro_estado', 'filtro_estado_cm', 'filtro_tipo_cuenta', 
                        'filtro_vendedor', 'filtro_fecha_inicio', 'filtro_fecha_fin', 
                        'filtro_precio_min', 'filtro_precio_max', 'ordenar_por', 'orden'
                    ];
                    
                    foreach ($filtros_preservar as $filtro) {
                        if (isset($_GET[$filtro]) && $_GET[$filtro] !== '') {
                            echo '<input type="hidden" name="' . htmlspecialchars($filtro) . '" value="' . htmlspecialchars($_GET[$filtro]) . '">';
                        }
                    }
                    ?>
                    <p>Vas a gestionar la suscripción del cliente <strong id="cliente_nombre_eliminar_modal"></strong>. Por favor, elige una opción:</p>
                    
                    <div class="form-check mb-3 p-3 border rounded-3 bg-light">
                        <input class="form-check-input" type="radio" name="tipo_gestion" id="opcionCancelar" value="cancelar" checked>
                        <label class="form-check-label fw-bold" for="opcionCancelar">
                            Cancelar Suscripción
                        </label>
                        <small class="d-block text-muted mt-1">
                            La suscripción se marcará como "Cancelada" y no aparecerá en las alertas, pero <strong>se mantendrá el registro de la venta</strong> en tus finanzas. Ideal si no hay reembolso.
                        </small>
                    </div>

                    <div class="form-check mb-3 p-3 border rounded-3 bg-light">
                        <input class="form-check-input" type="radio" name="tipo_gestion" id="opcionEliminar" value="eliminar_con_reembolso">
                        <label class="form-check-label fw-bold text-danger" for="opcionEliminar">
                            Eliminar Permanentemente (con Reembolso)
                        </label>
                        <small class="d-block text-muted mt-1">
                            <i class="fas fa-exclamation-triangle text-danger me-1"></i> ¡Acción irreversible! La suscripción y <strong>el registro de la venta se eliminarán por completo</strong>. Úsalo solo si has devuelto el dinero al cliente.
                        </small>
                    </div>

                </div>
                <div class="modal-footer d-flex justify-content-between p-3 border-top-0">
                    <button type="button" class="btn btn-secondary rounded-pill" data-bs-dismiss="modal">Cerrar</button>
                    <button type="submit" class="btn btn-danger rounded-pill shadow-sm"><i class="fas fa-check-circle me-1"></i> Confirmar Acción</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php include_once '../includes/footer.php'; ?>

<script>
document.addEventListener('DOMContentLoaded', function () {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (el) { return new bootstrap.Tooltip(el); });

    const searchInput = document.getElementById('quickSearchInput');
    const allSubscriptionWrappers = document.querySelectorAll('.subscription-card-wrapper');
    const noResultsMessage = document.getElementById('no-results-message');

    function filterCards(searchTerm) {
        let visibleCount = 0;
        allSubscriptionWrappers.forEach(wrapper => {
            const searchableText = (wrapper.querySelector('.subscription-card').getAttribute('data-search-text') || '').toLowerCase();
            if (searchableText.includes(searchTerm)) {
                wrapper.style.display = '';
                visibleCount++;
            } else {
                wrapper.style.display = 'none';
            }
        });
        if (noResultsMessage) {
            noResultsMessage.style.display = (visibleCount === 0 && allSubscriptionWrappers.length > 0) ? 'block' : 'none';
        }
    }
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            filterCards(this.value.toLowerCase().trim());
        });
    }
    window.filterByName = function(name) {
        if (searchInput) {
            searchInput.value = name;
            filterCards(name.toLowerCase().trim());
            const filterCollapse = document.getElementById('filtrosPerfilesCollapse');
            if (filterCollapse && filterCollapse.classList.contains('show')) {
                const bsCollapse = bootstrap.Collapse.getInstance(filterCollapse);
                if (bsCollapse) bsCollapse.hide();
            }
            searchInput.focus();
        }
    }

    window.editarPerfilVendido = function(idPerfil) {
        // Estrategia de preservación de filtros más robusta
        const filtros = [
            'buscar', 'filtro_estado', 'filtro_estado_cm', 'filtro_tipo_cuenta', 
            'filtro_vendedor', 'filtro_fecha_inicio', 'filtro_fecha_fin', 
            'filtro_precio_min', 'filtro_precio_max', 'ordenar_por', 'orden'
        ];
        
        // Construir URL con filtros preservados
        let url = '/digitalpremium/pages/perfiles_vendidos_crear.php?editar_id=' + idPerfil;
        let filtrosEncontrados = false;
        
        // 1. Prioridad: Filtros de la URL actual
        const urlParams = new URLSearchParams(window.location.search);
        filtros.forEach(filtro => {
            const valor = urlParams.get(filtro);
            if (valor && valor !== '') {
                url += '&' + filtro + '=' + encodeURIComponent(valor);
                filtrosEncontrados = true;
            }
        });
        
        // 2. Si no hay filtros en URL, usar localStorage como respaldo
        if (!filtrosEncontrados) {
            const filtrosGuardados = localStorage.getItem('filtros_perfiles_vendidos');
            if (filtrosGuardados) {
                try {
                    const filtrosObj = JSON.parse(filtrosGuardados);
                    Object.keys(filtrosObj).forEach(filtro => {
                        if (filtrosObj[filtro] && filtrosObj[filtro] !== '') {
                            url += '&' + filtro + '=' + encodeURIComponent(filtrosObj[filtro]);
                        }
                    });
                    console.log('Filtros preservados desde localStorage:', filtrosObj);
                } catch (e) {
                    console.error('Error al leer filtros de localStorage:', e);
                }
            }
        }
        
        console.log('URL de edición con filtros preservados:', url);
        window.location.href = url;
    }

    // Funciones para el Centro de Notificaciones
    window.expandAllAlerts = function() {
        const accordionItems = document.querySelectorAll('#accordionAlertas .accordion-collapse');
        accordionItems.forEach(item => {
            const bsCollapse = bootstrap.Collapse.getInstance(item);
            if (bsCollapse && !item.classList.contains('show')) {
                bsCollapse.show();
            }
        });
    }

    window.collapseAllAlerts = function() {
        const accordionItems = document.querySelectorAll('#accordionAlertas .accordion-collapse');
        accordionItems.forEach(item => {
            const bsCollapse = bootstrap.Collapse.getInstance(item);
            if (bsCollapse && item.classList.contains('show')) {
                bsCollapse.hide();
            }
        });
    }

    // Lógica para el modal de mensaje post-compra
    var modalEl = document.getElementById('modalMensajePostCompra');
    if (modalEl) {
        var modal = new bootstrap.Modal(modalEl);
        modal.show();
        var btnCopiar = document.getElementById('btnCopiarMensajeCliente');
        var textarea = document.getElementById('textoMensajeCliente');
        var feedback = document.getElementById('mensajeCopiadoFeedback');
        if (btnCopiar && textarea) {
            btnCopiar.addEventListener('click', function() {
                navigator.clipboard.writeText(textarea.value).then(() => {
                    feedback.classList.remove('d-none');
                    setTimeout(() => feedback.classList.add('d-none'), 2000);
                }).catch(err => console.error('Error al copiar: ', err));
            });
        }
    }



    const modalEliminar = document.getElementById('modalEliminarSuscripcion');
    if (modalEliminar) {
        modalEliminar.addEventListener('show.bs.modal', function (event) {
            const btn = event.relatedTarget;
            document.getElementById('id_perfil_vendido_eliminar').value = btn.getAttribute('data-id_perfil_vendido');
            document.getElementById('cliente_nombre_eliminar_modal').textContent = btn.getAttribute('data-cliente_nombre');
        });
    }
    
    const modalRenovar = document.getElementById('modalRenovarSuscripcion');
    if (modalRenovar) {
        modalRenovar.addEventListener('show.bs.modal', function(event) {
            const btn = event.relatedTarget;
            document.getElementById('id_perfil_vendido_renew').value = btn.getAttribute('data-id_perfil_vendido');
            document.getElementById('info_cliente_renew').textContent = btn.getAttribute('data-info_cliente');

            const precioVenta = btn.getAttribute('data-precio_venta');
            const precioInput = document.getElementById('precio_venta_renew');
            if (precioVenta) {
                precioInput.value = parseFloat(precioVenta).toFixed(2);
            } else {
                precioInput.value = '';
            }

            const fechaVencimientoActual = btn.getAttribute('data-fecha-vencimiento');
            let fechaBase;
            
            if (fechaVencimientoActual) {
                fechaBase = new Date(fechaVencimientoActual + 'T00:00:00');
            } else {
                const today = new Date();
                fechaBase = new Date();
                fechaBase.setDate(today.getDate());
            }
            
            const defaultDate = new Date(fechaBase);
            defaultDate.setMonth(fechaBase.getMonth() + 1);
            document.getElementById('fecha_fin_servicio_renew').value = defaultDate.toISOString().split('T')[0];
            
            document.querySelectorAll('.preset-renew').forEach(btn => {
                btn.classList.remove('btn-primary');
                btn.classList.add('btn-outline-primary');
            });
        });

        modalRenovar.addEventListener('click', function(event) {
            const t = event.target.closest('.preset-renew');
            if (!t) return;
            event.preventDefault();
            
            const days = parseInt(t.getAttribute('data-days'), 10) || 30;
            const fechaInput = document.getElementById('fecha_fin_servicio_renew');

            const isSelected = t.classList.contains('btn-primary');
            
            document.querySelectorAll('.preset-renew').forEach(btn => {
                btn.classList.remove('btn-primary');
                btn.classList.add('btn-outline-primary');
            });

            let fechaBase;
            const modalTrigger = document.querySelector('[data-bs-target="#modalRenovarSuscripcion"][data-id_perfil_vendido="' + document.getElementById('id_perfil_vendido_renew').value + '"]');
            const fechaVencimientoActual = modalTrigger.getAttribute('data-fecha-vencimiento');
            fechaBase = new Date(fechaVencimientoActual + 'T00:00:00');
            
            const now = new Date();
            const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());

            if (fechaBase < today) {
                fechaBase = today;
            }
            
            if (isSelected) {
                fechaInput.value = (new Date(fechaVencimientoActual + 'T00:00:00')).toISOString().split('T')[0];
            } else {
                t.classList.add('btn-primary');
                t.classList.remove('btn-outline-primary');
                
                const newDate = new Date(fechaBase);
                newDate.setDate(fechaBase.getDate() + days);
                fechaInput.value = newDate.toISOString().split('T')[0];
            }
        });
    }

    const modalMsg = document.getElementById('modalMensajeCliente');
    const txtArea = document.getElementById('mensajeClienteModalTextarea');
    const btnCopy = document.getElementById('btnCopiarMsgModal');
    const btnWa = document.getElementById('btnWhatsAppMsgModal');
    const hintNoPhone = document.getElementById('hintNoPhone');

    if (modalMsg) {
        modalMsg.addEventListener('show.bs.modal', function(event){
            const btn = event.relatedTarget;
            const mensaje = btn.getAttribute('data-mensaje') || '';
            const telefono = btn.getAttribute('data-telefono') || '';

            txtArea.value = mensaje;
            if (telefono) {
                btnWa.classList.remove('disabled');
                btnWa.href = 'https://wa.me/' + telefono + '?text=' + encodeURIComponent(mensaje);
                hintNoPhone.style.display = 'none';
            } else {
                btnWa.classList.add('disabled');
                btnWa.removeAttribute('href');
                hintNoPhone.style.display = 'block';
            }
        });

        btnCopy.addEventListener('click', function() {
            navigator.clipboard.writeText(txtArea.value).then(() => {
                const original = btnCopy.innerHTML;
                btnCopy.innerHTML = '<i class="fas fa-check me-1"></i> ¡Copiado!';
                btnCopy.classList.remove('btn-secondary');
                btnCopy.classList.add('btn-success');
                btnCopy.disabled = true;
                setTimeout(() => {
                    btnCopy.innerHTML = original;
                    btnCopy.classList.remove('btn-success');
                    btnCopy.classList.add('btn-secondary');
                    btnCopy.disabled = false;
                }, 1500);
            }).catch(() => alert('No se pudo copiar. Copie manualmente.'));
        });
    }

    const ventaModalEl = document.getElementById('ventaCreadaModal');
    if (ventaModalEl) {
        const ventaModal = new bootstrap.Modal(ventaModalEl);
        ventaModal.show();

        const mensajeTextarea = document.getElementById('mensajeClienteTextarea');
        const copiarBtn = document.getElementById('copiarMensajeBtn');
        const whatsappBtn = document.getElementById('enviarWhatsAppBtn');
        const telefonoCliente = "<?php echo $telefono_para_whatsapp ?? ''; ?>";

        copiarBtn.addEventListener('click', function() {
            navigator.clipboard.writeText(mensajeTextarea.value).then(() => {
                const originalText = this.innerHTML;
                this.innerHTML = '<i class="fas fa-check me-1"></i> ¡Copiado!';
                this.classList.remove('btn-secondary');
                this.classList.add('btn-success');
                this.disabled = true;
                setTimeout(() => {
                    this.innerHTML = originalText;
                    this.classList.remove('btn-success');
                    this.classList.add('btn-secondary');
                    this.disabled = false;
                }, 2000);
            }).catch(err => {
                alert('No se pudo copiar el texto. Por favor, hazlo manualmente.');
            });
        });

        if (telefonoCliente) {
            const mensajeCodificado = encodeURIComponent(mensajeTextarea.value);
            whatsappBtn.href = `https://api.whatsapp.com/send?phone=${telefonoCliente}&text=${mensajeCodificado}`;
        }
    }

    if (window.location.search.includes('mensaje=')) {
        setTimeout(function() {
            window.history.replaceState({}, document.title, window.location.pathname);
        }, 5000);
    }
    
    const btnCargarMas = document.getElementById('btnCargarMas');
    if (btnCargarMas) {
        btnCargarMas.addEventListener('click', function() {
            const registrosActuales = parseInt(this.getAttribute('data-registros-actuales'));
            const total = parseInt(this.getAttribute('data-total'));
            const registrosPorCarga = 50;
            const nuevosRegistros = Math.min(registrosActuales + registrosPorCarga, total);
            
            const originalText = this.innerHTML;
            this.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Cargando...';
            this.disabled = true;
            
            const url = new URL(window.location);
            url.searchParams.set('cargar_mas', nuevosRegistros);
            
            window.location.href = url.toString();
        });
    }
});

// Universal copyToClipboard function
function copyToClipboard(selectorOrText, element) {
    let textToCopy;
    let targetInput = null;

    if (typeof selectorOrText === 'string' && selectorOrText.startsWith('#')) {
        targetInput = document.querySelector(selectorOrText);
        if (targetInput) {
            if (targetInput.tagName === 'INPUT' || targetInput.tagName === 'TEXTAREA') {
                textToCopy = targetInput.value;
            } else {
                textToCopy = targetInput.innerText;
            }
        } else {
            console.error('Elemento no encontrado para copiar:', selectorOrText);
            return;
        }
    } else {
        textToCopy = selectorOrText;
    }

    if (!textToCopy) return;

    navigator.clipboard.writeText(textToCopy).then(() => {
        const tooltip = bootstrap.Tooltip.getInstance(element);
        const originalTitle = element.getAttribute('data-bs-original-title');
        
        if (tooltip) {
            element.setAttribute('data-bs-original-title', '¡Copiado!');
            tooltip.show();
            setTimeout(() => {
                tooltip.hide();
                setTimeout(() => { element.setAttribute('data-bs-original-title', originalTitle); }, 150);
            }, 1000);
        } else {
            const originalIconClass = element.className;
            element.className = 'fas fa-check text-success ms-2';
            setTimeout(() => {
                element.className = originalIconClass;
            }, 1000);
        }
    }).catch(err => console.error('Error al copiar: ', err));
}

// Universal togglePasswordVisibility function
function togglePasswordVisibility(inputId, iconElement) {
    const input = document.getElementById(inputId);
    if (!input) return;

    if (input.type === 'password') {
        input.type = 'text';
        iconElement.classList.remove('fa-eye');
        iconElement.classList.add('fa-eye-slash');
        iconElement.setAttribute('data-bs-original-title', 'Ocultar Contraseña');
    } else {
        input.type = 'password';
        iconElement.classList.remove('fa-eye-slash');
        iconElement.classList.add('fa-eye');
        iconElement.setAttribute('data-bs-original-title', 'Mostrar Contraseña');
    }
    const tooltip = bootstrap.Tooltip.getInstance(iconElement);
    if (tooltip) {
        tooltip.update();
    } else {
        new bootstrap.Tooltip(iconElement);
    }
}
</script>
</body>
</html>