<?php
// FILE: /digitalpremium/pages/proveedores.php

// Protección de la página y inicio de sesión
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
if (!isset($_SESSION['id_usuario'])) {
    header("Location: /digitalpremium/login.php");
    exit();
}

require_once '../config/db.php'; // Conexión a la BD
$page_title = "Gestión de Proveedores - DigitalPremium";
include_once '../includes/header.php'; // Cabecera HTML

// <<< INICIO DE CORRECCIONES PHP >>>

// --- 1. OBTENER DATOS DE SESIÓN ---
$id_usuario_sesion = $_SESSION['id_usuario'];
$rol_usuario_sesion = $_SESSION['rol'];

// --- 2. LÓGICA DE FILTROS CORREGIDA ---
$where_clauses_prov = [];
$params_prov = [];
$filtros_aplicados = false; // Variable para saber si se aplicó algún filtro

// Regla de Oro: Si no es superadmin, siempre filtrar por su ID de usuario.
if ($rol_usuario_sesion !== 'superadmin') {
    $where_clauses_prov[] = "id_usuario = :id_usuario_sesion";
    $params_prov[':id_usuario_sesion'] = $id_usuario_sesion;
}

// Filtro por Nombre de Proveedor (búsqueda parcial)
if (!empty($_GET['filtro_nombre_proveedor'])) {
    $where_clauses_prov[] = "nombre_proveedor LIKE :nombre_proveedor";
    $params_prov[':nombre_proveedor'] = '%' . trim($_GET['filtro_nombre_proveedor']) . '%';
    $filtros_aplicados = true;
}

// Filtro por Estado del Proveedor
if (isset($_GET['filtro_estado_proveedor']) && $_GET['filtro_estado_proveedor'] !== '') {
    $where_clauses_prov[] = "estado_proveedor = :estado_proveedor";
    $params_prov[':estado_proveedor'] = $_GET['filtro_estado_proveedor'];
    $filtros_aplicados = true;
}

$sql_where_prov = "";
if (!empty($where_clauses_prov)) {
    $sql_where_prov = " WHERE " . implode(" AND ", $where_clauses_prov);
}

// --- 3. CONSULTA PRINCIPAL CORREGIDA ---
// La consulta ahora es segura porque se alimenta del $sql_where_prov que ya contiene el filtro de usuario.
$sql_proveedores = "SELECT * FROM proveedores $sql_where_prov ORDER BY nombre_proveedor ASC";
$stmt_proveedores = $pdo->prepare($sql_proveedores);
$stmt_proveedores->execute($params_prov);
$proveedores = $stmt_proveedores->fetchAll(PDO::FETCH_ASSOC);

// Estados de Proveedor para el filtro
$estados_proveedor_filtro = [
    "1" => "Activo",
    "0" => "Inactivo"
];
// <<< FIN DE CORRECCIONES PHP >>>
?>

<div class="container-fluid px-4">
    <header class="d-flex justify-content-between align-items-center mt-4 mb-3">
         <h1 class="h3 text-primary"><i class="fas fa-truck-loading me-2"></i>Gestión de Proveedores</h1>
        <button type="button" class="btn btn-primary shadow-sm" data-bs-toggle="modal" data-bs-target="#modalAgregarProveedor">
            <i class="fas fa-plus-circle"></i> Agregar Nuevo Proveedor
        </button>
    </header>

    <?php if (isset($_GET['mensaje'])): ?>
        <div class="alert alert-<?php echo htmlspecialchars($_GET['tipo'] ?? 'info'); ?> alert-dismissible fade show" role="alert">
            <?php echo htmlspecialchars(urldecode($_GET['mensaje'])); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>

    <div class="card mb-4 shadow-sm">
        <div class="card-header bg-light py-3">
            <h6 class="mb-0 fw-bold text-secondary"><i class="fas fa-filter me-2"></i>Filtros de Búsqueda</h6>
        </div>
        <div class="card-body">
            <form action="" method="GET" class="row g-3 align-items-end">
                <div class="col-md-5 col-sm-6">
                    <label for="filtro_nombre_proveedor" class="form-label">Nombre del Proveedor:</label>
                    <input type="text" name="filtro_nombre_proveedor" id="filtro_nombre_proveedor" class="form-control form-control-sm" value="<?php echo htmlspecialchars($_GET['filtro_nombre_proveedor'] ?? ''); ?>">
                </div>
                <div class="col-md-4 col-sm-6">
                    <label for="filtro_estado_proveedor" class="form-label">Estado:</label>
                    <select name="filtro_estado_proveedor" id="filtro_estado_proveedor" class="form-select form-select-sm">
                        <option value="">Todos los Estados</option>
                        <?php foreach ($estados_proveedor_filtro as $val => $texto): ?>
                             <option value="<?php echo $val; ?>" <?php echo (isset($_GET['filtro_estado_proveedor']) && $_GET['filtro_estado_proveedor'] !== '' && $_GET['filtro_estado_proveedor'] == $val) ? 'selected' : ''; ?>>
                                <?php echo $texto; ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-3 col-sm-12 d-flex align-items-end mt-3 mt-md-0">
                    <button type="submit" class="btn btn-info btn-sm flex-grow-1 me-2"><i class="fas fa-search"></i> Filtrar</button>
                    <a href="/digitalpremium/pages/proveedores.php" class="btn btn-outline-secondary btn-sm flex-grow-1"><i class="fas fa-eraser"></i> Limpiar</a>
                </div>
            </form>
        </div>
    </div>

    <div class="card shadow-sm">
        <div class="card-header bg-light py-3">
            <h6 class="mb-0 fw-bold text-secondary"><i class="fas fa-list me-2"></i>Listado de Proveedores</h6>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-sm table-striped table-hover table-bordered align-middle table-compact">
                    <thead class="table-dark">
                        <tr>
                            <th>ID</th>
                            <th>Nombre Proveedor</th>
                            <th>Contacto</th>
                            <th>Teléfono</th>
                            <th>Email</th>
                            <th class="text-center">Estado</th>
                            <th class="text-center" style="width: 100px;">Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (count($proveedores) > 0): ?>
                            <?php foreach ($proveedores as $proveedor): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($proveedor['id_proveedor']); ?></td>
                                    <td><strong><?php echo htmlspecialchars($proveedor['nombre_proveedor']); ?></strong></td>
                                    <td><?php echo htmlspecialchars($proveedor['contacto_nombre'] ?? 'N/A'); ?></td>
                                    <td><?php echo htmlspecialchars($proveedor['contacto_telefono'] ?? 'N/A'); ?></td>
                                    <td>
                                        <?php if (!empty($proveedor['contacto_email'])): ?>
                                            <a href="mailto:<?php echo htmlspecialchars($proveedor['contacto_email']); ?>"><?php echo htmlspecialchars($proveedor['contacto_email']); ?></a>
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                    </td>
                                    <td class="text-center">
                                        <?php if ($proveedor['estado_proveedor'] == 1): ?>
                                            <span class="badge bg-success">Activo</span>
                                        <?php else: ?>
                                            <span class="badge bg-danger">Inactivo</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="text-center">
                                        <div class="btn-group btn-group-sm" role="group">
                                            <button type="button" class="btn btn-outline-primary py-1 px-2 btn-edit" 
                                                    data-id="<?php echo $proveedor['id_proveedor']; ?>"
                                                    data-nombre_proveedor="<?php echo htmlspecialchars($proveedor['nombre_proveedor']); ?>"
                                                    data-contacto_nombre="<?php echo htmlspecialchars($proveedor['contacto_nombre'] ?? ''); ?>"
                                                    data-contacto_telefono="<?php echo htmlspecialchars($proveedor['contacto_telefono'] ?? ''); ?>"
                                                    data-contacto_email="<?php echo htmlspecialchars($proveedor['contacto_email'] ?? ''); ?>"
                                                    data-notas_proveedor="<?php echo htmlspecialchars($proveedor['notas_proveedor'] ?? ''); ?>"
                                                    data-estado_proveedor="<?php echo $proveedor['estado_proveedor']; ?>"
                                                    data-bs-toggle="modal" data-bs-target="#modalEditarProveedor" title="Editar Proveedor">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary py-1 px-2 btn-toggle-estado"
                                                    data-id="<?php echo $proveedor['id_proveedor']; ?>"
                                                    data-nombre="<?php echo htmlspecialchars($proveedor['nombre_proveedor']); ?>"
                                                    data-estado_actual="<?php echo $proveedor['estado_proveedor']; ?>"
                                                    data-bs-toggle="modal" data-bs-target="#modalCambiarEstadoProveedor" 
                                                    title="<?php echo ($proveedor['estado_proveedor'] == 1 ? 'Marcar como Inactivo' : 'Marcar como Activo'); ?>">
                                                <i class="fas fa-<?php echo ($proveedor['estado_proveedor'] == 1 ? 'toggle-off' : 'toggle-on'); ?>"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="7" class="text-center py-4 text-muted"><i>
                                 <?php echo ($filtros_aplicados) ? 'No se encontraron proveedores con los filtros aplicados.' : 'No hay proveedores registrados.'; ?>
                                </i></td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="modalAgregarProveedor" tabindex="-1" aria-labelledby="modalAgregarProveedorLabel" aria-hidden="true" data-bs-backdrop="static">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title" id="modalAgregarProveedorLabel"><i class="fas fa-plus-circle"></i> Agregar Nuevo Proveedor</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="/digitalpremium/actions/proveedores_controlador.php" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="accion" value="agregar">
                    <div class="mb-3">
                        <label for="nombre_proveedor_add" class="form-label">Nombre del Proveedor <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="nombre_proveedor_add" name="nombre_proveedor" required>
                    </div>
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label for="contacto_nombre_add" class="form-label">Nombre de Contacto</label>
                            <input type="text" class="form-control" id="contacto_nombre_add" name="contacto_nombre">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="contacto_telefono_add" class="form-label">Teléfono de Contacto</label>
                            <input type="tel" class="form-control" id="contacto_telefono_add" name="contacto_telefono">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="contacto_email_add" class="form-label">Email de Contacto</label>
                            <input type="email" class="form-control" id="contacto_email_add" name="contacto_email">
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="notas_proveedor_add" class="form-label">Notas Adicionales</label>
                        <textarea class="form-control" id="notas_proveedor_add" name="notas_proveedor" rows="3"></textarea>
                    </div>
                     <div class="mb-3">
                        <label for="estado_proveedor_add" class="form-label">Estado <span class="text-danger">*</span></label>
                        <select class="form-select" id="estado_proveedor_add" name="estado_proveedor" required>
                            <option value="1" selected>Activo</option>
                            <option value="0">Inactivo</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
                    <button type="submit" class="btn btn-primary"><i class="fas fa-save"></i> Guardar Proveedor</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="modalEditarProveedor" tabindex="-1" aria-labelledby="modalEditarProveedorLabel" aria-hidden="true" data-bs-backdrop="static">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-warning text-dark">
                <h5 class="modal-title" id="modalEditarProveedorLabel"><i class="fas fa-edit"></i> Editar Proveedor</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="/digitalpremium/actions/proveedores_controlador.php" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="accion" value="editar">
                    <input type="hidden" name="id_proveedor" id="id_proveedor_edit">
                    <div class="mb-3">
                        <label for="nombre_proveedor_edit" class="form-label">Nombre del Proveedor <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="nombre_proveedor_edit" name="nombre_proveedor" required>
                    </div>
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label for="contacto_nombre_edit" class="form-label">Nombre de Contacto</label>
                            <input type="text" class="form-control" id="contacto_nombre_edit" name="contacto_nombre">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="contacto_telefono_edit" class="form-label">Teléfono de Contacto</label>
                            <input type="tel" class="form-control" id="contacto_telefono_edit" name="contacto_telefono">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="contacto_email_edit" class="form-label">Email de Contacto</label>
                            <input type="email" class="form-control" id="contacto_email_edit" name="contacto_email">
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="notas_proveedor_edit" class="form-label">Notas Adicionales</label>
                        <textarea class="form-control" id="notas_proveedor_edit" name="notas_proveedor" rows="3"></textarea>
                    </div>
                    <div class="mb-3">
                        <label for="estado_proveedor_edit" class="form-label">Estado <span class="text-danger">*</span></label>
                        <select class="form-select" id="estado_proveedor_edit" name="estado_proveedor" required>
                            <option value="1">Activo</option>
                            <option value="0">Inactivo</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
                    <button type="submit" class="btn btn-warning"><i class="fas fa-save"></i> Actualizar Proveedor</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="modalCambiarEstadoProveedor" tabindex="-1" aria-labelledby="modalCambiarEstadoProveedorLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header"> <h5 class="modal-title" id="modalCambiarEstadoProveedorLabel"><i class="fas fa-exchange-alt"></i> Cambiar Estado de Proveedor</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="/digitalpremium/actions/proveedores_controlador.php" method="POST">
                <div class="modal-body">
                    <input type="hidden" name="accion" value="cambiar_estado">
                    <input type="hidden" name="id_proveedor" id="id_proveedor_estado_js"> <input type="hidden" name="nuevo_estado" id="nuevo_estado_proveedor_js">
                    <p>¿Estás seguro de que deseas cambiar el estado del proveedor "<strong id="nombre_proveedor_estado_js"></strong>" a <strong id="texto_nuevo_estado_proveedor_js"></strong>?</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-info"><i class="fas fa-check"></i> Sí, Cambiar Estado</button> </div>
            </form>
        </div>
    </div>
</div>

<?php include_once '../includes/footer.php'; ?>

<script>
document.addEventListener('DOMContentLoaded', function () {
    var modalEditarProveedor = document.getElementById('modalEditarProveedor');
    if(modalEditarProveedor) {
        modalEditarProveedor.addEventListener('show.bs.modal', function (event) {
            var button = event.relatedTarget;
            document.getElementById('id_proveedor_edit').value = button.getAttribute('data-id');
            document.getElementById('nombre_proveedor_edit').value = button.getAttribute('data-nombre_proveedor');
            document.getElementById('contacto_nombre_edit').value = button.getAttribute('data-contacto_nombre');
            document.getElementById('contacto_telefono_edit').value = button.getAttribute('data-contacto_telefono');
            document.getElementById('contacto_email_edit').value = button.getAttribute('data-contacto_email');
            document.getElementById('notas_proveedor_edit').value = button.getAttribute('data-notas_proveedor');
            document.getElementById('estado_proveedor_edit').value = button.getAttribute('data-estado_proveedor');
        });
    }

    var modalCambiarEstadoProveedor = document.getElementById('modalCambiarEstadoProveedor');
    if(modalCambiarEstadoProveedor) {
        modalCambiarEstadoProveedor.addEventListener('show.bs.modal', function (event) {
            var button = event.relatedTarget;
            var id = button.getAttribute('data-id');
            var nombre = button.getAttribute('data-nombre');
            var estadoActual = button.getAttribute('data-estado_actual');
            
            var nuevoEstado = (estadoActual == '1' ? '0' : '1');
            var textoNuevoEstado = (nuevoEstado == '1' ? 'Activo' : 'Inactivo');

            document.getElementById('id_proveedor_estado_js').value = id;
            document.getElementById('nombre_proveedor_estado_js').textContent = nombre;
            document.getElementById('nuevo_estado_proveedor_js').value = nuevoEstado;
            document.getElementById('texto_nuevo_estado_proveedor_js').textContent = textoNuevoEstado;

            var modalHeader = modalCambiarEstadoProveedor.querySelector('.modal-header');
            var btnConfirmar = modalCambiarEstadoProveedor.querySelector('button[type="submit"]');
            
            modalHeader.classList.remove('bg-success', 'bg-danger', 'text-white');
            btnConfirmar.classList.remove('btn-success', 'btn-danger');
            btnConfirmar.classList.add('btn-info'); 

            if (nuevoEstado == '1') {
                modalHeader.classList.add('bg-success', 'text-white');
                btnConfirmar.classList.replace('btn-info','btn-success');
                btnConfirmar.innerHTML = '<i class="fas fa-toggle-on"></i> Sí, Marcar Activo';
            } else { 
                modalHeader.classList.add('bg-danger', 'text-white');
                btnConfirmar.classList.replace('btn-info','btn-danger');
                btnConfirmar.innerHTML = '<i class="fas fa-toggle-off"></i> Sí, Marcar Inactivo';
            }
        });
    }

    if (window.location.search.includes('mensaje=')) {
        setTimeout(function() {
            var newUrl = window.location.protocol + "//" + window.location.host + window.location.pathname;
            window.history.replaceState({ path: newUrl }, '', newUrl);
        }, 5000);
    }
});
</script>